import {platform} from './../utils/platform';
import {AgoraElectronClient} from './../utils/agora-electron-client';
import {AgoraStream, ChatMessage, TOOL_TYPE} from '../utils/types';
import {Subject} from 'rxjs';
import {List, OrderedMap, OrderedSet} from 'immutable';
import AgoraRTMClient, {RoomMessage} from '../utils/agora-rtm-client';
import {globalStore} from './global';
import AgoraWebClient from '../utils/agora-rtc-client';
import {get} from 'lodash';
import {isElectron} from '../utils/platform';
import GlobalStorage from '../utils/custom-storage';
import {STAGE_NUM} from "../utils/consts";

function canJoin({uid, onlineStatus, roomType, channelCount, role}: { uid: any, onlineStatus: any, role: string, channelCount: number, roomType: number}) {
  const result = {
    permitted: true,
    reason: ''
  }
  const channelCountLimit = [2, 17, 17];

  let maximum = channelCountLimit[roomType];
  if (channelCount >= maximum) {
    result.permitted = false;
    result.reason = 'The number of students and teacher have reached upper limit';
    return result;
  }

  const teacher = get(onlineStatus, 'teacher', false);
  const totalCount: number = get(onlineStatus, 'totalCount', 0);

  if (role === 'teacher') {
    const isOnline = teacher && `${uid}` !== `${onlineStatus.teacherId}`;
    if (isOnline) {
      result.permitted = false;
      result.reason = 'Teacher already existed';
      return result;
    }
  }

  if (role === 'student') {
    if (totalCount+1 > maximum) {
      result.permitted = false;
      result.reason = 'Student have reached upper limit';
      return result;
    }
  }

  return result;
}


export interface AgoraUser {
  uid: string
  account: string
  role: string
  video: number
  audio: number
  chat: number
  boardId: string // whiteboard_uuid
  sharedId: number // shared_uid
  linkId: number // link_uid
  lockBoard?: number // lock_board
  grantBoard: number
}

export interface ClassState {
  rid: string
  roomName: string
  teacherId: string
  roomType: number
  boardId: string // whiteboard_uuid
  sharedId: number // shared_uid
  linkId: number // link_uid
  lockBoard: number // lock_board
  courseState: number
  muteChat: number,
  currentTool: TOOL_TYPE
}

type RtcState = {
  published: boolean
  joined: boolean
  users: OrderedSet<number>
  shared: boolean
  localStream: AgoraMediaStream | null
  localSharedStream: AgoraMediaStream | null
  remoteStreams: OrderedMap<number, AgoraMediaStream>
}

export type MediaDeviceState = {
  microphoneId: string
  speakerId: string
  cameraId: string
  speakerVolume: number
  camera: number
  microphone: number
  speaker: number
}

export type SessionInfo = {
  uid: string
  rid: string
  account: string
  roomName: string
  roomType: number
  role: string
}

export type RtmState = {
  joined: boolean
  memberCount: number
}

export type RoomState = {
  rtmLock: boolean
  me: AgoraUser
  users: OrderedMap<string, AgoraUser>
  course: ClassState
  applyUid: number
  rtc: RtcState
  rtm: RtmState
  mediaDevice: MediaDeviceState
  messages: List<ChatMessage>
  studentsOrder: number[]
}

export type AgoraMediaStream = {
  streamID: number
  stream?: any
}

export class RoomStore {
  private subject: Subject<RoomState> | null;
  public _state: RoomState;

  get state () {
    return this._state;
  }

  set state (newState) {
    this._state = newState;
  }
  public rtmClient: AgoraRTMClient = new AgoraRTMClient();
  public rtcClient: AgoraWebClient | AgoraElectronClient = isElectron ? new AgoraElectronClient () : new AgoraWebClient();
  public readonly defaultState: RoomState = Object.freeze({
    rtmLock: false,
    me: {
      account: "",
      uid: "",
      role: "",
      video: 1,
      audio: 1,
      chat: 1,
      linkId: 0,
      sharedId: 0,
      boardId: '',
    },
    users: OrderedMap<string, AgoraUser>(),
    applyUid: 0,
    rtm: {
      joined: false,
      memberCount: 0,
    },
    rtc: {
      published: false,
      joined: false,
      shared: false,
      users: OrderedSet<number>(),
      localStream: null,
      localSharedStream: null,
      remoteStreams: OrderedMap<number, AgoraMediaStream>(),
    },
    course: {
      teacherId: '',
      boardId: '',
      sharedId: 0,
      linkId: 0,
      courseState: 0,
      muteChat: 0,
      rid: '',
      roomName: '',
      roomType: 0,
      currentTool: TOOL_TYPE.WHITEBOARD
    },
    mediaDevice: {
      microphoneId: '',
      speakerId: '',
      cameraId: '',
      speakerVolume: 100,
      camera: 0,
      speaker: 0,
      microphone: 0
    },
    messages: List<ChatMessage>(),
    studentsOrder: [],
    ...GlobalStorage.read('agora_room')
  });

  private applyLock: number = 0;

  public windowId: number = 0;

  constructor() {
    this.subject = null;
    this._state = {
      ...this.defaultState
    };
    // @ts-ignore
    window['roomStore'] = this;
  }
  /*
  async stageUpRtc(uid: string) {
    let store = this._state.rtc.users;
    let itemsList = store.toList();
    const idx = itemsList.indexOf(+uid);
    if (idx < STAGE_NUM) {
      return;
    }
    itemsList = itemsList.insert(0, +uid);
    const downVal = itemsList.get(STAGE_NUM);
    if (downVal != undefined) {
      itemsList = itemsList.splice(STAGE_NUM, 1).splice(idx, 1, downVal);
      store = OrderedSet(itemsList)
    } else {
      return ;
    }
    this.state = {
      ...this.state,
      rtc: {
        ...this.state.rtc,
        users: store,
      }
    }
    this.commit(this.state);
  }
  async stageDownRtc(uid: string) {
    let store = this._state.rtc.users;
    console.log(store);
    // let uid = 2;
    // stageNum = 3;
    let itemsList = store.toList();
    let idx = itemsList.indexOf(+uid);
    if (idx >= STAGE_NUM) {
      return;
    }
    let upVal = itemsList.get(STAGE_NUM);
    let downVal = itemsList.get(idx);
    if (upVal != undefined && downVal != undefined) {
      itemsList = itemsList.splice(STAGE_NUM, 1).splice(idx, 1, upVal).push(downVal)
      store = OrderedSet(itemsList)
    } else {
      return
    }
    this.state = {
      ...this.state,
      rtc: {
        ...this.state.rtc,
        users: store,
      }
    }
    this.commit(this.state);
  }
  async stageUp1(uid: string) {
    const attrs = await this.rtmClient.getChannelAttrs();
    let studentsOrder: any[] = [];
    // if ((attrs as any).studentsOrder && (attrs as any).studentsOrder.value) {
    //   studentsOrder = JSON.parse((attrs as any).studentsOrder.value) || [];
    // }
    const data: any = {};
    const onlineStatus = JSON.parse(attrs)
    const target = jsonParse(get(onlineStatus[uid], 'value'));
    // 上台需要开启视频
    // if (!+target.video) {
      target.video = 1;
      data[uid] = JSON.stringify(target)
    // }

    const tmp: [string, AgoraUser][] =[];
    this._state.users.forEach((v, k)=>{
      if (/^[0-9]+$/.test(k)) {
        tmp.push([k, v])
      }
    })
    const uidIdx = tmp.findIndex((o) => {
      return `${o[0]}` == `${uid}`
    })
    const uidObj = tmp[uidIdx]
    tmp.splice(uidIdx, 1)
    const stageEndObj = tmp[STAGE_NUM - 1]
    tmp.splice(STAGE_NUM - 1, 1)
    tmp.unshift(uidObj);
    tmp.push(stageEndObj)
    // 下台关闭音视频
    // stageEndObj[1].audio = 0;
    // stageEndObj[1].video = 0;
    data[stageEndObj[1].uid] = JSON.stringify(stageEndObj[1])

    tmp.forEach(o => {
      studentsOrder.push(+o[0]);
    });
    // const queue = {'studentsOrder': JSON.stringify(studentsOrder)};
    data['studentsOrder'] = JSON.stringify(studentsOrder)
    const aa = await this.rtmClient.addOrUpdateChannelAttributes(data);
    console.log(aa)
    const users = OrderedMap(tmp);


    // this.state = {
    //   ...this.state,
    //   users
    // }


    // this.commit(this.state);
  }
  async stageDown2(uid: string) {
    const tmp: [string, AgoraUser][]  =[];
    this._state.users.forEach((v, k)=>{
      tmp.push([k, v])
    })
    console.log(tmp);
    const uidIdx = tmp.findIndex((o) => {
      return `${o[0]}` == `${uid}`
    })
    const uidObj = tmp[uidIdx]

    const waitUpObj = tmp[STAGE_NUM]
    tmp.splice(uidIdx, 1, waitUpObj)
    tmp.splice(STAGE_NUM , 1)
    tmp.push(uidObj)
    const users = OrderedMap(tmp);
    this.state = {
      ...this.state,
      users
    }
    this.commit(this.state);
  }
  async stageUp2(uid: string) {
    const attrs = await this.rtmClient.getChannelAttrs();
    let studentsOrder: any[] = [];
    // if ((attrs as any).studentsOrder && (attrs as any).studentsOrder.value) {
    //   studentsOrder = JSON.parse((attrs as any).studentsOrder.value) || [];
    // }
    const data: any = {};
    const onlineStatus = jsonParse(attrs)
    let target = jsonParse(get(onlineStatus[uid], 'value'));
    if (typeof(target) == 'string') {
      target = jsonParse(target)
    }
    // 上台需要开启视频
    // if (!+target.video) {
    target.video = 1;
    data[uid] = target
    // }

    const tmp: [string, AgoraUser][] =[];
    this._state.users.forEach((v, k)=>{
      if (/^[0-9]+$/.test(k)) {
        tmp.push([k, v])
      }
    })
    const uidIdx = tmp.findIndex((o) => {
      return `${o[0]}` == `${uid}`
    })
    const uidObj = tmp[uidIdx]
    tmp.splice(uidIdx, 1)
    const stageEndObj = tmp[STAGE_NUM - 1]
    tmp.splice(STAGE_NUM - 1, 1)
    tmp.unshift(uidObj);
    tmp.push(stageEndObj)
    // 下台关闭音视频
    // stageEndObj[1].audio = 0;
    // stageEndObj[1].video = 0;
    data[stageEndObj[1].uid] = stageEndObj[1]

    tmp.forEach(o => {
      studentsOrder.push(+o[0]);
    });
    // const queue = {'studentsOrder': JSON.stringify(studentsOrder)};
    data['studentsOrder'] = studentsOrder
    const aa = await this.rtmClient.addOrUpdateChannelAttributes(data);
    console.log(aa)
    const users = OrderedMap(tmp);


    // this.state = {
    //   ...this.state,
    //   users
    // }


    // this.commit(this.state);
  }
  */
  async stageDown(uid: string) {}
  async stageUp(uid: string) {
    debugger
    let studentsOrder: any[] = [...this.state.studentsOrder];
    studentsOrder.unshift(+uid);
    const lastUid = studentsOrder[studentsOrder.length - 1];
    const lastUser: AgoraUser | undefined = this.state.users.get(`${lastUid}`);
    const upUser: AgoraUser | undefined = this.state.users.get(`${uid}`);
    if (!lastUser || !upUser) {
      return
    }
    lastUser.video = 0;
    lastUser.audio = 0;
    upUser.video = 1;
    upUser.audio = 1;

    studentsOrder.length = STAGE_NUM;


    const data: any = {};
    data[upUser.uid] = upUser;
    data[lastUser.uid] = lastUser;
    data['studentsOrder'] = studentsOrder;
    await this.rtmClient.addOrUpdateChannelAttributes(data);






    // this.state = {
    //   ...this.state,
    //   users
    // }


    // this.commit(this.state);
  }

  async setCurrentTool (tool: TOOL_TYPE) {
    this.state = {
      ...this.state,
      course: {
        ...this.state.course,
        currentTool: tool
      },
    }

    const me = this.state.me;
    let res = await this.updateAttrsBy(me.uid, {
      current_tool: tool
    }).then(() => {
      console.log("update success");
    }).catch(console.warn)

    // console.log("[update tool uid] res", uid);
    return res;
  }
  initialize() {
    this.subject = new Subject<RoomState>();
    this.state = {
      ...this.defaultState,
    }
    this.applyLock = 0;
    this.subject.next(this.state);
  }

  get applyUid () {
    return this.applyLock;
  }

  subscribe(updateState: any) {
    this.initialize();
    this.subject && this.subject.subscribe(updateState);
  }

  unsubscribe() {
    this.subject && this.subject.unsubscribe();
    this.subject = null;
  }

  commit (state: RoomState) {
    this.subject && this.subject.next(state);
  }

  updateState(rootState: RoomState) {
    this.state = {
      ...this.state,
      ...rootState,
    }
    this.commit(this.state);
  }

  isTeacher(peerId: string) {
    if (!peerId) return false;
    const user = this.state.users.get(peerId)
    if (!user) return false;
    if (user.role === 'teacher') return true;
    return false;
  }

  isStudent (peerId: string) {
    if (!peerId) return false;
    const user = this.state.users.get(peerId);
    if (!user) return false;
    if (user.role === 'student') return true;
    return false;
  }

  addLocalStream(stream: AgoraStream) {
    this.state = {
      ...this.state,
      rtc: {
        ...this.state.rtc,
        localStream: stream
      }
    }
    this.commit(this.state);
  }

  removeLocalStream() {
    this.state = {
      ...this.state,
      rtc: {
        ...this.state.rtc,
        localStream: null,
        localSharedStream: null
      }
    }
    this.commit(this.state);
  }

  addLocalSharedStream(stream: any) {
    this.state = {
      ...this.state,
      rtc: {
        ...this.state.rtc,
        localSharedStream: stream
      }
    }
    this.commit(this.state);
  }

  removeLocalSharedStream() {
    this.state = {
      ...this.state,
      rtc: {
        ...this.state.rtc,
        localSharedStream: null
      }
    }
    this.commit(this.state);
  }

  addPeerUser(uid: number) {
    this.state = {
      ...this.state,
      rtc: {
        ...this.state.rtc,
        users: this.state.rtc.users.add(uid),
      }
    }
    this.commit(this.state);
  }

  removePeerUser(uid: number) {
    this.state = {
      ...this.state,
      rtc: {
        ...this.state.rtc,
        users: this.state.rtc.users.delete(uid),
      }
    }
    this.commit(this.state);
  }

  addRemoteStream(stream: AgoraStream) {
    console.log('remoteStreams added',5 , stream.streamID)
    this.state = {
      ...this.state,
      rtc: {
        ...this.state.rtc,
        remoteStreams: this.state.rtc.remoteStreams.set(stream.streamID, stream)
      }
    }
    this.commit(this.state);
  }

  removeRemoteStream(uid: number) {
    const remoteStream = this.state.rtc.remoteStreams.get(uid);
    if (platform === 'web') {
      if (remoteStream && remoteStream.stream && remoteStream.stream.isPlaying) {
        remoteStream.stream.isPlaying() && remoteStream.stream.stop();
      }
    }


    this.state = {
      ...this.state,
      rtc: {
        ...this.state.rtc,
        remoteStreams: this.state.rtc.remoteStreams.delete(uid)
      }
    }
  }

  updateMemberCount(count: number) {
    this.state = {
      ...this.state,
      rtm: {
        ...this.state.rtm,
        memberCount: count,
      }
    }
    this.commit(this.state);
  }

  updateRtc(newState: any) {
    this.state = {
      ...this.state,
      rtc: {
        ...this.state.rtc,
        ...newState,
      }
    }
    this.commit(this.state);
  }

  updateDevice(state: MediaDeviceState) {
    this.state = {
      ...this.state,
      mediaDevice: state
    }
    this.commit(this.state);
  }

  async handlePeerMessage(cmd: RoomMessage, peerId: string) {
    if (!peerId) return console.warn('state is not assigned');
    const myUid = this.state.me.uid;
    console.log("Teacher: ", this.isTeacher(myUid) , ", peerId: ", this.isStudent(peerId));
    // student follow teacher peer message
    if (!this.isTeacher(myUid) && this.isTeacher(peerId)) {

      const me = this.state.me;
      switch(cmd) {
        case RoomMessage.muteChat: {
          return await this.updateMe({...me, chat: 0});
        }
        case RoomMessage.muteAudio: {
          return await this.updateMe({...me, audio: 0});
        }
        case RoomMessage.muteVideo: {
          return await this.updateMe({...me, video: 0});
        }
        case RoomMessage.muteBoard: {
          globalStore.showToast({
            message: `Teacher already cancel your whiteboard`,
            type: 'notice'
          });
          return await this.updateMe({...me, grantBoard: 0});
        }
        case RoomMessage.unmuteAudio: {
          return await this.updateMe({...me, audio: 1});
        }
        case RoomMessage.unmuteVideo: {
          return await this.updateMe({...me, video: 1});
        }
        case RoomMessage.unmuteChat: {
          return await this.updateMe({...me, chat: 1});
        }
        case RoomMessage.unmuteBoard: {
          globalStore.showToast({
            message: `Teacher already permit your whiteboard`,
            type: 'notice'
          });
          return await this.updateMe({...me, grantBoard: 1});
        }
        case RoomMessage.acceptCoVideo: {
          globalStore.showToast({
            type: 'co-video',
            message: 'teacher already accept co-video'
          });
          return;
        }
        case RoomMessage.rejectCoVideo: {
          globalStore.showToast({
            type: 'co-video',
            message: 'teacher already rejected co-video'
          });
          return;
        }
        case RoomMessage.cancelCoVideo: {
          globalStore.showToast({
            type: 'co-video',
            message: 'teacher already canceled co-video'
          });
          return;
        }
        default:
      }
      return;
    }

    // when i m teacher & received student message
    if (this.isTeacher(myUid) && this.isStudent(peerId)) {
      switch(cmd) {
        case RoomMessage.applyCoVideo: {
          // WARN: LOCK
          if (this.state.course.linkId) {
            return console.warn('already received apply id: ', this.applyLock);
          }
          const applyUser = roomStore.state.users.get(`${peerId}`);
          if (applyUser) {
            this.applyLock = +peerId;
            console.log("applyUid: ", this.applyLock);
            this.state = {
              ...this.state,
              applyUid: this.applyLock,
            }
            this.commit(this.state);
            globalStore.showNotice({
              reason: 'peer_hands_up',
              text: `"${applyUser.account}" wants to interact with you`
            });
          }
          return;
        }
        case RoomMessage.cancelCoVideo: {
          // WARN: LOCK
          if (this.state.course.linkId && `${this.state.course.linkId}` === peerId) {
            roomStore.updateCourseLinkUid(0).then(() => {
            }).catch(console.warn);

            globalStore.showToast({
              type: 'co-video',
              message: 'student canceled co-video'
            });
          }
          return;
        }
        default:
      }
      return;
    }
  }

  async mute(uid: string, type: string) {
    const me = this.state.me;
    console.log(">>>>> mute, uid, ", uid, " type: ", type);
    if (me.uid === `${uid}`) {
      if (type === 'audio') {
        await this.updateAttrsBy(me.uid, {
          audio: 0
        });
      }
      if (type === 'video') {
        await this.updateAttrsBy(me.uid, {
          video: 0
        });
      }
      if (type === 'chat') {
        await this.updateAttrsBy(me.uid, {
          chat: 0
        });
      }
      // if (type === 'grantBoard') {
      //   await this.updateAttrsBy(me.uid, {
      //     grant_board: 0
      //   });
      // }
    }
    else if (me.role === 'teacher') {
      if (type === 'audio') {
        await this.rtmClient.sendPeerMessage(`${uid}`, {cmd: RoomMessage.muteAudio});
      }
      if (type === 'video') {
        await this.rtmClient.sendPeerMessage(`${uid}`, {cmd: RoomMessage.muteVideo});
      }
      if (type === 'chat') {
        await this.rtmClient.sendPeerMessage(`${uid}`, {cmd: RoomMessage.muteChat});
      }
      if (type === 'grantBoard') {
        await this.rtmClient.sendPeerMessage(`${uid}`, {cmd: RoomMessage.muteBoard});
      }
    }
  }

  async unmute(uid: string, type: string) {
    const me = this.state.me;
    if (me.uid === `${uid}`) {
      if (type === 'audio') {
        await this.updateAttrsBy(me.uid, {
          audio: 1
        });
      }
      if (type === 'video') {
        await this.updateAttrsBy(me.uid, {
          video: 1
        });
      }
      if (type === 'chat') {
        await this.updateAttrsBy(me.uid, {
          chat: 1
        });
      }
      // if (type === 'grantBoard') {
      //   await this.updateAttrsBy(me.uid, {
      //     grant_board: 1
      //   });
      // }
    }
    else if (me.role === 'teacher') {
      if (type === 'audio') {
        await this.rtmClient.sendPeerMessage(`${uid}`, {cmd: RoomMessage.unmuteAudio});
      }
      if (type === 'video') {
        await this.rtmClient.sendPeerMessage(`${uid}`, {cmd: RoomMessage.unmuteVideo});
      }
      if (type === 'chat') {
        await this.rtmClient.sendPeerMessage(`${uid}`, {cmd: RoomMessage.unmuteChat});
      }
      if (type === 'grantBoard') {
        await this.rtmClient.sendPeerMessage(`${uid}`, {cmd: RoomMessage.unmuteBoard});
      }
    }
  }

  async loginAndJoin(payload: any, pass: boolean = false) {
    payload.rid = `channel_${payload.rid}`
    console.log('channel:', payload.rid);
    const {roomType, role, uid, rid, token} = payload;
    await this.rtmClient.logout();
    await this.rtmClient.login(uid, token);
    const channelMemberCount = await this.rtmClient.getChannelMemberCount([rid]);
    const channelCount = channelMemberCount[rid];
    const channelMeta = await this.rtmClient.getChannelAttributeBy(rid);
    let accounts = channelMeta.accounts;
    const onlineStatus = await this.rtmClient.queryOnlineStatusBy(accounts);
    /*
    const me = accounts.find((o)=>{return `${payload.uid}` === `${o.uid}`});
    if (onlineStatus[payload.uid]) {
      payload.audio = me.audio;
      payload.video = me.video;

    } else {
      if (accounts.length >= STAGE_NUM) {
        payload.audio = 0;
        payload.video = 0;
        // payload.stage = 0;
      } else {
        // payload.stage = 1;
        payload.audio = 0;
        payload.video = 1;
      }
    }*/
    // for default
    payload.audio = 0;
    payload.video = 0;
    if (role === 'teacher') {
      payload.audio = 1;
      payload.video = 1;
    }

    const argsJoin = {
      uid,
      channelCount,
      onlineStatus,
      role,
      accounts,
      roomType
    };
    let result = pass === false ? canJoin(argsJoin) : {permitted: true, reason: ''};

    let studentsOrder: number[] = [];
    if (result.permitted) {
      // studentsOrder = this.state.studentsOrder;
      /*
      if (!studentsOrder.length) {
        accounts.forEach(a => {
          if (!studentsOrder.includes(+a.uid)) {
            studentsOrder.push(+a.uid);
          }
        })
      } else {
        const accs = accounts.filter(a=> a.role != 'teacher');
        const accIds = accs.map(a=> +a.uid);

        // if (accIds.length >= studentsOrder.length) {
          let a = new Set(accIds);
          let b = new Set(studentsOrder);
          let diff = new Set([...a].filter(x => !b.has(x)));
          diff.forEach(d => {
            if (!studentsOrder.includes(+d)) {
              studentsOrder.push(+d);
            }
          })
        // }
      }

      if (!studentsOrder.includes(+uid)) {
        studentsOrder.push(+uid);
      } else {
        if (!pass) {
          const i = studentsOrder.indexOf(+uid)
          studentsOrder.splice(i,1)
          studentsOrder.push(+uid)
        }
      }
      */
      let onlineMe = accounts.find((it: any) => `${it.uid}` == `${payload.uid}`);
      if (onlineMe) {
        payload.video = onlineMe.video;
        payload.audio = onlineMe.audio;
      } else {
        if (channelMeta.studentsOrder && channelMeta.studentsOrder.includes(+payload.uid)) {
          payload.video = 1;
          payload.audio = 1;
        }
      }


      let res = await this.rtmClient.join(rid).then(async r => {
        this.state = {
          ...this.state,
          rtm: {
            ...this.state.rtm,
            joined: true
          }
        }
        const grantBoard = role === 'teacher' ? 1 : 0;
        console.log(">>>>>>>>>>#room: ", grantBoard);
        await this.updateMe({...payload, grantBoard});
        this.commit(this.state);
      });
      // this.state = {
      //   ...this.state,
      //   rtm: {
      //     ...this.state.rtm,
      //     joined: true
      //   }
      // }
      // const grantBoard = role === 'teacher' ? 1 : 0;
      // console.log(">>>>>>>>>>#room: ", grantBoard);
      // await this.updateMe({...payload, grantBoard});
      // this.commit(this.state);
      return;
    }
    await this.rtmClient.logout();
    throw {
      type: 'not_permitted',
      reason: result.reason
    }
  }
  publishMeStream() {
    console.log(3, 'set rtc join flag, should auto publish')
    // return await (this.rtcClient as AgoraWebClient).publishStream()
    // 通过index.tsx里的监听来触发发布
    this.setRTCJoined(true);
  }
  setRTCJoined(joined: boolean) {
    this.state = {
      ...this.state,
      rtc: {
        ...this.state.rtc,
        joined
      }
    }
    this.commit(this.state);
  }

  async updateCourseLinkUid(linkId: number) {
    const me = this.state.me;
    console.log("me: link_uid", me, linkId);
    let res = await this.updateAttrsBy(me.uid, {
      link_uid: linkId
    })
    // let res = await this.updateMe({...me, linkId: linkId});
    this.applyLock = linkId;
    console.log("current apply lock: ", this.applyLock);
    return res;
  }

  async updateWhiteboardUid(uid: string) {
    const me = this.state.me;
    let res = await this.updateAttrsBy(me.uid, {
      whiteboard_uid: uid
    });
    console.log("[update whiteboard uid] res", uid);
    return res;
  }

  updateChannelMessage(msg: ChatMessage) {
    this.state = {
      ...this.state,
      messages: this.state.messages.push(msg)
    };

    this.commit(this.state);
  }

  async updateAttrsBy(uid: string, attr: any) {
    if (attr['whiteboard_uid']) {
      console.log("[update whiteboard uid], ", attr['whiteboard_uid']);
    }
    const user = this.state.users.get(uid);
    if (!user) return;
    const key = user.role === 'teacher' ? 'teacher' : uid;
    const attrs = {
      uid: user.uid,
      whiteboard_uid: user.boardId,
      link_uid: user.linkId,
      shared_uid: user.sharedId,
      account: user.account,
      video: user.video,
      audio: user.audio,
      chat: user.chat,
      grant_board: user.grantBoard,
    }
    if (user.role === 'teacher') {
      Object.assign(attrs, {
        mute_chat: this.state.course.muteChat,
        class_state: this.state.course.courseState,
        whiteboard_uid: this.state.course.boardId,
        link_uid: this.state.course.linkId,
        shared_uid: this.state.course.sharedId,
        current_tool: this.state.course.currentTool,
      })
    }
    if (attr) {
      Object.assign(attrs, attr);
    }
    const data: any = {};
    data[key] = attrs;
    // await this.rtmClient.addOrUpdateChannelAttributes(data);
    let res = await this.rtmClient.updateChannelAttrsByKey(key, attrs);
    // return res;
  }

  async updateMe(user: any) {
    const {role, uid, account, rid, video, audio, chat, boardId, linkId, sharedId, muteChat, grantBoard} = user;
    const key = role === 'teacher' ? 'teacher' : uid;
    const me = this.state.me;
    const attrs = {
      uid: me.uid,
      account: me.account,
      chat: me.chat,
      video: me.video,
      audio: me.audio,
    }
    Object.assign(attrs, {
      uid,
      account,
      video,
      audio,
      chat,
      whiteboard_uid: boardId,
      link_uid: linkId,
      shared_uid: sharedId,
      grant_board: grantBoard,
    });

    console.log('users', user);
    if (grantBoard !== undefined) {
      Object.assign(attrs, {
        grant_board: grantBoard,
      });
    }
    if (role === 'teacher') {
      const class_state = get(user, 'courseState', this.state.course.courseState);
      const whiteboard_uid = get(user, 'boardId', this.state.course.boardId);
      const mute_chat = get(user, 'muteChat', this.state.course.muteChat);
      const shared_uid = get(user, 'sharedId', this.state.course.sharedId);
      const link_uid = get(user, 'linkId', this.state.course.linkId);
      const lock_board = get(user, 'lockBoard', this.state.course.lockBoard);
      const current_tool = get(user, 'currentTool', this.state.course.currentTool);

      Object.assign(attrs, {
        mute_chat,
        class_state,
        whiteboard_uid,
        link_uid,
        shared_uid,
        lock_board,
        current_tool
      })
      console.log("teacher attrs: >>>> ", attrs);
    }
    const data: any = {};
    data[key] = attrs;
    // if (studentsOrder && studentsOrder.length) {
    //   data['studentsOrder'] = studentsOrder;
    // }
    console.log('update me', data);
    await this.rtmClient.addOrUpdateChannelAttributes(data);
    // let res = await this.rtmClient.updateChannelAttrsByKey(key, attrs);
    // return res;
  }
  meUp() {
    console.log('meUp!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!')
  }
  meDown() {
    console.log('meDown!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!')
  }
  async syncStageStudent(accounts: any[], studentsOrder: number[]) {
    debugger
    let  needUpdate =false;
    const data: any = {};
    const accData: any = {};
      accounts.forEach(acc => {
      if (acc.role == 'student') {
        accData[acc.uid] = acc
      }
    })

    // 学生在台上并且退出了
    const accIds: any[] = Object.keys(accData).map(uid => +uid);
    if (!accIds.length) {
      return
    }

    const exitUserIds = studentsOrder.filter( uid => {
      return !accIds.includes(uid);
    });
    if (exitUserIds.length) {
      exitUserIds.forEach( uid => {
        if (studentsOrder.includes(uid)) {
          studentsOrder.splice(studentsOrder.indexOf(uid), 1)
          needUpdate = true
        }
      })

    }
    if (studentsOrder.length != STAGE_NUM) {
      if (studentsOrder.length > STAGE_NUM) {
        //台上队列尾部挤下去的人音视频关闭
        const lostIdx: number[] = [];
        [...Array(studentsOrder.length - STAGE_NUM)].forEach( (_, i) => {
          console.log(i)
          const idx = studentsOrder.length - 1 - i
          lostIdx.push(idx)
        })
        lostIdx.forEach( uid => {
          if (studentsOrder.includes(uid)) {
              if ( accData[uid]) {
                const u =  accData[uid];
                // if (u.video != 0 || u.audio != 0) {
                  u.video = 0;
                  u.audio = 0;
                  data[uid] = u;
                // }
              }
            studentsOrder.splice(studentsOrder.indexOf(uid), 1)
            needUpdate = true
          }
        })
        studentsOrder.length = STAGE_NUM;
      }
      if (studentsOrder.length < STAGE_NUM) {
        const newUid = accIds.find( uid => {
          return !studentsOrder.includes(+uid);
        });
        const newStudents = accData[newUid];
        if (newStudents) {
          const uid = newStudents.uid;
          // if (newStudents.video != 1 || newStudents.audio != 1) {
            //上台的打开音视频
            newStudents.video = 1;
            newStudents.audio = 1;
            data[uid] = newStudents;
            studentsOrder.push(+uid)
            needUpdate = true;
          // }
        }
      }
    }
    if (needUpdate) {
      data['studentsOrder'] = studentsOrder;
      // await this.rtmClient.addOrUpdateChannelAttributes(data);
      return data;
    }
  }

  async updateRoomAttrs ({teacher, accounts, room, studentsOrder}: {teacher: any, accounts: any, room: any, studentsOrder: number[]}) {
    console.log("[agora-board], room:  ",teacher, accounts, room, studentsOrder);

    const users = accounts.reduce((acc: OrderedMap<string, AgoraUser>, it: any) => {
      return acc.set(it.uid, {
        role: it.role,
        account: it.account,
        uid: it.uid,
        video: it.video,
        audio: it.audio,
        chat: it.chat,
        boardId: it.whiteboard_uid,
        sharedId: it.shared_uid,
        linkId: it.link_uid,
        lockBoard: it.lock_board,
        grantBoard: it.grant_board,
      });
    }, OrderedMap<string, AgoraUser>());

    /*
    let users = OrderedMap<string, AgoraUser>();
    for (const it of accounts) {
      users = users.set(it.uid, {
        role: it.role,
        account: it.account,
        uid: it.uid,
        video: it.video,
        audio: it.audio,
        chat: it.chat,
        boardId: it.whiteboard_uid,
        sharedId: it.shared_uid,
        linkId: it.link_uid,
        lockBoard: it.lock_board,
        grantBoard: it.grant_board,
      })
    }
    */
    console.log('users 4', users.toArray());
    const me = this.state.me;

    if (users.get(me.uid)) {
      Object.assign(me, users.get(me.uid));
    }
    // 每个学生都要维护一下状态，因为可能涉及到自己的上下台

    // if (newInfo && newInfo['studentsOrder'] && me.role == 'student') {
    //   if (newInfo['studentsOrder'].includes(+this.state.me.uid)) {
    //     this.meUp();
    //   } else {
    //     this.meDown()
    //   }
    // }
    let newStudentsOrder = studentsOrder;
    if (me.role === 'teacher') {
      Object.assign(me, {
        linkId: room.link_uid,
        boardId: room.whiteboard_uid,
        lockBoard: room.lock_board,
      })
      //教师维护学生顺序,仅保留STAGE_NUM个，并更新到频道属性
      const newInfo: any = await this.syncStageStudent(JSON.parse(JSON.stringify(accounts)), [...studentsOrder]);
      console.log('newInfo', newInfo)
      debugger
      if (newInfo) {
        newStudentsOrder = newInfo['studentsOrder'];
        await this.rtmClient.addOrUpdateChannelAttributes(newInfo);
      }
    }

    const newClassState: ClassState = {} as ClassState;
    Object.assign(newClassState, {
      teacherId: get(teacher, 'uid', 0),
      linkId: room.link_uid,
      boardId: room.whiteboard_uid,
      courseState: room.class_state,
      muteChat: room.mute_chat,
      lockBoard: room.lock_board,
      currentTool: room.current_tool
    })

    // console.log("... me", this.state.me);
    // console.log("... this.state.me", me);

    this.state = {
      ...this.state,
      users,
      me: {
        ...this.state.me,
        ...me,
      },
      course: {
        ...this.state.course,
        ...newClassState
      },
      studentsOrder: newStudentsOrder
    }
    this.commit(this.state);
  }

  updateSessionInfo (info: any) {
    this.state = {
      ...this.state,
      course: {
        ...this.state.course,
        rid: info.rid,
        roomName: info.roomName,
        roomType: info.roomType
      },
      me: {
        ...this.state.me,
        account: info.account,
        uid: info.uid,
        role: info.role,
        video: info.video,
        audio: info.audio,
        chat: info.chat,
        linkId: info.linkId,
        sharedId: info.sharedId,
        boardId: info.boardId,
      }
    }
    this.commit(this.state);
  }

  async exitAll() {
    console.log('exitAll')
    if (this.state.me.role == 'teacher') {
      await this.rtmClient.clearChannelAttributes()
    }
    try {
      await this.rtmClient.exit();
      await this.rtcClient.exit();
    } catch(err) {
      console.warn(err);
    } finally {
      GlobalStorage.clear('agora_room');
      this.state = {} as RoomState;
      this.state = {
        ...this.defaultState
      }
      this.commit(this.state);
    }
  }

  setScreenShare(shared: boolean) {
    this.state = {
      ...this.state,
      rtc: {
        ...this.state.rtc,
        shared,
      }
    }
    this.commit(this.state);
  }
}

export const roomStore = new RoomStore();

//@ts-ignore
window.roomStore = roomStore;
