import { Component, ElementRef, ViewChild, OnInit, Input, OnDestroy, HostListener } from '@angular/core';

import {
  Label,
  MySprite, tweenChange,
  ShapeRect
} from './Unit';
import { res, resAudio } from './resources';

import { Subject } from 'rxjs';
import { debounceTime } from 'rxjs/operators';

import TWEEN from '@tweenjs/tween.js';

import { playAudio, pauseAudio } from './AudioUtil';

import { Title, getDefaultTile } from './Title';

import { Listening } from './Listening';

import { Skin } from './Skin';
import { Exercises } from './Exercises';


@Component({
  selector: 'app-play',
  templateUrl: './play.component.html',
  styleUrls: ['./play.component.css']
})
export class PlayComponent implements OnInit, OnDestroy {

  @ViewChild('canvas', { static: true }) canvas: ElementRef;
  @ViewChild('wrap', { static: true }) wrap: ElementRef;

  // 数据
  data;

  ctx;

  canvasWidth = 1280; // canvas实际宽度
  canvasHeight = 720; // canvas实际高度

  canvasBaseW = 1280; // canvas 资源预设宽度
  canvasBaseH = 720;  // canvas 资源预设高度

  mx; // 点击x坐标
  my; // 点击y坐标


  // 资源
  rawImages = new Map(res);
  rawAudios = new Map(resAudio);

  images = new Map();

  animationId: any;
  winResizeEventStream = new Subject();

  audioObj = {};

  renderArr;
  mapScale = 1;

  mapScaleX = 1;
  mapScaleY = 1;

  canvasLeft;
  canvasTop;

  saveKey = 'YM-28';


  canTouch = true;
  /**标题*/
  title;

  listening;

  exercisesArr = [];

  /**当前播放的音频 */
  curAudio;

  skin;

  curAnswerIndex = 1;

  @HostListener('window:resize', ['$event'])
  onResize(event) {
    this.winResizeEventStream.next();
  }


  ngOnInit() {

    this.data = {};

    // 获取数据
    const getData = (<any>window).courseware.getData;
    getData((data) => {

      if (data && typeof data == 'object') {
        this.data = data;
      }
      console.log('data:', data);

      // 初始化 各事件监听
      this.initListener();

      // 若无数据 则为预览模式 需要填充一些默认数据用来显示
      this.initDefaultData();

      // 初始化 音频资源
      this.initAudio();
      // 初始化 图片资源
      this.initImg();
      // 开始预加载资源
      this.load();

    }, this.saveKey);

  }

  ngOnDestroy() {
    window['curCtx'] = null;
    window.cancelAnimationFrame(this.animationId);
  }


  load() {

    // 预加载资源
    this.loadResources().then(() => {
      window["air"].hideAirClassLoading(this.saveKey, this.data);
      this.init();
      this.update();
    });
  }


  init() {

    this.initCtx();
    this.initData();
    this.initView();
  }

  initCtx() {
    this.canvasWidth = this.wrap.nativeElement.clientWidth;
    this.canvasHeight = this.wrap.nativeElement.clientHeight;
    this.canvas.nativeElement.width = this.wrap.nativeElement.clientWidth;
    this.canvas.nativeElement.height = this.wrap.nativeElement.clientHeight;


    this.ctx = this.canvas.nativeElement.getContext('2d');
    this.canvas.nativeElement.width = this.canvasWidth;
    this.canvas.nativeElement.height = this.canvasHeight;

    window['curCtx'] = this.ctx;
  }






  updateItem(item) {
    if (item) {
      item.update();
    }
  }

  updateArr(arr) {
    if (!arr) {
      return;
    }
    for (let i = 0; i < arr.length; i++) {
      arr[i].update(this);
    }
  }







  initListener() {

    this.winResizeEventStream
      .pipe(debounceTime(500))
      .subscribe(data => {
        this.renderAfterResize();
      });


    // ---------------------------------------------
    const setParentOffset = () => {
      const rect = this.canvas.nativeElement.getBoundingClientRect();
      this.canvasLeft = rect.left;
      this.canvasTop = rect.top;
    };
    const setMxMyByTouch = (event) => {
      if (event.touches.length <= 0) {
        return;
      }
      if (this.canvasLeft == null) {
        setParentOffset();
      }
      this.mx = event.touches[0].pageX - this.canvasLeft;
      this.my = event.touches[0].pageY - this.canvasTop;
    };

    const setMxMyByMouse = (event) => {
      this.mx = event.offsetX;
      this.my = event.offsetY;
    };
    // ---------------------------------------------


    let firstTouch = true;

    const touchDownFunc = (e) => {
      if (firstTouch) {
        firstTouch = false;
        removeMouseListener();
      }
      setMxMyByTouch(e);
      this.mapDown(e);
    };
    const touchMoveFunc = (e) => {
      setMxMyByTouch(e);
      this.mapMove(e);
    };
    const touchUpFunc = (e) => {
      setMxMyByTouch(e);
      this.mapUp(e);
    };

    const mouseDownFunc = (e) => {
      if (firstTouch) {
        firstTouch = false;
        removeTouchListener();
      }
      setMxMyByMouse(e);
      this.mapDown(e);
    };
    const mouseMoveFunc = (e) => {
      setMxMyByMouse(e);
      this.mapMove(e);
    };
    const mouseUpFunc = (e) => {
      setMxMyByMouse(e);
      this.mapUp(e);
    };


    const element = this.canvas.nativeElement;

    const addTouchListener = () => {
      element.addEventListener('touchstart', touchDownFunc);
      element.addEventListener('touchmove', touchMoveFunc);
      element.addEventListener('touchend', touchUpFunc);
      element.addEventListener('touchcancel', touchUpFunc);
    };
    const removeTouchListener = () => {
      element.removeEventListener('touchstart', touchDownFunc);
      element.removeEventListener('touchmove', touchMoveFunc);
      element.removeEventListener('touchend', touchUpFunc);
      element.removeEventListener('touchcancel', touchUpFunc);
    };

    const addMouseListener = () => {
      element.addEventListener('mousedown', mouseDownFunc);
      element.addEventListener('mousemove', mouseMoveFunc);
      element.addEventListener('mouseup', mouseUpFunc);
    };
    const removeMouseListener = () => {
      element.removeEventListener('mousedown', mouseDownFunc);
      element.removeEventListener('mousemove', mouseMoveFunc);
      element.removeEventListener('mouseup', mouseUpFunc);
    };

    addMouseListener();
    addTouchListener();
  }


  playAudio(key, now = false, callback = null) {

    const audio = this.audioObj[key];
    if (audio) {
      if (now) {
        audio.pause();
        audio.currentTime = 0;
      }

      if (callback) {
        audio.onended = () => {
          callback();
        };
      }
      audio.play();
    }
  }



  loadResources() {
    const pr = [];
    this.rawImages.forEach((value, key) => {// 预加载图片

      const p = this.preload(value)
        .then(img => {
          this.images.set(key, img);
        })
        .catch(err => console.log(err));

      pr.push(p);
    });

    this.rawAudios.forEach((value, key) => {// 预加载音频

      const a = this.preloadAudio(value)
        .then(() => {
          // this.images.set(key, img);
        })
        .catch(err => console.log(err));

      pr.push(a);
    });
    return Promise.all(pr);
  }

  preload(url) {
    return new Promise((resolve, reject) => {
      const img = new Image();
      // img.crossOrigin = "anonymous";
      img.onload = () => resolve(img);
      img.onerror = reject;
      img.src = url;
    });
  }

  preloadAudio(url) {
    return new Promise((resolve, reject) => {
      const audio = new Audio();
      audio.oncanplay = (a) => {
        resolve();
      };
      audio.onerror = () => {
        reject();
      };
      audio.src = url;
      audio.load();
    });
  }


  renderAfterResize() {
    this.canvasWidth = this.wrap.nativeElement.clientWidth;
    this.canvasHeight = this.wrap.nativeElement.clientHeight;
    this.init();
  }





  checkClickTarget(target) {

    const rect = target.getBoundingBox();

    if (this.checkPointInRect(this.mx, this.my, rect)) {
      return true;
    }
    return false;
  }

  getWorlRect(target) {

    let rect = target.getBoundingBox();

    if (target.parent) {

      const pRect = this.getWorlRect(target.parent);
      rect.x += pRect.x;
      rect.y += pRect.y;
    }
    return rect;
  }

  checkPointInRect(x, y, rect) {
    if (x >= rect.x && x <= rect.x + rect.width) {
      if (y >= rect.y && y <= rect.y + rect.height) {
        return true;
      }
    }
    return false;
  }





  addUrlToAudioObj(key, url = null, vlomue = 1, loop = false, callback = null) {

    const audioObj = this.audioObj;

    if (url == null) {
      url = key;
    }

    this.rawAudios.set(key, url);

    const audio = new Audio();
    audio.src = url;
    audio.load();
    audio.loop = loop;
    audio.volume = vlomue;

    audioObj[key] = audio;

    if (callback) {
      audio.onended = () => {
        callback();
      };
    }
  }

  addUrlToImages(url) {
    this.rawImages.set(url, url);
  }






  // ======================================================编写区域==========================================================================





  /**
   * 添加默认数据 便于无数据时的展示
   */
  initDefaultData() {

    const getDefaultExercisesItem = () => {
      const exercises = {
        text: 'Lion',
        index: '',
        audio_url: '',
        index_audio_url: ''
      };

      return exercises;
    }

    if (!this.data.title) {
      this.data.title = getDefaultTile();

      if (!this.data.exercisesArr) {
        this.data.exercisesArr = [];

        for (let i = 0; i < 9; ++i) {
          let exer = getDefaultExercisesItem();
          exer.index = "" + 1 + i;
          this.data.exercisesArr.push(exer);
        }
      }
    }

    //设置默认皮肤
    if (!this.data.skin) {
      this.data.skin = 'A';
    }

  }


  /**
   * 添加预加载图片
   */
  initImg() {

  }

  /**
   * 添加预加载音频
   */
  initAudio() {

    // 音效
    this.addUrlToAudioObj('click', this.rawAudios.get('click'), 0.3);

    this.addUrlToAudioObj('right', this.rawAudios.get('right'));

    //标题发音
    this.addUrlToAudioObj('titleAudio', this.data.title.audio_url);

    //听力材料
    this.addUrlToAudioObj('listenAudio', this.data.listenAudioUrl);

    //练习题音效
    for (let i = 0; i < this.data.exercisesArr.length; ++i) {

      let exercises = this.data.exercisesArr[i];

      this.addUrlToAudioObj('exercises_' + i, exercises.audio_url);
      this.addUrlToAudioObj('exercises_index_' + i, exercises.index_audio_url);

    }
  }



  /**
   * 初始化数据
   */
  initData() {

    const sx = this.canvasWidth / this.canvasBaseW;
    const sy = this.canvasHeight / this.canvasBaseH;
    const s = Math.min(sx, sy);
    this.mapScale = s;

    this.mapScaleX = sx;
    this.mapScaleY = sy;


    this.renderArr = [];

    this.exercisesArr = [];

    this.curAnswerIndex = 1;

    if (this.curAudio) {
      this.curAudio.pause();
    }

  }



  /**
   * 初始化试图
   */
  initView() {

    this.initBg();

    this.initTitle();

    this.initBottomPart();

    this.initExercises();
  }

  initTitle() {

    const title = new Title(this.data.title, this.images);

    title.init(this.mapScale, this.mapScale);
    title.audio = this.audioObj['titleAudio'];
    this.title = title;

    title.titleBg.y = 0;
    title.titleBg.x = 65 * this.mapScaleX;

    this.renderArr.push(title.titleBg);
  }

  initBottomPart() {

    const listening = new Listening(this.audioObj['listenAudio'], this.images);
    listening.init(this.mapScale, this.mapScale);

    listening.bg.x = this.canvasWidth - listening.bg.width - 10 * this.mapScale;
    listening.bg.y = this.canvasHeight - listening.bg.height - 8 * this.mapScale;
    this.listening = listening;

    this.renderArr.push(listening.bg);

  }

  initBg() {

    const skin = new Skin(this.data.skin, this.images);
    skin.setScaleXY(this.mapScale);

    skin.initBg(this.renderArr, this.canvasWidth, this.canvasHeight);

    this.skin = skin;
  }

  /**
   * 初始化练习题
   */
  initExercises() {
    for (let i = 0; i < this.data.exercisesArr.length; ++i) {

      let exercises = this.data.exercisesArr[i];

      let exercisesObj = new Exercises(exercises, this.images);
      exercisesObj.audio = this.audioObj['exercises_' + i];
      exercisesObj.indexAudio = this.audioObj['exercises_index_' + i];
      exercisesObj.scaleX = this.mapScale;
      exercisesObj.scaleY = this.mapScale;
      exercisesObj.init(this.skin, i);

      this.exercisesArr.push(exercisesObj);

    }

    let intervalHeight = this.skin.getIntervalHeight();

    let intervalWidth = this.skin.getIntervalWidth();

    //背景固定宽度
    let bgWidth = this.skin.getContentBgWidth();

    //显示行
    let row = 0;

    //显示列
    let col = 0;

    if (this.exercisesArr.length <= 3) {
      row = 1;
      col = this.exercisesArr.length;
    }
    else if (this.exercisesArr.length == 4) {
      row = 2;
      col = 2;
    }
    else {
      col = 3;
      row = Math.ceil(this.exercisesArr.length / col);
    }

    //最大宽度
    let maxWidth = col * bgWidth + (col - 1) * intervalWidth;

    //起始x坐标
    let startX = (this.canvasWidth - maxWidth) / 2;


    //最大高度
    let tempMaxHeight = 0;

    let maxHeightArr = [];

    //计算每一行的最大高度
    for (let i = 0; i < this.exercisesArr.length; ++i) {

      let index = i + 1;

      if (tempMaxHeight < this.exercisesArr[i].getHeight()) {
        tempMaxHeight = this.exercisesArr[i].getHeight();
      }

      if (index % col == 0) {
        //行结尾
        maxHeightArr.push(tempMaxHeight);

        tempMaxHeight = 0;
      }
      else if (index % col != 0 && index == this.exercisesArr.length) {
        //最后一个
        maxHeightArr.push(tempMaxHeight);

        tempMaxHeight = 0;
      }

    }

    for (let i = 0; i < this.exercisesArr.length; ++i) {

      let j = Math.floor(i / col);

      let height = maxHeightArr[j];

      this.exercisesArr[i].initBg2(bgWidth, height);
    }

    let totalHeight = 0;

    for (let i = 0; i < maxHeightArr.length; ++i) {

      totalHeight += maxHeightArr[i];

      //如果不是最后一行，则增加间隔
      if (i != maxHeightArr.length - 1) {
        totalHeight += intervalHeight;
      }
    }

    //起始y坐标
    let startY = (this.canvasHeight - totalHeight) / 2;

    startY = startY > this.skin.getMinMarginTop() ? startY : this.skin.getMinMarginTop();

    let tempStartX = startX;

    //循环设置显示坐标
    for (let i = 0; i < this.exercisesArr.length; ++i) {

      let index = i + 1;

      let exercisesObj = this.exercisesArr[i];


      exercisesObj.bg2.x = tempStartX;

      exercisesObj.bg2.y = startY;

      //最后一列
      if (index % col == 0) {

        //y坐标增加
        startY += exercisesObj.bg2.height + intervalHeight;

        //x坐标复位
        tempStartX = startX;
      }
      else if (index % col != 0 && index == this.exercisesArr.length) {
        //y坐标增加
        startY += exercisesObj.bg2.height + intervalHeight;
      }
      else {
        //x坐标增加
        tempStartX += exercisesObj.bg2.width + intervalWidth;
      }

      this.renderArr.push(exercisesObj.bg2);

    }

  }





  mapDown(event) {

    if (!this.canTouch) {
      return;
    }

    if (this.curAudio) {
      if (this.curAudio != this.listening.audio) {
        this.curAudio.pause();
      }
    }

    if (this.checkClickTarget(this.listening.bg)) {

      this.curAudio = this.listening.play();

      return;
    }

    if (this.curAudio) {
      if (this.curAudio == this.listening.audio) {
        this.listening.pauseAudio();
      }
      else {
        this.curAudio.pause();
      }

    }

    if (this.checkClickTarget(this.title.titleBg)) {
      this.curAudio = this.title.playAudio();
      return;
    }

    for (let i = 0; i < this.exercisesArr.length; ++i) {

      let exercisesObj = this.exercisesArr[i];

      if (this.checkClickTarget(exercisesObj.indexBg)) {
        let flag = exercisesObj.checkRight(this.curAnswerIndex);

        if (flag) {
          this.playAudio('right', true, () => {
            this.curAudio = exercisesObj.playAudio('I');
          });
          this.curAnswerIndex += 1;
          return;
        }
        else {
          if (exercisesObj.indexShowed) {
            this.curAudio = exercisesObj.playAudio('I');
          }
          else {
            this.curAudio = exercisesObj.playAudio('T');
          }
        }

        return;
      }

      if (this.checkClickTarget(exercisesObj.bg)) {
        this.curAudio = exercisesObj.playAudio('T');
        return;
      }
    }

  }

  mapMove(event) {

  }

  mapUp(event) {

  }



  update() {

    // ----------------------------------------------------------
    this.animationId = window.requestAnimationFrame(this.update.bind(this));
    // 清除画布内容
    this.ctx.clearRect(0, 0, this.canvasWidth, this.canvasHeight);
    // tween 更新动画
    TWEEN.update();
    // ----------------------------------------------------------



    this.updateArr(this.renderArr);


  }



}
