(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("@angular/compiler-cli/src/ngtsc/imports/src/emitter", ["require", "exports", "tslib", "@angular/compiler", "typescript", "@angular/compiler-cli/src/ngtsc/file_system", "@angular/compiler-cli/src/ngtsc/file_system/src/util", "@angular/compiler-cli/src/ngtsc/util/src/typescript", "@angular/compiler-cli/src/ngtsc/imports/src/find_export"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    var tslib_1 = require("tslib");
    /**
     * @license
     * Copyright Google Inc. All Rights Reserved.
     *
     * Use of this source code is governed by an MIT-style license that can be
     * found in the LICENSE file at https://angular.io/license
     */
    var compiler_1 = require("@angular/compiler");
    var ts = require("typescript");
    var file_system_1 = require("@angular/compiler-cli/src/ngtsc/file_system");
    var util_1 = require("@angular/compiler-cli/src/ngtsc/file_system/src/util");
    var typescript_1 = require("@angular/compiler-cli/src/ngtsc/util/src/typescript");
    var find_export_1 = require("@angular/compiler-cli/src/ngtsc/imports/src/find_export");
    /**
     * Flags which alter the imports generated by the `ReferenceEmitter`.
     */
    var ImportFlags;
    (function (ImportFlags) {
        ImportFlags[ImportFlags["None"] = 0] = "None";
        /**
         * Force the generation of a new import when generating a reference, even if an identifier already
         * exists in the target file which could be used instead.
         *
         * This is sometimes required if there's a risk TypeScript might remove imports during emit.
         */
        ImportFlags[ImportFlags["ForceNewImport"] = 1] = "ForceNewImport";
        /**
         * Don't make use of any aliasing information when emitting a reference.
         *
         * This is sometimes required if emitting into a context where generated references will be fed
         * into TypeScript and type-checked (such as in template type-checking).
         */
        ImportFlags[ImportFlags["NoAliasing"] = 2] = "NoAliasing";
        /**
         * Indicates that an import to a type-only declaration is allowed.
         *
         * For references that occur in type-positions, the referred declaration may be a type-only
         * declaration that is not retained during emit. Including this flag allows to emit references to
         * type-only declarations as used in e.g. template type-checking.
         */
        ImportFlags[ImportFlags["AllowTypeImports"] = 4] = "AllowTypeImports";
    })(ImportFlags = exports.ImportFlags || (exports.ImportFlags = {}));
    /**
     * Generates `Expression`s which refer to `Reference`s in a given context.
     *
     * A `ReferenceEmitter` uses one or more `ReferenceEmitStrategy` implementations to produce an
     * `Expression` which refers to a `Reference` in the context of a particular file.
     */
    var ReferenceEmitter = /** @class */ (function () {
        function ReferenceEmitter(strategies) {
            this.strategies = strategies;
        }
        ReferenceEmitter.prototype.emit = function (ref, context, importFlags) {
            var e_1, _a;
            if (importFlags === void 0) { importFlags = ImportFlags.None; }
            try {
                for (var _b = tslib_1.__values(this.strategies), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var strategy = _c.value;
                    var emitted = strategy.emit(ref, context, importFlags);
                    if (emitted !== null) {
                        return emitted;
                    }
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_1) throw e_1.error; }
            }
            throw new Error("Unable to write a reference to " + typescript_1.nodeNameForError(ref.node) + " in " + ref.node.getSourceFile().fileName + " from " + context.fileName);
        };
        return ReferenceEmitter;
    }());
    exports.ReferenceEmitter = ReferenceEmitter;
    /**
     * A `ReferenceEmitStrategy` which will refer to declarations by any local `ts.Identifier`s, if
     * such identifiers are available.
     */
    var LocalIdentifierStrategy = /** @class */ (function () {
        function LocalIdentifierStrategy() {
        }
        LocalIdentifierStrategy.prototype.emit = function (ref, context, importFlags) {
            // If the emitter has specified ForceNewImport, then LocalIdentifierStrategy should not use a
            // local identifier at all, *except* in the source file where the node is actually declared.
            if (importFlags & ImportFlags.ForceNewImport &&
                typescript_1.getSourceFile(ref.node) !== typescript_1.getSourceFile(context)) {
                return null;
            }
            // A Reference can have multiple identities in different files, so it may already have an
            // Identifier in the requested context file.
            var identifier = ref.getIdentityIn(context);
            if (identifier !== null) {
                return new compiler_1.WrappedNodeExpr(identifier);
            }
            else {
                return null;
            }
        };
        return LocalIdentifierStrategy;
    }());
    exports.LocalIdentifierStrategy = LocalIdentifierStrategy;
    /**
     * A `ReferenceEmitStrategy` which will refer to declarations that come from `node_modules` using
     * an absolute import.
     *
     * Part of this strategy involves looking at the target entry point and identifying the exported
     * name of the targeted declaration, as it might be different from the declared name (e.g. a
     * directive might be declared as FooDirImpl, but exported as FooDir). If no export can be found
     * which maps back to the original directive, an error is thrown.
     */
    var AbsoluteModuleStrategy = /** @class */ (function () {
        function AbsoluteModuleStrategy(program, checker, moduleResolver, reflectionHost) {
            this.program = program;
            this.checker = checker;
            this.moduleResolver = moduleResolver;
            this.reflectionHost = reflectionHost;
            /**
             * A cache of the exports of specific modules, because resolving a module to its exports is a
             * costly operation.
             */
            this.moduleExportsCache = new Map();
        }
        AbsoluteModuleStrategy.prototype.emit = function (ref, context, importFlags) {
            if (ref.bestGuessOwningModule === null) {
                // There is no module name available for this Reference, meaning it was arrived at via a
                // relative path.
                return null;
            }
            else if (!typescript_1.isDeclaration(ref.node)) {
                // It's not possible to import something which isn't a declaration.
                throw new Error("Debug assert: unable to import a Reference to non-declaration of type " + ts.SyntaxKind[ref.node.kind] + ".");
            }
            else if ((importFlags & ImportFlags.AllowTypeImports) === 0 && typescript_1.isTypeDeclaration(ref.node)) {
                throw new Error("Importing a type-only declaration of type " + ts.SyntaxKind[ref.node.kind] + " in a value position is not allowed.");
            }
            // Try to find the exported name of the declaration, if one is available.
            var _a = ref.bestGuessOwningModule, specifier = _a.specifier, resolutionContext = _a.resolutionContext;
            var symbolName = this.resolveImportName(specifier, ref.node, resolutionContext);
            if (symbolName === null) {
                // TODO(alxhub): make this error a ts.Diagnostic pointing at whatever caused this import to be
                // triggered.
                throw new Error("Symbol " + ref.debugName + " declared in " + typescript_1.getSourceFile(ref.node).fileName + " is not exported from " + specifier + " (import into " + context.fileName + ")");
            }
            return new compiler_1.ExternalExpr(new compiler_1.ExternalReference(specifier, symbolName));
        };
        AbsoluteModuleStrategy.prototype.resolveImportName = function (moduleName, target, fromFile) {
            var exports = this.getExportsOfModule(moduleName, fromFile);
            if (exports !== null && exports.has(target)) {
                return exports.get(target);
            }
            else {
                return null;
            }
        };
        AbsoluteModuleStrategy.prototype.getExportsOfModule = function (moduleName, fromFile) {
            if (!this.moduleExportsCache.has(moduleName)) {
                this.moduleExportsCache.set(moduleName, this.enumerateExportsOfModule(moduleName, fromFile));
            }
            return this.moduleExportsCache.get(moduleName);
        };
        AbsoluteModuleStrategy.prototype.enumerateExportsOfModule = function (specifier, fromFile) {
            // First, resolve the module specifier to its entry point, and get the ts.Symbol for it.
            var entryPointFile = this.moduleResolver.resolveModule(specifier, fromFile);
            if (entryPointFile === null) {
                return null;
            }
            var exports = this.reflectionHost.getExportsOfModule(entryPointFile);
            if (exports === null) {
                return null;
            }
            var exportMap = new Map();
            exports.forEach(function (declaration, name) {
                // It's okay to skip inline declarations, since by definition they're not target-able with a
                // ts.Declaration anyway.
                if (declaration.node !== null) {
                    exportMap.set(declaration.node, name);
                }
            });
            return exportMap;
        };
        return AbsoluteModuleStrategy;
    }());
    exports.AbsoluteModuleStrategy = AbsoluteModuleStrategy;
    /**
     * A `ReferenceEmitStrategy` which will refer to declarations via relative paths, provided they're
     * both in the logical project "space" of paths.
     *
     * This is trickier than it sounds, as the two files may be in different root directories in the
     * project. Simply calculating a file system relative path between the two is not sufficient.
     * Instead, `LogicalProjectPath`s are used.
     */
    var LogicalProjectStrategy = /** @class */ (function () {
        function LogicalProjectStrategy(reflector, logicalFs) {
            this.reflector = reflector;
            this.logicalFs = logicalFs;
        }
        LogicalProjectStrategy.prototype.emit = function (ref, context) {
            var destSf = typescript_1.getSourceFile(ref.node);
            // Compute the relative path from the importing file to the file being imported. This is done
            // as a logical path computation, because the two files might be in different rootDirs.
            var destPath = this.logicalFs.logicalPathOfSf(destSf);
            if (destPath === null) {
                // The imported file is not within the logical project filesystem.
                return null;
            }
            var originPath = this.logicalFs.logicalPathOfSf(context);
            if (originPath === null) {
                throw new Error("Debug assert: attempt to import from " + context.fileName + " but it's outside the program?");
            }
            // There's no way to emit a relative reference from a file to itself.
            if (destPath === originPath) {
                return null;
            }
            var name = find_export_1.findExportedNameOfNode(ref.node, destSf, this.reflector);
            if (name === null) {
                // The target declaration isn't exported from the file it's declared in. This is an issue!
                return null;
            }
            // With both files expressed as LogicalProjectPaths, getting the module specifier as a relative
            // path is now straightforward.
            var moduleName = file_system_1.LogicalProjectPath.relativePathBetween(originPath, destPath);
            return new compiler_1.ExternalExpr({ moduleName: moduleName, name: name });
        };
        return LogicalProjectStrategy;
    }());
    exports.LogicalProjectStrategy = LogicalProjectStrategy;
    /**
     * A `ReferenceEmitStrategy` which constructs relatives paths between `ts.SourceFile`s.
     *
     * This strategy can be used if there is no `rootDir`/`rootDirs` structure for the project which
     * necessitates the stronger logic of `LogicalProjectStrategy`.
     */
    var RelativePathStrategy = /** @class */ (function () {
        function RelativePathStrategy(reflector) {
            this.reflector = reflector;
        }
        RelativePathStrategy.prototype.emit = function (ref, context) {
            var destSf = typescript_1.getSourceFile(ref.node);
            var moduleName = util_1.stripExtension(file_system_1.relative(file_system_1.dirname(file_system_1.absoluteFromSourceFile(context)), file_system_1.absoluteFromSourceFile(destSf)));
            if (!moduleName.startsWith('../')) {
                moduleName = ('./' + moduleName);
            }
            var name = find_export_1.findExportedNameOfNode(ref.node, destSf, this.reflector);
            return new compiler_1.ExternalExpr({ moduleName: moduleName, name: name });
        };
        return RelativePathStrategy;
    }());
    exports.RelativePathStrategy = RelativePathStrategy;
    /**
     * A `ReferenceEmitStrategy` which uses a `UnifiedModulesHost` to generate absolute import
     * references.
     */
    var UnifiedModulesStrategy = /** @class */ (function () {
        function UnifiedModulesStrategy(reflector, unifiedModulesHost) {
            this.reflector = reflector;
            this.unifiedModulesHost = unifiedModulesHost;
        }
        UnifiedModulesStrategy.prototype.emit = function (ref, context) {
            var destSf = typescript_1.getSourceFile(ref.node);
            var name = find_export_1.findExportedNameOfNode(ref.node, destSf, this.reflector);
            if (name === null) {
                return null;
            }
            var moduleName = this.unifiedModulesHost.fileNameToModuleName(destSf.fileName, context.fileName);
            return new compiler_1.ExternalExpr({ moduleName: moduleName, name: name });
        };
        return UnifiedModulesStrategy;
    }());
    exports.UnifiedModulesStrategy = UnifiedModulesStrategy;
});
//# sourceMappingURL=data:application/json;base64,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