/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { SimpleChange } from '../../interface/simple_change';
import { EMPTY_OBJ } from '../empty';
var PRIVATE_PREFIX = '__ngOnChanges_';
/**
 * The NgOnChangesFeature decorates a component with support for the ngOnChanges
 * lifecycle hook, so it should be included in any component that implements
 * that hook.
 *
 * If the component or directive uses inheritance, the NgOnChangesFeature MUST
 * be included as a feature AFTER {@link InheritDefinitionFeature}, otherwise
 * inherited properties will not be propagated to the ngOnChanges lifecycle
 * hook.
 *
 * Example usage:
 *
 * ```
 * static ɵcmp = defineComponent({
 *   ...
 *   inputs: {name: 'publicName'},
 *   features: [NgOnChangesFeature]
 * });
 * ```
 *
 * @codeGenApi
 */
export function ɵɵNgOnChangesFeature(definition) {
    if (definition.type.prototype.ngOnChanges) {
        definition.setInput = ngOnChangesSetInput;
        definition.onChanges = wrapOnChanges();
    }
}
// This option ensures that the ngOnChanges lifecycle hook will be inherited
// from superclasses (in InheritDefinitionFeature).
/** @nocollapse */
// tslint:disable-next-line:no-toplevel-property-access
ɵɵNgOnChangesFeature.ngInherit = true;
function wrapOnChanges() {
    return function wrapOnChangesHook_inPreviousChangesStorage() {
        var simpleChangesStore = getSimpleChangesStore(this);
        var current = simpleChangesStore && simpleChangesStore.current;
        if (current) {
            var previous = simpleChangesStore.previous;
            if (previous === EMPTY_OBJ) {
                simpleChangesStore.previous = current;
            }
            else {
                // New changes are copied to the previous store, so that we don't lose history for inputs
                // which were not changed this time
                for (var key in current) {
                    previous[key] = current[key];
                }
            }
            simpleChangesStore.current = null;
            this.ngOnChanges(current);
        }
    };
}
function ngOnChangesSetInput(instance, value, publicName, privateName) {
    var simpleChangesStore = getSimpleChangesStore(instance) ||
        setSimpleChangesStore(instance, { previous: EMPTY_OBJ, current: null });
    var current = simpleChangesStore.current || (simpleChangesStore.current = {});
    var previous = simpleChangesStore.previous;
    var declaredName = this.declaredInputs[publicName];
    var previousChange = previous[declaredName];
    current[declaredName] = new SimpleChange(previousChange && previousChange.currentValue, value, previous === EMPTY_OBJ);
    instance[privateName] = value;
}
var SIMPLE_CHANGES_STORE = '__ngSimpleChanges__';
function getSimpleChangesStore(instance) {
    return instance[SIMPLE_CHANGES_STORE] || null;
}
function setSimpleChangesStore(instance, store) {
    return instance[SIMPLE_CHANGES_STORE] = store;
}
//# sourceMappingURL=data:application/json;base64,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