/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { utf8Encode } from '../util';
// https://docs.google.com/document/d/1U1RGAehQwRypUTovF1KRlpiOFze0b-_2gc6fAH0KY0k/edit
const VERSION = 3;
const JS_B64_PREFIX = '# sourceMappingURL=data:application/json;base64,';
export class SourceMapGenerator {
    constructor(file = null) {
        this.file = file;
        this.sourcesContent = new Map();
        this.lines = [];
        this.lastCol0 = 0;
        this.hasMappings = false;
    }
    // The content is `null` when the content is expected to be loaded using the URL
    addSource(url, content = null) {
        if (!this.sourcesContent.has(url)) {
            this.sourcesContent.set(url, content);
        }
        return this;
    }
    addLine() {
        this.lines.push([]);
        this.lastCol0 = 0;
        return this;
    }
    addMapping(col0, sourceUrl, sourceLine0, sourceCol0) {
        if (!this.currentLine) {
            throw new Error(`A line must be added before mappings can be added`);
        }
        if (sourceUrl != null && !this.sourcesContent.has(sourceUrl)) {
            throw new Error(`Unknown source file "${sourceUrl}"`);
        }
        if (col0 == null) {
            throw new Error(`The column in the generated code must be provided`);
        }
        if (col0 < this.lastCol0) {
            throw new Error(`Mapping should be added in output order`);
        }
        if (sourceUrl && (sourceLine0 == null || sourceCol0 == null)) {
            throw new Error(`The source location must be provided when a source url is provided`);
        }
        this.hasMappings = true;
        this.lastCol0 = col0;
        this.currentLine.push({ col0, sourceUrl, sourceLine0, sourceCol0 });
        return this;
    }
    /**
    * @internal strip this from published d.ts files due to
    * https://github.com/microsoft/TypeScript/issues/36216
    */
    get currentLine() { return this.lines.slice(-1)[0]; }
    toJSON() {
        if (!this.hasMappings) {
            return null;
        }
        const sourcesIndex = new Map();
        const sources = [];
        const sourcesContent = [];
        Array.from(this.sourcesContent.keys()).forEach((url, i) => {
            sourcesIndex.set(url, i);
            sources.push(url);
            sourcesContent.push(this.sourcesContent.get(url) || null);
        });
        let mappings = '';
        let lastCol0 = 0;
        let lastSourceIndex = 0;
        let lastSourceLine0 = 0;
        let lastSourceCol0 = 0;
        this.lines.forEach(segments => {
            lastCol0 = 0;
            mappings += segments
                .map(segment => {
                // zero-based starting column of the line in the generated code
                let segAsStr = toBase64VLQ(segment.col0 - lastCol0);
                lastCol0 = segment.col0;
                if (segment.sourceUrl != null) {
                    // zero-based index into the “sources” list
                    segAsStr +=
                        toBase64VLQ(sourcesIndex.get(segment.sourceUrl) - lastSourceIndex);
                    lastSourceIndex = sourcesIndex.get(segment.sourceUrl);
                    // the zero-based starting line in the original source
                    segAsStr += toBase64VLQ(segment.sourceLine0 - lastSourceLine0);
                    lastSourceLine0 = segment.sourceLine0;
                    // the zero-based starting column in the original source
                    segAsStr += toBase64VLQ(segment.sourceCol0 - lastSourceCol0);
                    lastSourceCol0 = segment.sourceCol0;
                }
                return segAsStr;
            })
                .join(',');
            mappings += ';';
        });
        mappings = mappings.slice(0, -1);
        return {
            'file': this.file || '',
            'version': VERSION,
            'sourceRoot': '',
            'sources': sources,
            'sourcesContent': sourcesContent,
            'mappings': mappings,
        };
    }
    toJsComment() {
        return this.hasMappings ? '//' + JS_B64_PREFIX + toBase64String(JSON.stringify(this, null, 0)) :
            '';
    }
}
export function toBase64String(value) {
    let b64 = '';
    value = utf8Encode(value);
    for (let i = 0; i < value.length;) {
        const i1 = value.charCodeAt(i++);
        const i2 = value.charCodeAt(i++);
        const i3 = value.charCodeAt(i++);
        b64 += toBase64Digit(i1 >> 2);
        b64 += toBase64Digit(((i1 & 3) << 4) | (isNaN(i2) ? 0 : i2 >> 4));
        b64 += isNaN(i2) ? '=' : toBase64Digit(((i2 & 15) << 2) | (i3 >> 6));
        b64 += isNaN(i2) || isNaN(i3) ? '=' : toBase64Digit(i3 & 63);
    }
    return b64;
}
function toBase64VLQ(value) {
    value = value < 0 ? ((-value) << 1) + 1 : value << 1;
    let out = '';
    do {
        let digit = value & 31;
        value = value >> 5;
        if (value > 0) {
            digit = digit | 32;
        }
        out += toBase64Digit(digit);
    } while (value > 0);
    return out;
}
const B64_DIGITS = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';
function toBase64Digit(value) {
    if (value < 0 || value >= 64) {
        throw new Error(`Can only encode value in the range [0, 63]`);
    }
    return B64_DIGITS[value];
}
//# sourceMappingURL=data:application/json;base64,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