/**
 * @license Angular v9.0.7
 * (c) 2010-2020 Google LLC. https://angular.io/
 * License: MIT
 */

import { Observable } from 'rxjs';
import { Subject } from 'rxjs';
import { Subscription } from 'rxjs';

/**
 * @description
 *
 * Represents an abstract class `T`, if applied to a concrete class it would stop being
 * instantiatable.
 *
 * @publicApi
 */
import * as ɵngcc0 from './src/r3_symbols';
export declare interface AbstractType<T> extends Function {
    prototype: T;
}

/**
 * Below are constants for LContainer indices to help us look up LContainer members
 * without having to remember the specific indices.
 * Uglify will inline these when minifying so there shouldn't be a cost.
 */
declare const ACTIVE_INDEX = 2;

/**
 * Used to track:
 *  - Inline embedded views (see: `ɵɵembeddedViewStart`)
 *  - Transplanted `LView`s (see: `LView[DECLARATION_COMPONENT_VIEW])`
 */
declare const enum ActiveIndexFlag {
    /**
     * Flag which signifies that the `LContainer` does not have any inline embedded views.
     */
    DYNAMIC_EMBEDDED_VIEWS_ONLY = -1,
    /**
     * Flag to signify that this `LContainer` may have transplanted views which need to be change
     * detected. (see: `LView[DECLARATION_COMPONENT_VIEW])`.
     *
     * This flag once set is never unset for the `LContainer`. This means that when unset we can skip
     * a lot of work in `refreshDynamicEmbeddedViews`. But when set we still need to verify
     * that the `MOVED_VIEWS` are transplanted and on-push.
     */
    HAS_TRANSPLANTED_VIEWS = 1,
    /**
     * Number of bits to shift inline embedded views counter to make space for other flags.
     */
    SHIFT = 1,
    /**
     * When incrementing the active index for inline embedded views, the amount to increment to leave
     * space for other flags.
     */
    INCREMENT = 2
}

/**
 * @description
 * A lifecycle hook that is called after the default change detector has
 * completed checking all content of a directive.
 *
 * @see `AfterViewChecked`
 * @see [Lifecycle Hooks](guide/lifecycle-hooks#onchanges) guide
 *
 * @usageNotes
 * The following snippet shows how a component can implement this interface to
 * define its own after-check functionality.
 *
 * {@example core/ts/metadata/lifecycle_hooks_spec.ts region='AfterContentChecked'}
 *
 * @publicApi
 */
export declare interface AfterContentChecked {
    /**
     * A callback method that is invoked immediately after the
     * default change detector has completed checking all of the directive's
     * content.
     */
    ngAfterContentChecked(): void;
}

/**
 * @description
 * A lifecycle hook that is called after Angular has fully initialized
 * all content of a directive.
 * Define an `ngAfterContentInit()` method to handle any additional initialization tasks.
 *
 * @see `OnInit`
 * @see `AfterViewInit`
 * @see [Lifecycle Hooks](guide/lifecycle-hooks#onchanges) guide
 *
 * @usageNotes
 * The following snippet shows how a component can implement this interface to
 * define its own content initialization method.
 *
 * {@example core/ts/metadata/lifecycle_hooks_spec.ts region='AfterContentInit'}
 *
 * @publicApi
 */
export declare interface AfterContentInit {
    /**
     * A callback method that is invoked immediately after
     * Angular has completed initialization of all of the directive's
     * content.
     * It is invoked only once when the directive is instantiated.
     */
    ngAfterContentInit(): void;
}

/**
 * @description
 * A lifecycle hook that is called after the default change detector has
 * completed checking a component's view for changes.
 *
 * @see `AfterContentChecked`
 * @see [Lifecycle Hooks](guide/lifecycle-hooks#onchanges) guide
 *
 * @usageNotes
 * The following snippet shows how a component can implement this interface to
 * define its own after-check functionality.
 *
 * {@example core/ts/metadata/lifecycle_hooks_spec.ts region='AfterViewChecked'}
 *
 * @publicApi
 */
export declare interface AfterViewChecked {
    /**
     * A callback method that is invoked immediately after the
     * default change detector has completed one change-check cycle
     * for a component's view.
     */
    ngAfterViewChecked(): void;
}

/**
 * @description
 * A lifecycle hook that is called after Angular has fully initialized
 * a component's view.
 * Define an `ngAfterViewInit()` method to handle any additional initialization tasks.
 *
 * @see `OnInit`
 * @see `AfterContentInit`
 * @see [Lifecycle Hooks](guide/lifecycle-hooks#onchanges) guide
 *
 * @usageNotes
 * The following snippet shows how a component can implement this interface to
 * define its own view initialization method.
 *
 * {@example core/ts/metadata/lifecycle_hooks_spec.ts region='AfterViewInit'}
 *
 * @publicApi
 */
export declare interface AfterViewInit {
    /**
     * A callback method that is invoked immediately after
     * Angular has completed initialization of a component's view.
     * It is invoked only once when the view is instantiated.
     *
     */
    ngAfterViewInit(): void;
}

/**
 * A DI token that you can use to create a virtual [provider](guide/glossary#provider)
 * that will populate the `entryComponents` field of components and NgModules
 * based on its `useValue` property value.
 * All components that are referenced in the `useValue` value (either directly
 * or in a nested array or map) are added to the `entryComponents` property.
 *
 * @usageNotes
 *
 * The following example shows how the router can populate the `entryComponents`
 * field of an NgModule based on a router configuration that refers
 * to components.
 *
 * ```typescript
 * // helper function inside the router
 * function provideRoutes(routes) {
 *   return [
 *     {provide: ROUTES, useValue: routes},
 *     {provide: ANALYZE_FOR_ENTRY_COMPONENTS, useValue: routes, multi: true}
 *   ];
 * }
 *
 * // user code
 * let routes = [
 *   {path: '/root', component: RootComp},
 *   {path: '/teams', component: TeamsComp}
 * ];
 *
 * @NgModule({
 *   providers: [provideRoutes(routes)]
 * })
 * class ModuleWithRoutes {}
 * ```
 *
 * @publicApi
 * @deprecated Since 9.0.0. With Ivy, this property is no longer necessary.
 */
export declare const ANALYZE_FOR_ENTRY_COMPONENTS: InjectionToken<any>;

/**
 * All callbacks provided via this token will be called for every component that is bootstrapped.
 * Signature of the callback:
 *
 * `(componentRef: ComponentRef) => void`.
 *
 * @publicApi
 */
export declare const APP_BOOTSTRAP_LISTENER: InjectionToken<((compRef: ComponentRef<any>) => void)[]>;

/**
 * A DI Token representing a unique string id assigned to the application by Angular and used
 * primarily for prefixing application attributes and CSS styles when
 * {@link ViewEncapsulation#Emulated ViewEncapsulation.Emulated} is being used.
 *
 * If you need to avoid randomly generated value to be used as an application id, you can provide
 * a custom value via a DI provider <!-- TODO: provider --> configuring the root {@link Injector}
 * using this token.
 * @publicApi
 */
export declare const APP_ID: InjectionToken<string>;

/**
 * An injection token that allows you to provide one or more initialization functions.
 * These function are injected at application startup and executed during
 * app initialization. If any of these functions returns a Promise, initialization
 * does not complete until the Promise is resolved.
 *
 * You can, for example, create a factory function that loads language data
 * or an external configuration, and provide that function to the `APP_INITIALIZER` token.
 * That way, the function is executed during the application bootstrap process,
 * and the needed data is available on startup.
 *
 * @publicApi
 */
export declare const APP_INITIALIZER: InjectionToken<(() => void)[]>;

/**
 * A class that reflects the state of running {@link APP_INITIALIZER}s.
 *
 * @publicApi
 */
export declare class ApplicationInitStatus {
    private appInits;
    private resolve;
    private reject;
    private initialized;
    readonly donePromise: Promise<any>;
    readonly done = false;
    constructor(appInits: (() => any)[]);
    static ɵfac: ɵngcc0.ɵɵFactoryDef<ApplicationInitStatus>;
    static ɵprov: ɵngcc0.ɵɵInjectableDef<ApplicationInitStatus>;
}

/**
 * Configures the root injector for an app with
 * providers of `@angular/core` dependencies that `ApplicationRef` needs
 * to bootstrap components.
 *
 * Re-exported by `BrowserModule`, which is included automatically in the root
 * `AppModule` when you create a new app with the CLI `new` command.
 *
 * @publicApi
 */
export declare class ApplicationModule {
    constructor(appRef: ApplicationRef);
    static ɵmod: ɵngcc0.ɵɵNgModuleDefWithMeta<ApplicationModule, never, never, never>;
    static ɵinj: ɵngcc0.ɵɵInjectorDef<ApplicationModule>;
}

/**
 * A reference to an Angular application running on a page.
 *
 * @usageNotes
 *
 * {@a is-stable-examples}
 * ### isStable examples and caveats
 *
 * Note two important points about `isStable`, demonstrated in the examples below:
 * - the application will never be stable if you start any kind
 * of recurrent asynchronous task when the application starts
 * (for example for a polling process, started with a `setInterval`, a `setTimeout`
 * or using RxJS operators like `interval`);
 * - the `isStable` Observable runs outside of the Angular zone.
 *
 * Let's imagine that you start a recurrent task
 * (here incrementing a counter, using RxJS `interval`),
 * and at the same time subscribe to `isStable`.
 *
 * ```
 * constructor(appRef: ApplicationRef) {
 *   appRef.isStable.pipe(
 *      filter(stable => stable)
 *   ).subscribe(() => console.log('App is stable now');
 *   interval(1000).subscribe(counter => console.log(counter));
 * }
 * ```
 * In this example, `isStable` will never emit `true`,
 * and the trace "App is stable now" will never get logged.
 *
 * If you want to execute something when the app is stable,
 * you have to wait for the application to be stable
 * before starting your polling process.
 *
 * ```
 * constructor(appRef: ApplicationRef) {
 *   appRef.isStable.pipe(
 *     first(stable => stable),
 *     tap(stable => console.log('App is stable now')),
 *     switchMap(() => interval(1000))
 *   ).subscribe(counter => console.log(counter));
 * }
 * ```
 * In this example, the trace "App is stable now" will be logged
 * and then the counter starts incrementing every second.
 *
 * Note also that this Observable runs outside of the Angular zone,
 * which means that the code in the subscription
 * to this Observable will not trigger the change detection.
 *
 * Let's imagine that instead of logging the counter value,
 * you update a field of your component
 * and display it in its template.
 *
 * ```
 * constructor(appRef: ApplicationRef) {
 *   appRef.isStable.pipe(
 *     first(stable => stable),
 *     switchMap(() => interval(1000))
 *   ).subscribe(counter => this.value = counter);
 * }
 * ```
 * As the `isStable` Observable runs outside the zone,
 * the `value` field will be updated properly,
 * but the template will not be refreshed!
 *
 * You'll have to manually trigger the change detection to update the template.
 *
 * ```
 * constructor(appRef: ApplicationRef, cd: ChangeDetectorRef) {
 *   appRef.isStable.pipe(
 *     first(stable => stable),
 *     switchMap(() => interval(1000))
 *   ).subscribe(counter => {
 *     this.value = counter;
 *     cd.detectChanges();
 *   });
 * }
 * ```
 *
 * Or make the subscription callback run inside the zone.
 *
 * ```
 * constructor(appRef: ApplicationRef, zone: NgZone) {
 *   appRef.isStable.pipe(
 *     first(stable => stable),
 *     switchMap(() => interval(1000))
 *   ).subscribe(counter => zone.run(() => this.value = counter));
 * }
 * ```
 *
 * @publicApi
 */
export declare class ApplicationRef {
    private _zone;
    private _console;
    private _injector;
    private _exceptionHandler;
    private _componentFactoryResolver;
    private _initStatus;
    private _views;
    private _runningTick;
    private _enforceNoNewChanges;
    private _stable;
    /**
     * Get a list of component types registered to this application.
     * This list is populated even before the component is created.
     */
    readonly componentTypes: Type<any>[];
    /**
     * Get a list of components registered to this application.
     */
    readonly components: ComponentRef<any>[];
    /**
     * Returns an Observable that indicates when the application is stable or unstable.
     *
     * @see  [Usage notes](#is-stable-examples) for examples and caveats when using this API.
     */
    readonly isStable: Observable<boolean>;
    /**
     * Bootstrap a new component at the root level of the application.
     *
     * @usageNotes
     * ### Bootstrap process
     *
     * When bootstrapping a new root component into an application, Angular mounts the
     * specified application component onto DOM elements identified by the componentType's
     * selector and kicks off automatic change detection to finish initializing the component.
     *
     * Optionally, a component can be mounted onto a DOM element that does not match the
     * componentType's selector.
     *
     * ### Example
     * {@example core/ts/platform/platform.ts region='longform'}
     */
    bootstrap<C>(componentOrFactory: ComponentFactory<C> | Type<C>, rootSelectorOrNode?: string | any): ComponentRef<C>;
    /**
     * Invoke this method to explicitly process change detection and its side-effects.
     *
     * In development mode, `tick()` also performs a second change detection cycle to ensure that no
     * further changes are detected. If additional changes are picked up during this second cycle,
     * bindings in the app have side-effects that cannot be resolved in a single change detection
     * pass.
     * In this case, Angular throws an error, since an Angular application can only have one change
     * detection pass during which all change detection must complete.
     */
    tick(): void;
    /**
     * Attaches a view so that it will be dirty checked.
     * The view will be automatically detached when it is destroyed.
     * This will throw if the view is already attached to a ViewContainer.
     */
    attachView(viewRef: ViewRef): void;
    /**
     * Detaches a view from dirty checking again.
     */
    detachView(viewRef: ViewRef): void;
    private _loadComponent;
    private _unloadComponent;
    /**
     * Returns the number of attached views.
     */
    get viewCount(): number;
    static ɵfac: ɵngcc0.ɵɵFactoryDef<ApplicationRef>;
    static ɵprov: ɵngcc0.ɵɵInjectableDef<ApplicationRef>;
}

/**
 * @publicApi
 */
export declare function asNativeElements(debugEls: DebugElement[]): any;

/**
 * Checks that there currently is a platform which contains the given token as a provider.
 *
 * @publicApi
 */
export declare function assertPlatform(requiredToken: any): PlatformRef;

/**
 * Type of the Attribute metadata.
 *
 * @publicApi
 */
export declare interface Attribute {
    /**
     * The name of the attribute whose value can be injected.
     */
    attributeName?: string;
}

/**
 * Attribute decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const Attribute: AttributeDecorator;

/**
 * Type of the Attribute decorator / constructor function.
 *
 * @publicApi
 */
export declare interface AttributeDecorator {
    /**
     * Parameter decorator for a directive constructor that designates
     * a host-element attribute whose value is injected as a constant string literal.
     *
     * @usageNotes
     *
     * Suppose we have an `<input>` element and want to know its `type`.
     *
     * ```html
     * <input type="text">
     * ```
     *
     * The following example uses the decorator to inject the string literal `text`.
     *
     * {@example core/ts/metadata/metadata.ts region='attributeMetadata'}
     *
     * ### Example as TypeScript Decorator
     *
     * {@example core/ts/metadata/metadata.ts region='attributeFactory'}
     *
     */
    (name: string): any;
    new (name: string): Attribute;
}

declare interface BindingDef {
    flags: ɵBindingFlags;
    ns: string | null;
    name: string | null;
    nonMinifiedName: string | null;
    securityContext: SecurityContext | null;
    suffix: string | null;
}

/**
 * Provides additional options to the bootstraping process.
 *
 *
 */
declare interface BootstrapOptions {
    /**
     * Optionally specify which `NgZone` should be used.
     *
     * - Provide your own `NgZone` instance.
     * - `zone.js` - Use default `NgZone` which requires `Zone.js`.
     * - `noop` - Use `NoopNgZone` which does nothing.
     */
    ngZone?: NgZone | 'zone.js' | 'noop';
    /**
     * Optionally specify coalescing event change detections or not.
     * Consider the following case.
     *
     * <div (click)="doSomething()">
     *   <button (click)="doSomethingElse()"></button>
     * </div>
     *
     * When button is clicked, because of the event bubbling, both
     * event handlers will be called and 2 change detections will be
     * triggered. We can colesce such kind of events to only trigger
     * change detection only once.
     *
     * By default, this option will be false. So the events will not be
     * coalesced and the change detection will be triggered multiple times.
     * And if this option be set to true, the change detection will be
     * triggered async by scheduling a animation frame. So in the case above,
     * the change detection will only be trigged once.
     */
    ngZoneEventCoalescing?: boolean;
}


/**
 * The strategy that the default change detector uses to detect changes.
 * When set, takes effect the next time change detection is triggered.
 *
 * @publicApi
 */
export declare enum ChangeDetectionStrategy {
    /**
     * Use the `CheckOnce` strategy, meaning that automatic change detection is deactivated
     * until reactivated by setting the strategy to `Default` (`CheckAlways`).
     * Change detection can still be explicitly invoked.
     * This strategy applies to all child directives and cannot be overridden.
     */
    OnPush = 0,
    /**
     * Use the default `CheckAlways` strategy, in which change detection is automatic until
     * explicitly deactivated.
     */
    Default = 1
}

/**
 * Base class for Angular Views, provides change detection functionality.
 * A change-detection tree collects all views that are to be checked for changes.
 * Use the methods to add and remove views from the tree, initiate change-detection,
 * and explicitly mark views as _dirty_, meaning that they have changed and need to be rerendered.
 *
 * @usageNotes
 *
 * The following examples demonstrate how to modify default change-detection behavior
 * to perform explicit detection when needed.
 *
 * ### Use `markForCheck()` with `CheckOnce` strategy
 *
 * The following example sets the `OnPush` change-detection strategy for a component
 * (`CheckOnce`, rather than the default `CheckAlways`), then forces a second check
 * after an interval. See [live demo](http://plnkr.co/edit/GC512b?p=preview).
 *
 * <code-example path="core/ts/change_detect/change-detection.ts"
 * region="mark-for-check"></code-example>
 *
 * ### Detach change detector to limit how often check occurs
 *
 * The following example defines a component with a large list of read-only data
 * that is expected to change constantly, many times per second.
 * To improve performance, we want to check and update the list
 * less often than the changes actually occur. To do that, we detach
 * the component's change detector and perform an explicit local check every five seconds.
 *
 * <code-example path="core/ts/change_detect/change-detection.ts" region="detach"></code-example>
 *
 *
 * ### Reattaching a detached component
 *
 * The following example creates a component displaying live data.
 * The component detaches its change detector from the main change detector tree
 * when the `live` property is set to false, and reattaches it when the property
 * becomes true.
 *
 * <code-example path="core/ts/change_detect/change-detection.ts" region="reattach"></code-example>
 *
 * @publicApi
 */
export declare abstract class ChangeDetectorRef {
    /**
     * When a view uses the {@link ChangeDetectionStrategy#OnPush OnPush} (checkOnce)
     * change detection strategy, explicitly marks the view as changed so that
     * it can be checked again.
     *
     * Components are normally marked as dirty (in need of rerendering) when inputs
     * have changed or events have fired in the view. Call this method to ensure that
     * a component is checked even if these triggers have not occured.
     *
     * <!-- TODO: Add a link to a chapter on OnPush components -->
     *
     */
    abstract markForCheck(): void;
    /**
     * Detaches this view from the change-detection tree.
     * A detached view is  not checked until it is reattached.
     * Use in combination with `detectChanges()` to implement local change detection checks.
     *
     * Detached views are not checked during change detection runs until they are
     * re-attached, even if they are marked as dirty.
     *
     * <!-- TODO: Add a link to a chapter on detach/reattach/local digest -->
     * <!-- TODO: Add a live demo once ref.detectChanges is merged into master -->
     *
     */
    abstract detach(): void;
    /**
     * Checks this view and its children. Use in combination with {@link ChangeDetectorRef#detach
     * detach}
     * to implement local change detection checks.
     *
     * <!-- TODO: Add a link to a chapter on detach/reattach/local digest -->
     * <!-- TODO: Add a live demo once ref.detectChanges is merged into master -->
     *
     */
    abstract detectChanges(): void;
    /**
     * Checks the change detector and its children, and throws if any changes are detected.
     *
     * Use in development mode to verify that running change detection doesn't introduce
     * other changes.
     */
    abstract checkNoChanges(): void;
    /**
     * Re-attaches the previously detached view to the change detection tree.
     * Views are attached to the tree by default.
     *
     * <!-- TODO: Add a link to a chapter on detach/reattach/local digest -->
     *
     */
    abstract reattach(): void;
}

declare const CHILD_HEAD = 13;

declare const CHILD_TAIL = 14;

/**
 * Configures the `Injector` to return an instance of `useClass` for a token.
 * @see ["Dependency Injection Guide"](guide/dependency-injection).
 *
 * @usageNotes
 *
 * {@example core/di/ts/provider_spec.ts region='ClassProvider'}
 *
 * Note that following two providers are not equal:
 *
 * {@example core/di/ts/provider_spec.ts region='ClassProviderDifference'}
 *
 * ### Multi-value example
 *
 * {@example core/di/ts/provider_spec.ts region='MultiProviderAspect'}
 *
 * @publicApi
 */
export declare interface ClassProvider extends ClassSansProvider {
    /**
     * An injection token. (Typically an instance of `Type` or `InjectionToken`, but can be `any`).
     */
    provide: any;
    /**
     * When true, injector returns an array of instances. This is useful to allow multiple
     * providers spread across many files to provide configuration information to a common token.
     */
    multi?: boolean;
}

/**
 * Configures the `Injector` to return a value by invoking a `useClass` function.
 * Base for `ClassProvider` decorator.
 *
 * @see ["Dependency Injection Guide"](guide/dependency-injection).
 *
 * @publicApi
 */
export declare interface ClassSansProvider {
    /**
     * Class to instantiate for the `token`.
     */
    useClass: Type<any>;
}

declare const CLEANUP = 7;

/**
 * @deprecated v4.0.0 - Use IterableChangeRecord instead.
 * @publicApi
 */
export declare interface CollectionChangeRecord<V> extends IterableChangeRecord<V> {
}

/**
 * Marks that the next string is for comment.
 *
 * See `I18nMutateOpCodes` documentation.
 */
declare const COMMENT_MARKER: COMMENT_MARKER;

declare interface COMMENT_MARKER {
    marker: 'comment';
}

/**
 * Compile an Angular injectable according to its `Injectable` metadata, and patch the resulting
 * injectable def (`ɵprov`) onto the injectable type.
 */
declare function compileInjectable(type: Type<any>, srcMeta?: Injectable): void;

/**
 * Low-level service for running the angular compiler during runtime
 * to create {@link ComponentFactory}s, which
 * can later be used to create and render a Component instance.
 *
 * Each `@NgModule` provides an own `Compiler` to its injector,
 * that will use the directives/pipes of the ng module for compilation
 * of components.
 *
 * @publicApi
 */
export declare class Compiler {
    /**
     * Compiles the given NgModule and all of its components. All templates of the components listed
     * in `entryComponents` have to be inlined.
     */
    compileModuleSync: <T>(moduleType: Type<T>) => NgModuleFactory<T>;
    /**
     * Compiles the given NgModule and all of its components
     */
    compileModuleAsync: <T>(moduleType: Type<T>) => Promise<NgModuleFactory<T>>;
    /**
     * Same as {@link #compileModuleSync} but also creates ComponentFactories for all components.
     */
    compileModuleAndAllComponentsSync: <T>(moduleType: Type<T>) => ModuleWithComponentFactories<T>;
    /**
     * Same as {@link #compileModuleAsync} but also creates ComponentFactories for all components.
     */
    compileModuleAndAllComponentsAsync: <T>(moduleType: Type<T>) => Promise<ModuleWithComponentFactories<T>>;
    /**
     * Clears all caches.
     */
    clearCache(): void;
    /**
     * Clears the cache for the given component/ngModule.
     */
    clearCacheFor(type: Type<any>): void;
    /**
     * Returns the id for a given NgModule, if one is defined and known to the compiler.
     */
    getModuleId(moduleType: Type<any>): string | undefined;
    static ɵfac: ɵngcc0.ɵɵFactoryDef<Compiler>;
    static ɵprov: ɵngcc0.ɵɵInjectableDef<Compiler>;
}

/**
 * Token to provide CompilerOptions in the platform injector.
 *
 * @publicApi
 */
export declare const COMPILER_OPTIONS: InjectionToken<CompilerOptions[]>;

/**
 * A factory for creating a Compiler
 *
 * @publicApi
 */
export declare abstract class CompilerFactory {
    abstract createCompiler(options?: CompilerOptions[]): Compiler;
}

/**
 * Options for creating a compiler
 *
 * @publicApi
 */
export declare type CompilerOptions = {
    useJit?: boolean;
    defaultEncapsulation?: ViewEncapsulation;
    providers?: StaticProvider[];
    missingTranslation?: MissingTranslationStrategy;
    preserveWhitespaces?: boolean;
};

/**
 * Supplies configuration metadata for an Angular component.
 *
 * @publicApi
 */
export declare interface Component extends Directive {
    /**
     * The change-detection strategy to use for this component.
     *
     * When a component is instantiated, Angular creates a change detector,
     * which is responsible for propagating the component's bindings.
     * The strategy is one of:
     * - `ChangeDetectionStrategy#OnPush` sets the strategy to `CheckOnce` (on demand).
     * - `ChangeDetectionStrategy#Default` sets the strategy to `CheckAlways`.
     */
    changeDetection?: ChangeDetectionStrategy;
    /**
     * Defines the set of injectable objects that are visible to its view DOM children.
     * See [example](#injecting-a-class-with-a-view-provider).
     *
     */
    viewProviders?: Provider[];
    /**
     * The module ID of the module that contains the component.
     * The component must be able to resolve relative URLs for templates and styles.
     * SystemJS exposes the `__moduleName` variable within each module.
     * In CommonJS, this can  be set to `module.id`.
     *
     */
    moduleId?: string;
    /**
     * The relative path or absolute URL of a template file for an Angular component.
     * If provided, do not supply an inline template using `template`.
     *
     */
    templateUrl?: string;
    /**
     * An inline template for an Angular component. If provided,
     * do not supply a template file using `templateUrl`.
     *
     */
    template?: string;
    /**
     * One or more relative paths or absolute URLs for files containing CSS stylesheets to use
     * in this component.
     */
    styleUrls?: string[];
    /**
     * One or more inline CSS stylesheets to use
     * in this component.
     */
    styles?: string[];
    /**
     * One or more animation `trigger()` calls, containing
     * `state()` and `transition()` definitions.
     * See the [Animations guide](/guide/animations) and animations API documentation.
     *
     */
    animations?: any[];
    /**
     * An encapsulation policy for the template and CSS styles. One of:
     * - `ViewEncapsulation.Native`: Deprecated. Use `ViewEncapsulation.ShadowDom` instead.
     * - `ViewEncapsulation.Emulated`: Use shimmed CSS that
     * emulates the native behavior.
     * - `ViewEncapsulation.None`: Use global CSS without any
     * encapsulation.
     * - `ViewEncapsulation.ShadowDom`: Use Shadow DOM v1 to encapsulate styles.
     *
     * If not supplied, the value is taken from `CompilerOptions`. The default compiler option is
     * `ViewEncapsulation.Emulated`.
     *
     * If the policy is set to `ViewEncapsulation.Emulated` and the component has no `styles`
     * or `styleUrls` specified, the policy is automatically switched to `ViewEncapsulation.None`.
     */
    encapsulation?: ViewEncapsulation;
    /**
     * Overrides the default encapsulation start and end delimiters (`{{` and `}}`)
     */
    interpolation?: [string, string];
    /**
     * A set of components that should be compiled along with
     * this component. For each component listed here,
     * Angular creates a {@link ComponentFactory} and stores it in the
     * {@link ComponentFactoryResolver}.
     * @deprecated Since 9.0.0. With Ivy, this property is no longer necessary.
     */
    entryComponents?: Array<Type<any> | any[]>;
    /**
     * True to preserve or false to remove potentially superfluous whitespace characters
     * from the compiled template. Whitespace characters are those matching the `\s`
     * character class in JavaScript regular expressions. Default is false, unless
     * overridden in compiler options.
     */
    preserveWhitespaces?: boolean;
}

/**
 * Component decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const Component: ComponentDecorator;

/**
 * Component decorator interface
 *
 * @publicApi
 */
export declare interface ComponentDecorator {
    /**
     * Decorator that marks a class as an Angular component and provides configuration
     * metadata that determines how the component should be processed,
     * instantiated, and used at runtime.
     *
     * Components are the most basic UI building block of an Angular app.
     * An Angular app contains a tree of Angular components.
     *
     * Angular components are a subset of directives, always associated with a template.
     * Unlike other directives, only one component can be instantiated per an element in a template.
     *
     * A component must belong to an NgModule in order for it to be available
     * to another component or application. To make it a member of an NgModule,
     * list it in the `declarations` field of the `NgModule` metadata.
     *
     * Note that, in addition to these options for configuring a directive,
     * you can control a component's runtime behavior by implementing
     * life-cycle hooks. For more information, see the
     * [Lifecycle Hooks](guide/lifecycle-hooks) guide.
     *
     * @usageNotes
     *
     * ### Setting component inputs
     *
     * The following example creates a component with two data-bound properties,
     * specified by the `inputs` value.
     *
     * <code-example path="core/ts/metadata/directives.ts" region="component-input"></code-example>
     *
     *
     * ### Setting component outputs
     *
     * The following example shows two event emitters that emit on an interval. One
     * emits an output every second, while the other emits every five seconds.
     *
     * {@example core/ts/metadata/directives.ts region='component-output-interval'}
     *
     * ### Injecting a class with a view provider
     *
     * The following simple example injects a class into a component
     * using the view provider specified in component metadata:
     *
     * ```ts
     * class Greeter {
     *    greet(name:string) {
     *      return 'Hello ' + name + '!';
     *    }
     * }
     *
     * @Directive({
     *   selector: 'needs-greeter'
     * })
     * class NeedsGreeter {
     *   greeter:Greeter;
     *
     *   constructor(greeter:Greeter) {
     *     this.greeter = greeter;
     *   }
     * }
     *
     * @Component({
     *   selector: 'greet',
     *   viewProviders: [
     *     Greeter
     *   ],
     *   template: `<needs-greeter></needs-greeter>`
     * })
     * class HelloWorld {
     * }
     *
     * ```
     *
     * ### Preserving whitespace
     *
     * Removing whitespace can greatly reduce AOT-generated code size and speed up view creation.
     * As of Angular 6, the default for `preserveWhitespaces` is false (whitespace is removed).
     * To change the default setting for all components in your application, set
     * the `preserveWhitespaces` option of the AOT compiler.
     *
     * By default, the AOT compiler removes whitespace characters as follows:
     * * Trims all whitespaces at the beginning and the end of a template.
     * * Removes whitespace-only text nodes. For example,
     *
     * ```html
     * <button>Action 1</button>  <button>Action 2</button>
     * ```
     *
     * becomes:
     *
     * ```html
     * <button>Action 1</button><button>Action 2</button>
     * ```
     *
     * * Replaces a series of whitespace characters in text nodes with a single space.
     * For example, `<span>\n some text\n</span>` becomes `<span> some text </span>`.
     * * Does NOT alter text nodes inside HTML tags such as `<pre>` or `<textarea>`,
     * where whitespace characters are significant.
     *
     * Note that these transformations can influence DOM nodes layout, although impact
     * should be minimal.
     *
     * You can override the default behavior to preserve whitespace characters
     * in certain fragments of a template. For example, you can exclude an entire
     * DOM sub-tree by using the `ngPreserveWhitespaces` attribute:
     *
     * ```html
     * <div ngPreserveWhitespaces>
     *     whitespaces are preserved here
     *     <span>    and here </span>
     * </div>
     * ```
     *
     * You can force a single space to be preserved in a text node by using `&ngsp;`,
     * which is replaced with a space character by Angular's template
     * compiler:
     *
     * ```html
     * <a>Spaces</a>&ngsp;<a>between</a>&ngsp;<a>links.</a>
     * <!-->compiled to be equivalent to:</>
     *  <a>Spaces</a> <a>between</a> <a>links.</a>
     * ```
     *
     * Note that sequences of `&ngsp;` are still collapsed to just one space character when
     * the `preserveWhitespaces` option is set to `false`.
     *
     * ```html
     * <a>before</a>&ngsp;&ngsp;&ngsp;<a>after</a>
     * <!-->compiled to be equivalent to:</>
     *  <a>Spaces</a> <a>between</a> <a>links.</a>
     * ```
     *
     * To preserve sequences of whitespace characters, use the
     * `ngPreserveWhitespaces` attribute.
     *
     * @Annotation
     */
    (obj: Component): TypeDecorator;
    /**
     * See the `Component` decorator.
     */
    new (obj: Component): Component;
}

declare interface ComponentDefFeature {
    <T>(componentDef: ɵComponentDef<T>): void;
    /**
     * Marks a feature as something that {@link InheritDefinitionFeature} will execute
     * during inheritance.
     *
     * NOTE: DO NOT SET IN ROOT OF MODULE! Doing so will result in tree-shakers/bundlers
     * identifying the change as a side effect, and the feature will be included in
     * every bundle.
     */
    ngInherit?: true;
}

/**
 * Base class for a factory that can create a component dynamically.
 * Instantiate a factory for a given type of component with `resolveComponentFactory()`.
 * Use the resulting `ComponentFactory.create()` method to create a component of that type.
 *
 * @see [Dynamic Components](guide/dynamic-component-loader)
 *
 * @publicApi
 */
declare abstract class ComponentFactory<C> {
    /**
     * The component's HTML selector.
     */
    abstract get selector(): string;
    /**
     * The type of component the factory will create.
     */
    abstract get componentType(): Type<any>;
    /**
     * Selector for all <ng-content> elements in the component.
     */
    abstract get ngContentSelectors(): string[];
    /**
     * The inputs of the component.
     */
    abstract get inputs(): {
        propName: string;
        templateName: string;
    }[];
    /**
     * The outputs of the component.
     */
    abstract get outputs(): {
        propName: string;
        templateName: string;
    }[];
    /**
     * Creates a new component.
     */
    abstract create(injector: Injector, projectableNodes?: any[][], rootSelectorOrNode?: string | any, ngModule?: NgModuleRef<any>): ComponentRef<C>;
}
export { ComponentFactory }
export { ComponentFactory as ɵComponentFactory }

/**
 * A simple registry that maps `Components` to generated `ComponentFactory` classes
 * that can be used to create instances of components.
 * Use to obtain the factory for a given component type,
 * then use the factory's `create()` method to create a component of that type.
 *
 * @see [Dynamic Components](guide/dynamic-component-loader)
 * @publicApi
 */
export declare abstract class ComponentFactoryResolver {
    static NULL: ComponentFactoryResolver;
    /**
     * Retrieves the factory object that creates a component of the given type.
     * @param component The component type.
     */
    abstract resolveComponentFactory<T>(component: Type<T>): ComponentFactory<T>;
}

declare class ComponentFactoryResolver_2 extends ComponentFactoryResolver {
    private ngModule?;
    /**
     * @param ngModule The NgModuleRef to which all resolved factories are bound.
     */
    constructor(ngModule?: NgModuleRef<any> | undefined);
    resolveComponentFactory<T>(component: Type<T>): ComponentFactory<T>;
}

declare type ComponentInstance = {};

/**
 * Represents a component created by a `ComponentFactory`.
 * Provides access to the component instance and related objects,
 * and provides the means of destroying the instance.
 *
 * @publicApi
 */
export declare abstract class ComponentRef<C> {
    /**
     * The host or anchor [element](guide/glossary#element) for this component instance.
     */
    abstract get location(): ElementRef;
    /**
     * The [dependency injector](guide/glossary#injector) for this component instance.
     */
    abstract get injector(): Injector;
    /**
     * This component instance.
     */
    abstract get instance(): C;
    /**
     * The [host view](guide/glossary#view-tree) defined by the template
     * for this component instance.
     */
    abstract get hostView(): ViewRef;
    /**
     * The change detector for this component instance.
     */
    abstract get changeDetectorRef(): ChangeDetectorRef;
    /**
     * The type of this component (as created by a `ComponentFactory` class).
     */
    abstract get componentType(): Type<any>;
    /**
     * Destroys the component instance and all of the data structures associated with it.
     */
    abstract destroy(): void;
    /**
     * A lifecycle hook that provides additional developer-defined cleanup
     * functionality for the component.
     * @param callback A handler function that cleans up developer-defined data
     * associated with this component. Called when the `destroy()` method is invoked.
     */
    abstract onDestroy(callback: Function): void;
}

/**
 * Definition of what a template rendering function should look like for a component.
 */
declare type ComponentTemplate<T> = {
    <U extends T>(rf: ɵRenderFlags, ctx: T | U): void;
};

/**
 * Configures the `Injector` to return an instance of a token.
 *
 * @see ["Dependency Injection Guide"](guide/dependency-injection).
 *
 * @usageNotes
 *
 * {@example core/di/ts/provider_spec.ts region='ConstructorProvider'}
 *
 * ### Multi-value example
 *
 * {@example core/di/ts/provider_spec.ts region='MultiProviderAspect'}
 *
 * @publicApi
 */
export declare interface ConstructorProvider extends ConstructorSansProvider {
    /**
     * An injection token. Typically an instance of `Type` or `InjectionToken`, but can be `any`.
     */
    provide: Type<any>;
    /**
     * When true, injector returns an array of instances. This is useful to allow multiple
     * providers spread across many files to provide configuration information to a common token.
     */
    multi?: boolean;
}

/**
 * Configures the `Injector` to return an instance of a token.
 *
 * @see ["Dependency Injection Guide"](guide/dependency-injection).
 *
 * @usageNotes
 *
 * ```ts
 * @Injectable(SomeModule, {deps: []})
 * class MyService {}
 * ```
 *
 * @publicApi
 */
export declare interface ConstructorSansProvider {
    /**
     * A list of `token`s to be resolved by the injector.
     */
    deps?: any[];
}

/**
 * Type of the ContentChild metadata.
 *
 * @publicApi
 */
export declare type ContentChild = Query;

/**
 * ContentChild decorator and metadata.
 *
 *
 * @Annotation
 *
 * @publicApi
 */
export declare const ContentChild: ContentChildDecorator;

/**
 * Type of the ContentChild decorator / constructor function.
 *
 * @publicApi
 */
export declare interface ContentChildDecorator {
    /**
     * Parameter decorator that configures a content query.
     *
     * Use to get the first element or the directive matching the selector from the content DOM.
     * If the content DOM changes, and a new child matches the selector,
     * the property will be updated.
     *
     * Content queries are set before the `ngAfterContentInit` callback is called.
     *
     * Does not retrieve elements or directives that are in other components' templates,
     * since a component's template is always a black box to its ancestors.
     *
     * **Metadata Properties**:
     *
     * * **selector** - The directive type or the name used for querying.
     * * **read** - True to read a different token from the queried element.
     * * **static** - True to resolve query results before change detection runs,
     * false to resolve after change detection. Defaults to false.
     *
     * @usageNotes
     *
     * {@example core/di/ts/contentChild/content_child_howto.ts region='HowTo'}
     *
     * ### Example
     *
     * {@example core/di/ts/contentChild/content_child_example.ts region='Component'}
     *
     * @Annotation
     */
    (selector: Type<any> | Function | string, opts?: {
        read?: any;
        static?: boolean;
    }): any;
    new (selector: Type<any> | Function | string, opts?: {
        read?: any;
        static?: boolean;
    }): ContentChild;
}

/**
 * Type of the ContentChildren metadata.
 *
 *
 * @Annotation
 * @publicApi
 */
export declare type ContentChildren = Query;

/**
 * ContentChildren decorator and metadata.
 *
 *
 * @Annotation
 * @publicApi
 */
export declare const ContentChildren: ContentChildrenDecorator;

/**
 * Type of the ContentChildren decorator / constructor function.
 *
 * @see `ContentChildren`.
 * @publicApi
 */
export declare interface ContentChildrenDecorator {
    /**
     * Parameter decorator that configures a content query.
     *
     * Use to get the `QueryList` of elements or directives from the content DOM.
     * Any time a child element is added, removed, or moved, the query list will be
     * updated, and the changes observable of the query list will emit a new value.
     *
     * Content queries are set before the `ngAfterContentInit` callback is called.
     *
     * Does not retrieve elements or directives that are in other components' templates,
     * since a component's template is always a black box to its ancestors.
     *
     * **Metadata Properties**:
     *
     * * **selector** - The directive type or the name used for querying.
     * * **descendants** - True to include all descendants, otherwise include only direct children.
     * * **read** - True to read a different token from the queried elements.
     *
     * @usageNotes
     *
     * Here is a simple demonstration of how the `ContentChildren` decorator can be used.
     *
     * {@example core/di/ts/contentChildren/content_children_howto.ts region='HowTo'}
     *
     * ### Tab-pane example
     *
     * Here is a slightly more realistic example that shows how `ContentChildren` decorators
     * can be used to implement a tab pane component.
     *
     * {@example core/di/ts/contentChildren/content_children_example.ts region='Component'}
     *
     * @Annotation
     */
    (selector: Type<any> | Function | string, opts?: {
        descendants?: boolean;
        read?: any;
    }): any;
    new (selector: Type<any> | Function | string, opts?: {
        descendants?: boolean;
        read?: any;
    }): Query;
}

/**
 * Definition of what a content queries function should look like.
 */
declare type ContentQueriesFunction<T> = <U extends T>(rf: ɵRenderFlags, ctx: U, directiveIndex: number) => void;

declare const CONTEXT = 8;

/** Options that control how the component should be bootstrapped. */
declare interface CreateComponentOptions {
    /** Which renderer factory to use. */
    rendererFactory?: RendererFactory3;
    /** A custom sanitizer instance */
    sanitizer?: Sanitizer;
    /** A custom animation player handler */
    playerHandler?: ɵPlayerHandler;
    /**
     * Host element on which the component will be bootstrapped. If not specified,
     * the component definition's `tag` is used to query the existing DOM for the
     * element to bootstrap.
     */
    host?: RElement | string;
    /** Module injector for the component. If unspecified, the injector will be NULL_INJECTOR. */
    injector?: Injector;
    /**
     * List of features to be applied to the created component. Features are simply
     * functions that decorate a component with a certain behavior.
     *
     * Typically, the features in this list are features that cannot be added to the
     * other features list in the component definition because they rely on other factors.
     *
     * Example: `LifecycleHooksFeature` is a function that adds lifecycle hook capabilities
     * to root components in a tree-shakable way. It cannot be added to the component
     * features list because there's no way of knowing when the component will be used as
     * a root component.
     */
    hostFeatures?: HostFeature[];
    /**
     * A function which is used to schedule change detection work in the future.
     *
     * When marking components as dirty, it is necessary to schedule the work of
     * change detection in the future. This is done to coalesce multiple
     * {@link markDirty} calls into a single changed detection processing.
     *
     * The default value of the scheduler is the `requestAnimationFrame` function.
     *
     * It is also useful to override this function for testing purposes.
     */
    scheduler?: (work: () => void) => void;
}

/**
 * Creates a platform.
 * Platforms have to be eagerly created via this function.
 *
 * @publicApi
 */
export declare function createPlatform(injector: Injector): PlatformRef;

/**
 * Creates a factory for a platform
 *
 * @publicApi
 */
export declare function createPlatformFactory(parentPlatformFactory: ((extraProviders?: StaticProvider[]) => PlatformRef) | null, name: string, providers?: StaticProvider[]): (extraProviders?: StaticProvider[]) => PlatformRef;


/**
 * Expresses a single CSS Selector.
 *
 * Beginning of array
 * - First index: element name
 * - Subsequent odd indices: attr keys
 * - Subsequent even indices: attr values
 *
 * After SelectorFlags.CLASS flag
 * - Class name values
 *
 * SelectorFlags.NOT flag
 * - Changes the mode to NOT
 * - Can be combined with other flags to set the element / attr / class mode
 *
 * e.g. SelectorFlags.NOT | SelectorFlags.ELEMENT
 *
 * Example:
 * Original: `div.foo.bar[attr1=val1][attr2]`
 * Parsed: ['div', 'attr1', 'val1', 'attr2', '', SelectorFlags.CLASS, 'foo', 'bar']
 *
 * Original: 'div[attr1]:not(.foo[attr2])
 * Parsed: [
 *  'div', 'attr1', '',
 *  SelectorFlags.NOT | SelectorFlags.ATTRIBUTE 'attr2', '', SelectorFlags.CLASS, 'foo'
 * ]
 *
 * See more examples in node_selector_matcher_spec.ts
 */
declare type CssSelector = (string | SelectorFlags)[];

/**
 * Defines a schema that allows an NgModule to contain the following:
 * - Non-Angular elements named with dash case (`-`).
 * - Element properties named with dash case (`-`).
 * Dash case is the naming convention for custom elements.
 *
 * @publicApi
 */
export declare const CUSTOM_ELEMENTS_SCHEMA: SchemaMetadata;

/**
 * @publicApi
 */
export declare interface DebugElement extends DebugNode {
    readonly name: string;
    readonly properties: {
        [key: string]: any;
    };
    readonly attributes: {
        [key: string]: string | null;
    };
    readonly classes: {
        [key: string]: boolean;
    };
    readonly styles: {
        [key: string]: string | null;
    };
    readonly childNodes: DebugNode[];
    readonly nativeElement: any;
    readonly children: DebugElement[];
    query(predicate: Predicate<DebugElement>): DebugElement;
    queryAll(predicate: Predicate<DebugElement>): DebugElement[];
    queryAllNodes(predicate: Predicate<DebugNode>): DebugNode[];
    triggerEventHandler(eventName: string, eventObj: any): void;
}

/**
 * @publicApi
 */
export declare const DebugElement: {
    new (...args: any[]): DebugElement;
};

declare class DebugElement__POST_R3__ extends DebugNode__POST_R3__ implements DebugElement {
    constructor(nativeNode: Element);
    get nativeElement(): Element | null;
    get name(): string;
    /**
     *  Gets a map of property names to property values for an element.
     *
     *  This map includes:
     *  - Regular property bindings (e.g. `[id]="id"`)
     *  - Host property bindings (e.g. `host: { '[id]': "id" }`)
     *  - Interpolated property bindings (e.g. `id="{{ value }}")
     *
     *  It does not include:
     *  - input property bindings (e.g. `[myCustomInput]="value"`)
     *  - attribute bindings (e.g. `[attr.role]="menu"`)
     */
    get properties(): {
        [key: string]: any;
    };
    get attributes(): {
        [key: string]: string | null;
    };
    get styles(): {
        [key: string]: string | null;
    };
    get classes(): {
        [key: string]: boolean;
    };
    get childNodes(): DebugNode[];
    get children(): DebugElement[];
    query(predicate: Predicate<DebugElement>): DebugElement;
    queryAll(predicate: Predicate<DebugElement>): DebugElement[];
    queryAllNodes(predicate: Predicate<DebugNode>): DebugNode[];
    triggerEventHandler(eventName: string, eventObj: any): void;
}

/**
 * @publicApi
 */
export declare class DebugEventListener {
    name: string;
    callback: Function;
    constructor(name: string, callback: Function);
}

/**
 * @publicApi
 */
export declare interface DebugNode {
    readonly listeners: DebugEventListener[];
    readonly parent: DebugElement | null;
    readonly nativeNode: any;
    readonly injector: Injector;
    readonly componentInstance: any;
    readonly context: any;
    readonly references: {
        [key: string]: any;
    };
    readonly providerTokens: any[];
}

/**
 * @publicApi
 */
export declare const DebugNode: {
    new (...args: any[]): DebugNode;
};

declare class DebugNode__POST_R3__ implements DebugNode {
    readonly nativeNode: Node;
    constructor(nativeNode: Node);
    get parent(): DebugElement | null;
    get injector(): Injector;
    get componentInstance(): any;
    get context(): any;
    get listeners(): DebugEventListener[];
    get references(): {
        [key: string]: any;
    };
    get providerTokens(): any[];
}

declare const DECLARATION_COMPONENT_VIEW = 16;

declare const DECLARATION_LCONTAINER = 17;

declare const DECLARATION_VIEW = 15;

/**
 * Provide this token to set the default currency code your application uses for
 * CurrencyPipe when there is no currency code passed into it. This is only used by
 * CurrencyPipe and has no relation to locale currency. Defaults to USD if not configured.
 *
 * See the [i18n guide](guide/i18n#setting-up-locale) for more information.
 *
 * <div class="alert is-helpful">
 *
 * **Deprecation notice:**
 *
 * The default currency code is currently always `USD` but this is deprecated from v9.
 *
 * **In v10 the default currency code will be taken from the current locale.**
 *
 * If you need the previous behavior then set it by creating a `DEFAULT_CURRENCY_CODE` provider in
 * your application `NgModule`:
 *
 * ```ts
 * {provide: DEFAULT_CURRENCY_CODE, useValue: 'USD'}
 * ```
 *
 * </div>
 *
 * @usageNotes
 * ### Example
 *
 * ```typescript
 * import { platformBrowserDynamic } from '@angular/platform-browser-dynamic';
 * import { AppModule } from './app/app.module';
 *
 * platformBrowserDynamic().bootstrapModule(AppModule, {
 *   providers: [{provide: DEFAULT_CURRENCY_CODE, useValue: 'EUR' }]
 * });
 * ```
 *
 * @publicApi
 */
export declare const DEFAULT_CURRENCY_CODE: InjectionToken<string>;

/**
 * @deprecated v4.0.0 - Should not be part of public API.
 * @publicApi
 */
export declare class DefaultIterableDiffer<V> implements IterableDiffer<V>, IterableChanges<V> {
    readonly length: number;
    readonly collection: V[] | Iterable<V> | null;
    private _linkedRecords;
    private _unlinkedRecords;
    private _previousItHead;
    private _itHead;
    private _itTail;
    private _additionsHead;
    private _additionsTail;
    private _movesHead;
    private _movesTail;
    private _removalsHead;
    private _removalsTail;
    private _identityChangesHead;
    private _identityChangesTail;
    private _trackByFn;
    constructor(trackByFn?: TrackByFunction<V>);
    forEachItem(fn: (record: IterableChangeRecord_<V>) => void): void;
    forEachOperation(fn: (item: IterableChangeRecord<V>, previousIndex: number | null, currentIndex: number | null) => void): void;
    forEachPreviousItem(fn: (record: IterableChangeRecord_<V>) => void): void;
    forEachAddedItem(fn: (record: IterableChangeRecord_<V>) => void): void;
    forEachMovedItem(fn: (record: IterableChangeRecord_<V>) => void): void;
    forEachRemovedItem(fn: (record: IterableChangeRecord_<V>) => void): void;
    forEachIdentityChange(fn: (record: IterableChangeRecord_<V>) => void): void;
    diff(collection: NgIterable<V> | null | undefined): DefaultIterableDiffer<V> | null;
    onDestroy(): void;
    check(collection: NgIterable<V>): boolean;
    get isDirty(): boolean;
    private _addToRemovals;
}

/**
 * @deprecated in v8, delete after v10. This API should be used only be generated code, and that
 * code should now use ɵɵdefineInjectable instead.
 * @publicApi
 */
export declare const defineInjectable: typeof ɵɵdefineInjectable;

declare interface Definition<DF extends DefinitionFactory<any>> {
    factory: DF | null;
}

/**
 * Factory for ViewDefinitions/NgModuleDefinitions.
 * We use a function so we can reexeute it in case an error happens and use the given logger
 * function to log the error from the definition of the node, which is shown in all browser
 * logs.
 */
declare interface DefinitionFactory<D extends Definition<any>> {
    (logger: NodeLogger): D;
}

declare interface DepDef {
    flags: ɵDepFlags;
    token: any;
    tokenKey: string;
}

/**
 * Destroy the existing platform.
 *
 * @publicApi
 */
export declare function destroyPlatform(): void;

/**
 * Directive decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare interface Directive {
    /**
     * The CSS selector that identifies this directive in a template
     * and triggers instantiation of the directive.
     *
     * Declare as one of the following:
     *
     * - `element-name`: Select by element name.
     * - `.class`: Select by class name.
     * - `[attribute]`: Select by attribute name.
     * - `[attribute=value]`: Select by attribute name and value.
     * - `:not(sub_selector)`: Select only if the element does not match the `sub_selector`.
     * - `selector1, selector2`: Select if either `selector1` or `selector2` matches.
     *
     * Angular only allows directives to apply on CSS selectors that do not cross
     * element boundaries.
     *
     * For the following template HTML, a directive with an `input[type=text]` selector,
     * would be instantiated only on the `<input type="text">` element.
     *
     * ```html
     * <form>
     *   <input type="text">
     *   <input type="radio">
     * <form>
     * ```
     *
     */
    selector?: string;
    /**
     * Enumerates the set of data-bound input properties for a directive
     *
     * Angular automatically updates input properties during change detection.
     * The `inputs` property defines a set of `directiveProperty` to `bindingProperty`
     * configuration:
     *
     * - `directiveProperty` specifies the component property where the value is written.
     * - `bindingProperty` specifies the DOM property where the value is read from.
     *
     * When `bindingProperty` is not provided, it is assumed to be equal to `directiveProperty`.
     *
     * @usageNotes
     *
     * The following example creates a component with two data-bound properties.
     *
     * ```typescript
     * @Component({
     *   selector: 'bank-account',
     *   inputs: ['bankName', 'id: account-id'],
     *   template: `
     *     Bank Name: {{bankName}}
     *     Account Id: {{id}}
     *   `
     * })
     * class BankAccount {
     *   bankName: string;
     *   id: string;
     * }
     * ```
     *
     */
    inputs?: string[];
    /**
     * Enumerates the set of event-bound output properties.
     *
     * When an output property emits an event, an event handler attached to that event
     * in the template is invoked.
     *
     * The `outputs` property defines a set of `directiveProperty` to `bindingProperty`
     * configuration:
     *
     * - `directiveProperty` specifies the component property that emits events.
     * - `bindingProperty` specifies the DOM property the event handler is attached to.
     *
     * @usageNotes
     *
     * ```typescript
     * @Component({
     *   selector: 'child-dir',
     *   outputs: [ 'bankNameChange' ]
     *   template: `<input (input)="bankNameChange.emit($event.target.value)" />`
     * })
     * class ChildDir {
     *  bankNameChange: EventEmitter<string> = new EventEmitter<string>();
     * }
     *
     * @Component({
     *   selector: 'main',
     *   template: `
     *     {{ bankName }} <child-dir (bankNameChange)="onBankNameChange($event)"></child-dir>
     *   `
     * })
     * class MainComponent {
     *  bankName: string;
     *
     *   onBankNameChange(bankName: string) {
     *     this.bankName = bankName;
     *   }
     * }
     * ```
     *
     */
    outputs?: string[];
    /**
     * Configures the [injector](guide/glossary#injector) of this
     * directive or component with a [token](guide/glossary#di-token)
     * that maps to a [provider](guide/glossary#provider) of a dependency.
     */
    providers?: Provider[];
    /**
     * Defines the name that can be used in the template to assign this directive to a variable.
     *
     * @usageNotes
     *
     * ```ts
     * @Directive({
     *   selector: 'child-dir',
     *   exportAs: 'child'
     * })
     * class ChildDir {
     * }
     *
     * @Component({
     *   selector: 'main',
     *   template: `<child-dir #c="child"></child-dir>`
     * })
     * class MainComponent {
     * }
     * ```
     *
     */
    exportAs?: string;
    /**
     * Configures the queries that will be injected into the directive.
     *
     * Content queries are set before the `ngAfterContentInit` callback is called.
     * View queries are set before the `ngAfterViewInit` callback is called.
     *
     * @usageNotes
     *
     * The following example shows how queries are defined
     * and when their results are available in lifecycle hooks:
     *
     * ```ts
     * @Component({
     *   selector: 'someDir',
     *   queries: {
     *     contentChildren: new ContentChildren(ChildDirective),
     *     viewChildren: new ViewChildren(ChildDirective)
     *   },
     *   template: '<child-directive></child-directive>'
     * })
     * class SomeDir {
     *   contentChildren: QueryList<ChildDirective>,
     *   viewChildren: QueryList<ChildDirective>
     *
     *   ngAfterContentInit() {
     *     // contentChildren is set
     *   }
     *
     *   ngAfterViewInit() {
     *     // viewChildren is set
     *   }
     * }
     * ```
     *
     * @Annotation
     */
    queries?: {
        [key: string]: any;
    };
    /**
     * Maps class properties to host element bindings for properties,
     * attributes, and events, using a set of key-value pairs.
     *
     * Angular automatically checks host property bindings during change detection.
     * If a binding changes, Angular updates the directive's host element.
     *
     * When the key is a property of the host element, the property value is
     * the propagated to the specified DOM property.
     *
     * When the key is a static attribute in the DOM, the attribute value
     * is propagated to the specified property in the host element.
     *
     * For event handling:
     * - The key is the DOM event that the directive listens to.
     * To listen to global events, add the target to the event name.
     * The target can be `window`, `document` or `body`.
     * - The value is the statement to execute when the event occurs. If the
     * statement evaluates to `false`, then `preventDefault` is applied on the DOM
     * event. A handler method can refer to the `$event` local variable.
     *
     */
    host?: {
        [key: string]: string;
    };
    /**
     * If true, this directive/component will be skipped by the AOT compiler and so will always be
     * compiled using JIT.
     *
     * This exists to support future Ivy work and has no effect currently.
     */
    jit?: true;
}

/**
 * Type of the Directive metadata.
 *
 * @publicApi
 */
export declare const Directive: DirectiveDecorator;

/**
 * Type of the Directive decorator / constructor function.
 * @publicApi
 */
export declare interface DirectiveDecorator {
    /**
     * Decorator that marks a class as an Angular directive.
     * You can define your own directives to attach custom behavior to elements in the DOM.
     *
     * The options provide configuration metadata that determines
     * how the directive should be processed, instantiated and used at
     * runtime.
     *
     * Directive classes, like component classes, can implement
     * [life-cycle hooks](guide/lifecycle-hooks) to influence their configuration and behavior.
     *
     *
     * @usageNotes
     * To define a directive, mark the class with the decorator and provide metadata.
     *
     * ```ts
     * import {Directive} from '@angular/core';
     *
     * @Directive({
     *   selector: 'my-directive',
     * })
     * export class MyDirective {
     * ...
     * }
     * ```
     *
     * ### Declaring directives
     *
     * Directives are [declarables](guide/glossary#declarable).
     * They must be declared by an NgModule
     * in order to be usable in an app.
     *
     * A directive must belong to exactly one NgModule. Do not re-declare
     * a directive imported from another module.
     * List the directive class in the `declarations` field of an NgModule.
     *
     * ```ts
     * declarations: [
     *  AppComponent,
     *  MyDirective
     * ],
     * ```
     *
     * @Annotation
     */
    (obj?: Directive): TypeDecorator;
    /**
     * See the `Directive` decorator.
     */
    new (obj?: Directive): Directive;
}

declare interface DirectiveDefFeature {
    <T>(directiveDef: ɵDirectiveDef<T>): void;
    /**
     * Marks a feature as something that {@link InheritDefinitionFeature} will execute
     * during inheritance.
     *
     * NOTE: DO NOT SET IN ROOT OF MODULE! Doing so will result in tree-shakers/bundlers
     * identifying the change as a side effect, and the feature will be included in
     * every bundle.
     */
    ngInherit?: true;
}

declare type DirectiveDefList = (ɵDirectiveDef<any> | ɵComponentDef<any>)[];

/**
 * Type used for directiveDefs on component definition.
 *
 * The function is necessary to be able to support forward declarations.
 */
declare type DirectiveDefListOrFactory = (() => DirectiveDefList) | DirectiveDefList;

declare type DirectiveInstance = {};

declare type DirectiveTypeList = (ɵDirectiveType<any> | ɵComponentType<any> | Type<any>)[];

declare type DirectiveTypesOrFactory = (() => DirectiveTypeList) | DirectiveTypeList;

declare interface DisposableFn {
    (): void;
}

/**
 * @description
 * Hook for manual bootstrapping of the application instead of using bootstrap array in @NgModule
 * annotation.
 *
 * Reference to the current application is provided as a parameter.
 *
 * See ["Bootstrapping"](guide/bootstrapping) and ["Entry components"](guide/entry-components).
 *
 * @usageNotes
 * ```typescript
 * class AppModule implements DoBootstrap {
 *   ngDoBootstrap(appRef: ApplicationRef) {
 *     appRef.bootstrap(AppComponent); // Or some other component
 *   }
 * }
 * ```
 *
 * @publicApi
 */
export declare interface DoBootstrap {
    ngDoBootstrap(appRef: ApplicationRef): void;
}

/**
 * A lifecycle hook that invokes a custom change-detection function for a directive,
 * in addition to the check performed by the default change-detector.
 *
 * The default change-detection algorithm looks for differences by comparing
 * bound-property values by reference across change detection runs. You can use this
 * hook to check for and respond to changes by some other means.
 *
 * When the default change detector detects changes, it invokes `ngOnChanges()` if supplied,
 * regardless of whether you perform additional change detection.
 * Typically, you should not use both `DoCheck` and `OnChanges` to respond to
 * changes on the same input.
 *
 * @see `OnChanges`
 * @see [Lifecycle Hooks](guide/lifecycle-hooks#onchanges) guide
 *
 * @usageNotes
 * The following snippet shows how a component can implement this interface
 * to invoke it own change-detection cycle.
 *
 * {@example core/ts/metadata/lifecycle_hooks_spec.ts region='DoCheck'}
 *
 * @publicApi
 */
export declare interface DoCheck {
    /**
       * A callback method that performs change-detection, invoked
       * after the default change-detector runs.
       * See `KeyValueDiffers` and `IterableDiffers` for implementing
       * custom change checking for collections.
       *
       */
    ngDoCheck(): void;
}

/**
 * Marks that the next string is for element.
 *
 * See `I18nMutateOpCodes` documentation.
 */
declare const ELEMENT_MARKER: ELEMENT_MARKER;

declare interface ELEMENT_MARKER {
    marker: 'element';
}

declare interface ElementDef {
    name: string | null;
    ns: string | null;
    /** ns, name, value */
    attrs: [string, string, string][] | null;
    template: ɵViewDefinition | null;
    componentProvider: NodeDef | null;
    componentRendererType: RendererType2 | null;
    componentView: ViewDefinitionFactory | null;
    /**
     * visible public providers for DI in the view,
     * as see from this element. This does not include private providers.
     */
    publicProviders: {
        [tokenKey: string]: NodeDef;
    } | null;
    /**
     * same as visiblePublicProviders, but also includes private providers
     * that are located on this element.
     */
    allProviders: {
        [tokenKey: string]: NodeDef;
    } | null;
    handleEvent: ElementHandleEventFn | null;
}

declare interface ElementHandleEventFn {
    (view: ViewData, eventName: string, event: any): boolean;
}

/**
 * A wrapper around a native element inside of a View.
 *
 * An `ElementRef` is backed by a render-specific element. In the browser, this is usually a DOM
 * element.
 *
 * @security Permitting direct access to the DOM can make your application more vulnerable to
 * XSS attacks. Carefully review any use of `ElementRef` in your code. For more detail, see the
 * [Security Guide](http://g.co/ng/security).
 *
 * @publicApi
 */
export declare class ElementRef<T extends any = any> {
    /**
     * The underlying native element or `null` if direct access to native elements is not supported
     * (e.g. when the application runs in a web worker).
     *
     * <div class="callout is-critical">
     *   <header>Use with caution</header>
     *   <p>
     *    Use this API as the last resort when direct access to DOM is needed. Use templating and
     *    data-binding provided by Angular instead. Alternatively you can take a look at {@link
     * Renderer2}
     *    which provides API that can safely be used even when direct access to native elements is not
     *    supported.
     *   </p>
     *   <p>
     *    Relying on direct DOM access creates tight coupling between your application and rendering
     *    layers which will make it impossible to separate the two and deploy your application into a
     *    web worker.
     *   </p>
     * </div>
     *
     */
    nativeElement: T;
    constructor(nativeElement: T);
}

/**
 * Represents an Angular [view](guide/glossary#view) in a view container.
 * An [embedded view](guide/glossary#view-tree) can be referenced from a component
 * other than the hosting component whose template defines it, or it can be defined
 * independently by a `TemplateRef`.
 *
 * Properties of elements in a view can change, but the structure (number and order) of elements in
 * a view cannot. Change the structure of elements by inserting, moving, or
 * removing nested views in a view container.
 *
 * @see `ViewContainerRef`
 *
 * @usageNotes
 *
 * The following template breaks down into two separate `TemplateRef` instances,
 * an outer one and an inner one.
 *
 * ```
 * Count: {{items.length}}
 * <ul>
 *   <li *ngFor="let  item of items">{{item}}</li>
 * </ul>
 * ```
 *
 * This is the outer `TemplateRef`:
 *
 * ```
 * Count: {{items.length}}
 * <ul>
 *   <ng-template ngFor let-item [ngForOf]="items"></ng-template>
 * </ul>
 * ```
 *
 * This is the inner `TemplateRef`:
 *
 * ```
 *   <li>{{item}}</li>
 * ```
 *
 * The outer and inner `TemplateRef` instances are assembled into views as follows:
 *
 * ```
 * <!-- ViewRef: outer-0 -->
 * Count: 2
 * <ul>
 *   <ng-template view-container-ref></ng-template>
 *   <!-- ViewRef: inner-1 --><li>first</li><!-- /ViewRef: inner-1 -->
 *   <!-- ViewRef: inner-2 --><li>second</li><!-- /ViewRef: inner-2 -->
 * </ul>
 * <!-- /ViewRef: outer-0 -->
 * ```
 * @publicApi
 */
export declare abstract class EmbeddedViewRef<C> extends ViewRef {
    /**
     * The context for this view, inherited from the anchor element.
     */
    abstract get context(): C;
    /**
     * The root nodes for this embedded view.
     */
    abstract get rootNodes(): any[];
}

/**
 * Disable Angular's development mode, which turns off assertions and other
 * checks within the framework.
 *
 * One important assertion this disables verifies that a change detection pass
 * does not result in additional changes to any bindings (also known as
 * unidirectional data flow).
 *
 * @publicApi
 */
export declare function enableProdMode(): void;


/**
 * Provides a hook for centralized exception handling.
 *
 * The default implementation of `ErrorHandler` prints error messages to the `console`. To
 * intercept error handling, write a custom exception handler that replaces this default as
 * appropriate for your app.
 *
 * @usageNotes
 * ### Example
 *
 * ```
 * class MyErrorHandler implements ErrorHandler {
 *   handleError(error) {
 *     // do something with the exception
 *   }
 * }
 *
 * @NgModule({
 *   providers: [{provide: ErrorHandler, useClass: MyErrorHandler}]
 * })
 * class MyModule {}
 * ```
 *
 * @publicApi
 */
export declare class ErrorHandler {
    handleError(error: any): void;
}

/**
 * Use in components with the `@Output` directive to emit custom events
 * synchronously or asynchronously, and register handlers for those events
 * by subscribing to an instance.
 *
 * @usageNotes
 *
 * Extends
 * [RxJS `Subject`](https://rxjs.dev/api/index/class/Subject)
 * for Angular by adding the `emit()` method.
 *
 * In the following example, a component defines two output properties
 * that create event emitters. When the title is clicked, the emitter
 * emits an open or close event to toggle the current visibility state.
 *
 * ```html
 * @Component({
 *   selector: 'zippy',
 *   template: `
 *   <div class="zippy">
 *     <div (click)="toggle()">Toggle</div>
 *     <div [hidden]="!visible">
 *       <ng-content></ng-content>
 *     </div>
 *  </div>`})
 * export class Zippy {
 *   visible: boolean = true;
 *   @Output() open: EventEmitter<any> = new EventEmitter();
 *   @Output() close: EventEmitter<any> = new EventEmitter();
 *
 *   toggle() {
 *     this.visible = !this.visible;
 *     if (this.visible) {
 *       this.open.emit(null);
 *     } else {
 *       this.close.emit(null);
 *     }
 *   }
 * }
 * ```
 *
 * Access the event object with the `$event` argument passed to the output event
 * handler:
 *
 * ```html
 * <zippy (open)="onOpen($event)" (close)="onClose($event)"></zippy>
 * ```
 *
 * @see [Observables in Angular](guide/observables-in-angular)
 * @publicApi
 */
export declare class EventEmitter<T extends any> extends Subject<T> {
    /**
     * Creates an instance of this class that can
     * deliver events synchronously or asynchronously.
     *
     * @param isAsync When true, deliver events asynchronously.
     *
     */
    constructor(isAsync?: boolean);
    /**
     * Emits an event containing a given value.
     * @param value The value to emit.
     */
    emit(value?: T): void;
    /**
     * Registers handlers for events emitted by this instance.
     * @param generatorOrNext When supplied, a custom handler for emitted events.
     * @param error When supplied, a custom handler for an error notification
     * from this emitter.
     * @param complete When supplied, a custom handler for a completion
     * notification from this emitter.
     */
    subscribe(generatorOrNext?: any, error?: any, complete?: any): Subscription;
}

/**
 * Configures the `Injector` to return a value of another `useExisting` token.
 *
 * @see ["Dependency Injection Guide"](guide/dependency-injection).
 *
 * @usageNotes
 *
 * {@example core/di/ts/provider_spec.ts region='ExistingProvider'}
 *
 * ### Multi-value example
 *
 * {@example core/di/ts/provider_spec.ts region='MultiProviderAspect'}
 *
 * @publicApi
 */
export declare interface ExistingProvider extends ExistingSansProvider {
    /**
     * An injection token. Typically an instance of `Type` or `InjectionToken`, but can be `any`.
     */
    provide: any;
    /**
     * When true, injector returns an array of instances. This is useful to allow multiple
     * providers spread across many files to provide configuration information to a common token.
     */
    multi?: boolean;
}

/**
 * Configures the `Injector` to return a value of another `useExisting` token.
 *
 * @see `ExistingProvider`
 * @see ["Dependency Injection Guide"](guide/dependency-injection).
 *
 * @publicApi
 */
export declare interface ExistingSansProvider {
    /**
     * Existing `token` to return. (Equivalent to `injector.get(useExisting)`)
     */
    useExisting: any;
}

/**
 * Set of instructions used to process host bindings efficiently.
 *
 * See VIEW_DATA.md for more information.
 */
declare interface ExpandoInstructions extends Array<number | HostBindingsFunction<any> | null> {
}

/**
 * Definition of what a factory function should look like.
 */
declare type FactoryFn<T> = {
    /**
     * Subclasses without an explicit constructor call through to the factory of their base
     * definition, providing it with their own constructor to instantiate.
     */
    <U extends T>(t: Type<U>): U;
    /**
     * If no constructor to instantiate is provided, an instance of type T itself is created.
     */
    (t?: undefined): T;
};

/**
 * Configures the `Injector` to return a value by invoking a `useFactory` function.
 * @see ["Dependency Injection Guide"](guide/dependency-injection).
 *
 * @usageNotes
 *
 * {@example core/di/ts/provider_spec.ts region='FactoryProvider'}
 *
 * Dependencies can also be marked as optional:
 *
 * {@example core/di/ts/provider_spec.ts region='FactoryProviderOptionalDeps'}
 *
 * ### Multi-value example
 *
 * {@example core/di/ts/provider_spec.ts region='MultiProviderAspect'}
 *
 * @publicApi
 */
export declare interface FactoryProvider extends FactorySansProvider {
    /**
     * An injection token. (Typically an instance of `Type` or `InjectionToken`, but can be `any`).
     */
    provide: any;
    /**
     * When true, injector returns an array of instances. This is useful to allow multiple
     * providers spread across many files to provide configuration information to a common token.
     */
    multi?: boolean;
}

/**
 * Configures the `Injector` to return a value by invoking a `useFactory` function.
 *
 * @see `FactoryProvider`
 * @see ["Dependency Injection Guide"](guide/dependency-injection).
 *
 * @publicApi
 */
export declare interface FactorySansProvider {
    /**
     * A function to invoke to create a value for this `token`. The function is invoked with
     * resolved values of `token`s in the `deps` field.
     */
    useFactory: Function;
    /**
     * A list of `token`s to be resolved by the injector. The list of values is then
     * used as arguments to the `useFactory` function.
     */
    deps?: any[];
}

declare const FLAGS = 2;

/**
 * Allows to refer to references which are not yet defined.
 *
 * For instance, `forwardRef` is used when the `token` which we need to refer to for the purposes of
 * DI is declared, but not yet defined. It is also used when the `token` which we use when creating
 * a query is not yet defined.
 *
 * @usageNotes
 * ### Example
 * {@example core/di/ts/forward_ref/forward_ref_spec.ts region='forward_ref'}
 * @publicApi
 */
export declare function forwardRef(forwardRefFn: ForwardRefFn): Type<any>;

/**
 * An interface that a function passed into {@link forwardRef} has to implement.
 *
 * @usageNotes
 * ### Example
 *
 * {@example core/di/ts/forward_ref/forward_ref_spec.ts region='forward_ref_fn'}
 * @publicApi
 */
export declare interface ForwardRefFn {
    (): any;
}

/**
 * @publicApi
 */
export declare const getDebugNode: (nativeNode: any) => DebugNode | null;

/**
 * Returns the NgModuleFactory with the given id, if it exists and has been loaded.
 * Factories for modules that do not specify an `id` cannot be retrieved. Throws if the module
 * cannot be found.
 * @publicApi
 */
export declare const getModuleFactory: (id: string) => NgModuleFactory<any>;

/**
 * Returns the current platform.
 *
 * @publicApi
 */
export declare function getPlatform(): PlatformRef | null;

/**
 * Adapter interface for retrieving the `Testability` service associated for a
 * particular context.
 *
 * @publicApi
 */
export declare interface GetTestability {
    addToWindow(registry: TestabilityRegistry): void;
    findTestabilityInTree(registry: TestabilityRegistry, elem: any, findInAncestors: boolean): Testability | null;
}

declare type GlobalTargetName = 'document' | 'window' | 'body';

declare type GlobalTargetResolver = (element: any) => {
    name: GlobalTargetName;
    target: EventTarget;
};

/**
 * Array of hooks that should be executed for a view and their directive indices.
 *
 * For each node of the view, the following data is stored:
 * 1) Node index (optional)
 * 2) A series of number/function pairs where:
 *  - even indices are directive indices
 *  - odd indices are hook functions
 *
 * Special cases:
 *  - a negative directive index flags an init hook (ngOnInit, ngAfterContentInit, ngAfterViewInit)
 */
declare type HookData = (number | (() => void))[];

declare const HOST = 0;

/**
 * Type of the Host metadata.
 *
 * @publicApi
 */
export declare interface Host {
}

/**
 * Host decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const Host: HostDecorator;

/**
 * Type of the HostBinding metadata.
 *
 * @publicApi
 */
export declare interface HostBinding {
    /**
     * The DOM property that is bound to a data property.
     */
    hostPropertyName?: string;
}

/**
 * @Annotation
 * @publicApi
 */
export declare const HostBinding: HostBindingDecorator;

/**
 * Type of the HostBinding decorator / constructor function.
 *
 * @publicApi
 */
export declare interface HostBindingDecorator {
    /**
     * Decorator that marks a DOM property as a host-binding property and supplies configuration
     * metadata.
     * Angular automatically checks host property bindings during change detection, and
     * if a binding changes it updates the host element of the directive.
     *
     * @usageNotes
     *
     * The following example creates a directive that sets the `valid` and `invalid`
     * properties on the DOM element that has an `ngModel` directive on it.
     *
     * ```typescript
     * @Directive({selector: '[ngModel]'})
     * class NgModelStatus {
     *   constructor(public control: NgModel) {}
     *   @HostBinding('class.valid') get valid() { return this.control.valid; }
     *   @HostBinding('class.invalid') get invalid() { return this.control.invalid; }
     * }
     *
     * @Component({
     *   selector: 'app',
     *   template: `<input [(ngModel)]="prop">`,
     * })
     * class App {
     *   prop;
     * }
     * ```
     *
     */
    (hostPropertyName?: string): any;
    new (hostPropertyName?: string): any;
}

declare type HostBindingsFunction<T> = <U extends T>(rf: ɵRenderFlags, ctx: U) => void;

/**
 * Type of the Host decorator / constructor function.
 *
 * @publicApi
 */
export declare interface HostDecorator {
    /**
     * Parameter decorator on a view-provider parameter of a class constructor
     * that tells the DI framework to resolve the view by checking injectors of child
     * elements, and stop when reaching the host element of the current component.
     *
     * For an extended example, see
     * ["Dependency Injection Guide"](guide/dependency-injection-in-action#optional).
     *
     * @usageNotes
     *
     * The following shows use with the `@Optional` decorator, and allows for a null result.
     *
     * <code-example path="core/di/ts/metadata_spec.ts" region="Host">
     * </code-example>
     */
    (): any;
    new (): Host;
}

/** See CreateComponentOptions.hostFeatures */
declare type HostFeature = (<T>(component: T, componentDef: ɵComponentDef<T>) => void);

/**
 * Type of the HostListener metadata.
 *
 * @publicApi
 */
export declare interface HostListener {
    /**
     * The DOM event to listen for.
     */
    eventName?: string;
    /**
     * A set of arguments to pass to the handler method when the event occurs.
     */
    args?: string[];
}

/**
 * Decorator that binds a DOM event to a host listener and supplies configuration metadata.
 * Angular invokes the supplied handler method when the host element emits the specified event,
 * and updates the bound element with the result.
 *
 * If the handler method returns false, applies `preventDefault` on the bound element.
 *
 * @usageNotes
 *
 * The following example declares a directive
 * that attaches a click listener to a button and counts clicks.
 *
 * ```ts
 * @Directive({selector: 'button[counting]'})
 * class CountClicks {
 *   numberOfClicks = 0;
 *
 *   @HostListener('click', ['$event.target'])
 *   onClick(btn) {
 *     console.log('button', btn, 'number of clicks:', this.numberOfClicks++);
 *  }
 * }
 *
 * @Component({
 *   selector: 'app',
 *   template: '<button counting>Increment</button>',
 * })
 * class App {}
 *
 * ```
 *
 * The following example registers another DOM event handler that listens for key-press events.
 * ``` ts
 * import { HostListener, Component } from "@angular/core";
 *
 * @Component({
 *   selector: 'app',
 *   template: `<h1>Hello, you have pressed keys {{counter}} number of times!</h1> Press any key to
 * increment the counter.
 *   <button (click)="resetCounter()">Reset Counter</button>`
 * })
 * class AppComponent {
 *   counter = 0;
 *   @HostListener('window:keydown', ['$event'])
 *   handleKeyDown(event: KeyboardEvent) {
 *     this.counter++;
 *   }
 *   resetCounter() {
 *     this.counter = 0;
 *   }
 * }
 * ```
 *
 * @Annotation
 * @publicApi
 */
export declare const HostListener: HostListenerDecorator;

/**
 * Type of the HostListener decorator / constructor function.
 *
 * @publicApi
 */
export declare interface HostListenerDecorator {
    /**
     * Decorator that declares a DOM event to listen for,
     * and provides a handler method to run when that event occurs.
     */
    (eventName: string, args?: string[]): any;
    new (eventName: string, args?: string[]): any;
}

/**
 * Array storing OpCode for dynamically creating `i18n` blocks.
 *
 * Example:
 * ```ts
 * <I18nCreateOpCode>[
 *   // For adding text nodes
 *   // ---------------------
 *   // Equivalent to:
 *   //   const node = lView[index++] = document.createTextNode('abc');
 *   //   lView[1].insertBefore(node, lView[2]);
 *   'abc', 1 << SHIFT_PARENT | 2 << SHIFT_REF | InsertBefore,
 *
 *   // Equivalent to:
 *   //   const node = lView[index++] = document.createTextNode('xyz');
 *   //   lView[1].appendChild(node);
 *   'xyz', 1 << SHIFT_PARENT | AppendChild,
 *
 *   // For adding element nodes
 *   // ---------------------
 *   // Equivalent to:
 *   //   const node = lView[index++] = document.createElement('div');
 *   //   lView[1].insertBefore(node, lView[2]);
 *   ELEMENT_MARKER, 'div', 1 << SHIFT_PARENT | 2 << SHIFT_REF | InsertBefore,
 *
 *   // Equivalent to:
 *   //   const node = lView[index++] = document.createElement('div');
 *   //   lView[1].appendChild(node);
 *   ELEMENT_MARKER, 'div', 1 << SHIFT_PARENT | AppendChild,
 *
 *   // For adding comment nodes
 *   // ---------------------
 *   // Equivalent to:
 *   //   const node = lView[index++] = document.createComment('');
 *   //   lView[1].insertBefore(node, lView[2]);
 *   COMMENT_MARKER, '', 1 << SHIFT_PARENT | 2 << SHIFT_REF | InsertBefore,
 *
 *   // Equivalent to:
 *   //   const node = lView[index++] = document.createComment('');
 *   //   lView[1].appendChild(node);
 *   COMMENT_MARKER, '', 1 << SHIFT_PARENT | AppendChild,
 *
 *   // For moving existing nodes to a different location
 *   // --------------------------------------------------
 *   // Equivalent to:
 *   //   const node = lView[1];
 *   //   lView[2].insertBefore(node, lView[3]);
 *   1 << SHIFT_REF | Select, 2 << SHIFT_PARENT | 3 << SHIFT_REF | InsertBefore,
 *
 *   // Equivalent to:
 *   //   const node = lView[1];
 *   //   lView[2].appendChild(node);
 *   1 << SHIFT_REF | Select, 2 << SHIFT_PARENT | AppendChild,
 *
 *   // For removing existing nodes
 *   // --------------------------------------------------
 *   //   const node = lView[1];
 *   //   removeChild(tView.data(1), node, lView);
 *   1 << SHIFT_REF | Remove,
 *
 *   // For writing attributes
 *   // --------------------------------------------------
 *   //   const node = lView[1];
 *   //   node.setAttribute('attr', 'value');
 *   1 << SHIFT_REF | Select, 'attr', 'value'
 *            // NOTE: Select followed by two string (vs select followed by OpCode)
 * ];
 * ```
 * NOTE:
 *   - `index` is initial location where the extra nodes should be stored in the EXPANDO section of
 * `LVIewData`.
 *
 * See: `applyI18nCreateOpCodes`;
 */
declare interface I18nMutateOpCodes extends Array<number | string | ELEMENT_MARKER | COMMENT_MARKER | null> {
}

/**
 * Stores DOM operations which need to be applied to update DOM render tree due to changes in
 * expressions.
 *
 * The basic idea is that `i18nExp` OpCodes capture expression changes and update a change
 * mask bit. (Bit 1 for expression 1, bit 2 for expression 2 etc..., bit 32 for expression 32 and
 * higher.) The OpCodes then compare its own change mask against the expression change mask to
 * determine if the OpCodes should execute.
 *
 * These OpCodes can be used by both the i18n block as well as ICU sub-block.
 *
 * ## Example
 *
 * Assume
 * ```ts
 *   if (rf & RenderFlags.Update) {
 *    i18nExp(ctx.exp1); // If changed set mask bit 1
 *    i18nExp(ctx.exp2); // If changed set mask bit 2
 *    i18nExp(ctx.exp3); // If changed set mask bit 3
 *    i18nExp(ctx.exp4); // If changed set mask bit 4
 *    i18nApply(0);            // Apply all changes by executing the OpCodes.
 *  }
 * ```
 * We can assume that each call to `i18nExp` sets an internal `changeMask` bit depending on the
 * index of `i18nExp`.
 *
 * ### OpCodes
 * ```ts
 * <I18nUpdateOpCodes>[
 *   // The following OpCodes represent: `<div i18n-title="pre{{exp1}}in{{exp2}}post">`
 *   // If `changeMask & 0b11`
 *   //        has changed then execute update OpCodes.
 *   //        has NOT changed then skip `7` values and start processing next OpCodes.
 *   0b11, 7,
 *   // Concatenate `newValue = 'pre'+lView[bindIndex-4]+'in'+lView[bindIndex-3]+'post';`.
 *   'pre', -4, 'in', -3, 'post',
 *   // Update attribute: `elementAttribute(1, 'title', sanitizerFn(newValue));`
 *   1 << SHIFT_REF | Attr, 'title', sanitizerFn,
 *
 *   // The following OpCodes represent: `<div i18n>Hello {{exp3}}!">`
 *   // If `changeMask & 0b100`
 *   //        has changed then execute update OpCodes.
 *   //        has NOT changed then skip `4` values and start processing next OpCodes.
 *   0b100, 4,
 *   // Concatenate `newValue = 'Hello ' + lView[bindIndex -2] + '!';`.
 *   'Hello ', -2, '!',
 *   // Update text: `lView[1].textContent = newValue;`
 *   1 << SHIFT_REF | Text,
 *
 *   // The following OpCodes represent: `<div i18n>{exp4, plural, ... }">`
 *   // If `changeMask & 0b1000`
 *   //        has changed then execute update OpCodes.
 *   //        has NOT changed then skip `4` values and start processing next OpCodes.
 *   0b1000, 4,
 *   // Concatenate `newValue = lView[bindIndex -1];`.
 *   -1,
 *   // Switch ICU: `icuSwitchCase(lView[1], 0, newValue);`
 *   0 << SHIFT_ICU | 1 << SHIFT_REF | IcuSwitch,
 *
 *   // Note `changeMask & -1` is always true, so the IcuUpdate will always execute.
 *   -1, 1,
 *   // Update ICU: `icuUpdateCase(lView[1], 0);`
 *   0 << SHIFT_ICU | 1 << SHIFT_REF | IcuUpdate,
 *
 * ];
 * ```
 *
 */
declare interface I18nUpdateOpCodes extends Array<string | number | SanitizerFn | null> {
}

/**
 * Defines the ICU type of `select` or `plural`
 */
declare const enum IcuType {
    select = 0,
    plural = 1
}

/**
 * This array contains information about input properties that
 * need to be set once from attribute data. It's ordered by
 * directive index (relative to element) so it's simple to
 * look up a specific directive's initial input data.
 *
 * Within each sub-array:
 *
 * i+0: attribute name
 * i+1: minified/internal input name
 * i+2: initial value
 *
 * If a directive on a node does not have any input properties
 * that should be set from attributes, its index is set to null
 * to avoid a sparse array.
 *
 * e.g. [null, ['role-min', 'minified-input', 'button']]
 */
declare type InitialInputData = (InitialInputs | null)[];

/**
 * Used by InitialInputData to store input properties
 * that should be set once from attributes.
 *
 * i+0: attribute name
 * i+1: minified/internal input name
 * i+2: initial value
 *
 * e.g. ['role-min', 'minified-input', 'button']
 */
declare type InitialInputs = string[];

/**
 * Type of the Inject metadata.
 *
 * @publicApi
 */
export declare interface Inject {
    /**
     * A [DI token](guide/glossary#di-token) that maps to the dependency to be injected.
     */
    token: any;
}

/**
 * Inject decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const Inject: InjectDecorator;

/**
 * Injects a token from the currently active injector.
 *
 * Must be used in the context of a factory function such as one defined for an
 * `InjectionToken`. Throws an error if not called from such a context.
 *
 * Within such a factory function, using this function to request injection of a dependency
 * is faster and more type-safe than providing an additional array of dependencies
 * (as has been common with `useFactory` providers).
 *
 * @param token The injection token for the dependency to be injected.
 * @param flags Optional flags that control how injection is executed.
 * The flags correspond to injection strategies that can be specified with
 * parameter decorators `@Host`, `@Self`, `@SkipSef`, and `@Optional`.
 * @returns True if injection is successful, null otherwise.
 *
 * @usageNotes
 *
 * ### Example
 *
 * {@example core/di/ts/injector_spec.ts region='ShakableInjectionToken'}
 *
 * @publicApi
 */
export declare const inject: typeof ɵɵinject;

/**
 * Type of the Injectable metadata.
 *
 * @publicApi
 */
export declare interface Injectable {
    /**
     * Determines which injectors will provide the injectable,
     * by either associating it with an `@NgModule` or other `InjectorType`,
     * or by specifying that this injectable should be provided in one of the following injectors:
     * - 'root' : The application-level injector in most apps.
     * - 'platform' : A special singleton platform injector shared by all
     * applications on the page.
     * - 'any' : Provides a unique instance in every module (including lazy modules) that injects the
     * token.
     *
     */
    providedIn?: Type<any> | 'root' | 'platform' | 'any' | null;
}

/**
 * Injectable decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const Injectable: InjectableDecorator;

/**
 * Type of the Injectable decorator / constructor function.
 *
 * @publicApi
 */
export declare interface InjectableDecorator {
    /**
     * Decorator that marks a class as available to be
     * provided and injected as a dependency.
     *
     * @see [Introduction to Services and DI](guide/architecture-services)
     * @see [Dependency Injection Guide](guide/dependency-injection)
     *
     * @usageNotes
     *
     * Marking a class with `@Injectable` ensures that the compiler
     * will generate the necessary metadata to create the class's
     * dependencies when the class is injected.
     *
     * The following example shows how a service class is properly
     *  marked so that a supporting service can be injected upon creation.
     *
     * <code-example path="core/di/ts/metadata_spec.ts" region="Injectable"></code-example>
     *
     */
    (): TypeDecorator;
    (options?: {
        providedIn: Type<any> | 'root' | 'platform' | 'any' | null;
    } & InjectableProvider): TypeDecorator;
    new (): Injectable;
    new (options?: {
        providedIn: Type<any> | 'root' | 'platform' | 'any' | null;
    } & InjectableProvider): Injectable;
}

/**
 * Injectable providers used in `@Injectable` decorator.
 *
 * @publicApi
 */
export declare type InjectableProvider = ValueSansProvider | ExistingSansProvider | StaticClassSansProvider | ConstructorSansProvider | FactorySansProvider | ClassSansProvider;

/**
 * A `Type` which has an `InjectableDef` static field.
 *
 * `InjectableDefType`s contain their own Dependency Injection metadata and are usable in an
 * `InjectorDef`-based `StaticInjector.
 *
 * @publicApi
 */
export declare interface InjectableType<T> extends Type<T> {
    /**
     * Opaque type whose structure is highly version dependent. Do not rely on any properties.
     */
    ɵprov: never;
}

/** Returns a ChangeDetectorRef (a.k.a. a ViewRef) */
declare function injectChangeDetectorRef(isPipe?: boolean): ChangeDetectorRef;


/**
 * Type of the Inject decorator / constructor function.
 *
 * @publicApi
 */
export declare interface InjectDecorator {
    /**
     * Parameter decorator on a dependency parameter of a class constructor
     * that specifies a custom provider of the dependency.
     *
     * Learn more in the ["Dependency Injection Guide"](guide/dependency-injection).
     *
     * @usageNotes
     * The following example shows a class constructor that specifies a
     * custom provider of a dependency using the parameter decorator.
     *
     * When `@Inject()` is not present, the injector uses the type annotation of the
     * parameter as the provider.
     *
     * <code-example path="core/di/ts/metadata_spec.ts" region="InjectWithoutDecorator">
     * </code-example>
     */
    (token: any): any;
    new (token: any): Inject;
}

/**
 * Creates an ElementRef from the most recent node.
 *
 * @returns The ElementRef instance to use
 */
declare function injectElementRef(ElementRefToken: typeof ElementRef): ElementRef;


/**
 * Injection flags for DI.
 *
 * @publicApi
 */
export declare enum InjectFlags {
    /** Check self and check parent injector if needed */
    Default = 0,
    /**
     * Specifies that an injector should retrieve a dependency from any injector until reaching the
     * host element of the current component. (Only used with Element Injector)
     */
    Host = 1,
    /** Don't ascend to ancestors of the node requesting injection. */
    Self = 2,
    /** Skip the node that is requesting injection. */
    SkipSelf = 4,
    /** Inject `defaultValue` instead if token not found. */
    Optional = 8
}

/**
 * Creates a token that can be used in a DI Provider.
 *
 * Use an `InjectionToken` whenever the type you are injecting is not reified (does not have a
 * runtime representation) such as when injecting an interface, callable type, array or
 * parameterized type.
 *
 * `InjectionToken` is parameterized on `T` which is the type of object which will be returned by
 * the `Injector`. This provides additional level of type safety.
 *
 * ```
 * interface MyInterface {...}
 * var myInterface = injector.get(new InjectionToken<MyInterface>('SomeToken'));
 * // myInterface is inferred to be MyInterface.
 * ```
 *
 * When creating an `InjectionToken`, you can optionally specify a factory function which returns
 * (possibly by creating) a default value of the parameterized type `T`. This sets up the
 * `InjectionToken` using this factory as a provider as if it was defined explicitly in the
 * application's root injector. If the factory function, which takes zero arguments, needs to inject
 * dependencies, it can do so using the `inject` function. See below for an example.
 *
 * Additionally, if a `factory` is specified you can also specify the `providedIn` option, which
 * overrides the above behavior and marks the token as belonging to a particular `@NgModule`. As
 * mentioned above, `'root'` is the default value for `providedIn`.
 *
 * @usageNotes
 * ### Basic Example
 *
 * ### Plain InjectionToken
 *
 * {@example core/di/ts/injector_spec.ts region='InjectionToken'}
 *
 * ### Tree-shakable InjectionToken
 *
 * {@example core/di/ts/injector_spec.ts region='ShakableInjectionToken'}
 *
 *
 * @publicApi
 */
export declare class InjectionToken<T> {
    protected _desc: string;
    readonly ɵprov: never | undefined;
    constructor(_desc: string, options?: {
        providedIn?: Type<any> | 'root' | 'platform' | 'any' | null;
        factory: () => T;
    });
    toString(): string;
}

/**
 * An InjectionToken that gets the current `Injector` for `createInjector()`-style injectors.
 *
 * Requesting this token instead of `Injector` allows `StaticInjector` to be tree-shaken from a
 * project.
 *
 * @publicApi
 */
export declare const INJECTOR: InjectionToken<Injector>;

/**
 * Concrete injectors implement this interface. Injectors are configured
 * with [providers](guide/glossary#provider) that associate
 * dependencies of various types with [injection tokens](guide/glossary#di-token).
 *
 * @see ["DI Providers"](guide/dependency-injection-providers).
 * @see `StaticProvider`
 *
 * @usageNotes
 *
 *  The following example creates a service injector instance.
 *
 * {@example core/di/ts/provider_spec.ts region='ConstructorProvider'}
 *
 * ### Usage example
 *
 * {@example core/di/ts/injector_spec.ts region='Injector'}
 *
 * `Injector` returns itself when given `Injector` as a token:
 *
 * {@example core/di/ts/injector_spec.ts region='injectInjector'}
 *
 * @publicApi
 */
export declare abstract class Injector {
    static THROW_IF_NOT_FOUND: {};
    static NULL: Injector;
    /**
     * Retrieves an instance from the injector based on the provided token.
     * @returns The instance from the injector if defined, otherwise the `notFoundValue`.
     * @throws When the `notFoundValue` is `undefined` or `Injector.THROW_IF_NOT_FOUND`.
     */
    abstract get<T>(token: Type<T> | InjectionToken<T> | AbstractType<T>, notFoundValue?: T, flags?: InjectFlags): T;
    /**
     * @deprecated from v4.0.0 use Type<T> or InjectionToken<T>
     * @suppress {duplicate}
     */
    abstract get(token: any, notFoundValue?: any): any;
    /**
     * @deprecated from v5 use the new signature Injector.create(options)
     */
    static create(providers: StaticProvider[], parent?: Injector): Injector;
    /**
     * Creates a new injector instance that provides one or more dependencies,
     * according to a given type or types of `StaticProvider`.
     *
     * @param options An object with the following properties:
     * * `providers`: An array of providers of the [StaticProvider type](api/core/StaticProvider).
     * * `parent`: (optional) A parent injector.
     * * `name`: (optional) A developer-defined identifying name for the new injector.
     *
     * @returns The new injector instance.
     *
     */
    static create(options: {
        providers: StaticProvider[];
        parent?: Injector;
        name?: string;
    }): Injector;
    /** @nocollapse */
    static ɵprov: never;
}

declare const INJECTOR_2 = 9;

/**
 * A type which has an `InjectorDef` static field.
 *
 * `InjectorDefTypes` can be used to configure a `StaticInjector`.
 *
 * @publicApi
 */
export declare interface InjectorType<T> extends Type<T> {
    /**
     * Opaque type whose structure is highly version dependent. Do not rely on any properties.
     */
    ɵinj: never;
}

/**
 * Describes the `InjectorDef` equivalent of a `ModuleWithProviders`, an `InjectorDefType` with an
 * associated array of providers.
 *
 * Objects of this type can be listed in the imports section of an `InjectorDef`.
 *
 * NOTE: This is a private type and should not be exported
 */
declare interface InjectorTypeWithProviders<T> {
    ngModule: InjectorType<T>;
    providers?: (Type<any> | ValueProvider | ExistingProvider | FactoryProvider | ConstructorProvider | StaticClassProvider | ClassProvider | any[])[];
}

/** Injects a Renderer2 for the current component. */
declare function injectRenderer2(): Renderer2;

/**
 * Creates a TemplateRef given a node.
 *
 * @returns The TemplateRef instance to use
 */
declare function injectTemplateRef<T>(TemplateRefToken: typeof TemplateRef, ElementRefToken: typeof ElementRef): TemplateRef<T> | null;

/**
 * Creates a ViewContainerRef and stores it on the injector. Or, if the ViewContainerRef
 * already exists, retrieves the existing ViewContainerRef.
 *
 * @returns The ViewContainerRef instance to use
 */
declare function injectViewContainerRef(ViewContainerRefToken: typeof ViewContainerRef, ElementRefToken: typeof ElementRef): ViewContainerRef;

/**
 * Type of metadata for an `Input` property.
 *
 * @publicApi
 */
export declare interface Input {
    /**
     * The name of the DOM property to which the input property is bound.
     */
    bindingPropertyName?: string;
}

/**
 * @Annotation
 * @publicApi
 */
export declare const Input: InputDecorator;

/**
 * @publicApi
 */
export declare interface InputDecorator {
    /**
    * Decorator that marks a class field as an input property and supplies configuration metadata.
    * The input property is bound to a DOM property in the template. During change detection,
    * Angular automatically updates the data property with the DOM property's value.
    *
    * @usageNotes
    *
    * You can supply an optional name to use in templates when the
    * component is instantiated, that maps to the
    * name of the bound property. By default, the original
    * name of the bound property is used for input binding.
    *
    * The following example creates a component with two input properties,
    * one of which is given a special binding name.
    *
    * ```typescript
    * @Component({
    *   selector: 'bank-account',
    *   template: `
    *     Bank Name: {{bankName}}
    *     Account Id: {{id}}
    *   `
    * })
    * class BankAccount {
    *   // This property is bound using its original name.
    *   @Input() bankName: string;
    *   // this property value is bound to a different property name
    *   // when this component is instantiated in a template.
    *   @Input('account-id') id: string;
    *
    *   // this property is not bound, and is not automatically updated by Angular
    *   normalizedBankName: string;
    * }
    *
    * @Component({
    *   selector: 'app',
    *   template: `
    *     <bank-account bankName="RBC" account-id="4747"></bank-account>
    *   `
    * })
    * class App {}
    * ```
    *
    * @see [Input and Output properties](guide/template-syntax#input-and-output-properties)
    */
    (bindingPropertyName?: string): any;
    new (bindingPropertyName?: string): any;
}

/**
 * All implicit instruction state is stored here.
 *
 * It is useful to have a single object where all of the state is stored as a mental model
 * (rather it being spread across many different variables.)
 *
 * PERF NOTE: Turns out that writing to a true global variable is slower than
 * having an intermediate object with properties.
 */
declare interface InstructionState {
    /**
     * Current `LFrame`
     *
     * `null` if we have not called `enterView`
     */
    lFrame: LFrame;
    /**
     * Stores whether directives should be matched to elements.
     *
     * When template contains `ngNonBindable` then we need to prevent the runtime from matching
     * directives on children of that element.
     *
     * Example:
     * ```
     * <my-comp my-directive>
     *   Should match component / directive.
     * </my-comp>
     * <div ngNonBindable>
     *   <my-comp my-directive>
     *     Should not match component / directive because we are in ngNonBindable.
     *   </my-comp>
     * </div>
     * ```
     */
    bindingsEnabled: boolean;
    /**
     * In this mode, any changes in bindings will throw an ExpressionChangedAfterChecked error.
     *
     * Necessary to support ChangeDetectorRef.checkNoChanges().
     */
    checkNoChangesMode: boolean;
}

declare interface InternalNgModuleRef<T> extends NgModuleRef<T> {
    _bootstrapComponents: Type<any>[];
}

declare interface InternalViewRef extends ViewRef {
    detachFromAppRef(): void;
    attachToAppRef(appRef: ApplicationRef): void;
}


/**
 * Returns whether Angular is in development mode. After called once,
 * the value is locked and won't change any more.
 *
 * By default, this is true, unless a user calls `enableProdMode` before calling this.
 *
 * @publicApi
 */
export declare function isDevMode(): boolean;

/**
 * Record representing the item change information.
 *
 * @publicApi
 */
export declare interface IterableChangeRecord<V> {
    /** Current index of the item in `Iterable` or null if removed. */
    readonly currentIndex: number | null;
    /** Previous index of the item in `Iterable` or null if added. */
    readonly previousIndex: number | null;
    /** The item. */
    readonly item: V;
    /** Track by identity as computed by the `TrackByFunction`. */
    readonly trackById: any;
}

declare class IterableChangeRecord_<V> implements IterableChangeRecord<V> {
    item: V;
    trackById: any;
    currentIndex: number | null;
    previousIndex: number | null;
    constructor(item: V, trackById: any);
}

/**
 * An object describing the changes in the `Iterable` collection since last time
 * `IterableDiffer#diff()` was invoked.
 *
 * @publicApi
 */
export declare interface IterableChanges<V> {
    /**
     * Iterate over all changes. `IterableChangeRecord` will contain information about changes
     * to each item.
     */
    forEachItem(fn: (record: IterableChangeRecord<V>) => void): void;
    /**
     * Iterate over a set of operations which when applied to the original `Iterable` will produce the
     * new `Iterable`.
     *
     * NOTE: These are not necessarily the actual operations which were applied to the original
     * `Iterable`, rather these are a set of computed operations which may not be the same as the
     * ones applied.
     *
     * @param record A change which needs to be applied
     * @param previousIndex The `IterableChangeRecord#previousIndex` of the `record` refers to the
     *        original `Iterable` location, where as `previousIndex` refers to the transient location
     *        of the item, after applying the operations up to this point.
     * @param currentIndex The `IterableChangeRecord#currentIndex` of the `record` refers to the
     *        original `Iterable` location, where as `currentIndex` refers to the transient location
     *        of the item, after applying the operations up to this point.
     */
    forEachOperation(fn: (record: IterableChangeRecord<V>, previousIndex: number | null, currentIndex: number | null) => void): void;
    /**
     * Iterate over changes in the order of original `Iterable` showing where the original items
     * have moved.
     */
    forEachPreviousItem(fn: (record: IterableChangeRecord<V>) => void): void;
    /** Iterate over all added items. */
    forEachAddedItem(fn: (record: IterableChangeRecord<V>) => void): void;
    /** Iterate over all moved items. */
    forEachMovedItem(fn: (record: IterableChangeRecord<V>) => void): void;
    /** Iterate over all removed items. */
    forEachRemovedItem(fn: (record: IterableChangeRecord<V>) => void): void;
    /** Iterate over all items which had their identity (as computed by the `TrackByFunction`)
     * changed. */
    forEachIdentityChange(fn: (record: IterableChangeRecord<V>) => void): void;
}

/**
 * A strategy for tracking changes over time to an iterable. Used by {@link NgForOf} to
 * respond to changes in an iterable by effecting equivalent changes in the DOM.
 *
 * @publicApi
 */
export declare interface IterableDiffer<V> {
    /**
     * Compute a difference between the previous state and the new `object` state.
     *
     * @param object containing the new value.
     * @returns an object describing the difference. The return value is only valid until the next
     * `diff()` invocation.
     */
    diff(object: NgIterable<V> | undefined | null): IterableChanges<V> | null;
}

/**
 * Provides a factory for {@link IterableDiffer}.
 *
 * @publicApi
 */
export declare interface IterableDifferFactory {
    supports(objects: any): boolean;
    create<V>(trackByFn?: TrackByFunction<V>): IterableDiffer<V>;
}

/**
 * A repository of different iterable diffing strategies used by NgFor, NgClass, and others.
 *
 * @publicApi
 */
export declare class IterableDiffers {
    /** @nocollapse */
    static ɵprov: never;
    /**
     * @deprecated v4.0.0 - Should be private
     */
    factories: IterableDifferFactory[];
    constructor(factories: IterableDifferFactory[]);
    static create(factories: IterableDifferFactory[], parent?: IterableDiffers): IterableDiffers;
    /**
     * Takes an array of {@link IterableDifferFactory} and returns a provider used to extend the
     * inherited {@link IterableDiffers} instance with the provided factories and return a new
     * {@link IterableDiffers} instance.
     *
     * @usageNotes
     * ### Example
     *
     * The following example shows how to extend an existing list of factories,
     * which will only be applied to the injector for this component and its children.
     * This step is all that's required to make a new {@link IterableDiffer} available.
     *
     * ```
     * @Component({
     *   viewProviders: [
     *     IterableDiffers.extend([new ImmutableListDiffer()])
     *   ]
     * })
     * ```
     */
    static extend(factories: IterableDifferFactory[]): StaticProvider;
    find(iterable: any): IterableDifferFactory;
}

/**
 * `KeyValueArray` is an array where even positions contain keys and odd positions contain values.
 *
 * `KeyValueArray` provides a very efficient way of iterating over its contents. For small
 * sets (~10) the cost of binary searching an `KeyValueArray` has about the same performance
 * characteristics that of a `Map` with significantly better memory footprint.
 *
 * If used as a `Map` the keys are stored in alphabetical order so that they can be binary searched
 * for retrieval.
 *
 * See: `keyValueArraySet`, `keyValueArrayGet`, `keyValueArrayIndexOf`, `keyValueArrayDelete`.
 */
declare interface KeyValueArray<VALUE> extends Array<VALUE | string> {
    __brand__: 'array-map';
}

/**
 * Record representing the item change information.
 *
 * @publicApi
 */
export declare interface KeyValueChangeRecord<K, V> {
    /**
     * Current key in the Map.
     */
    readonly key: K;
    /**
     * Current value for the key or `null` if removed.
     */
    readonly currentValue: V | null;
    /**
     * Previous value for the key or `null` if added.
     */
    readonly previousValue: V | null;
}

/**
 * An object describing the changes in the `Map` or `{[k:string]: string}` since last time
 * `KeyValueDiffer#diff()` was invoked.
 *
 * @publicApi
 */
export declare interface KeyValueChanges<K, V> {
    /**
     * Iterate over all changes. `KeyValueChangeRecord` will contain information about changes
     * to each item.
     */
    forEachItem(fn: (r: KeyValueChangeRecord<K, V>) => void): void;
    /**
     * Iterate over changes in the order of original Map showing where the original items
     * have moved.
     */
    forEachPreviousItem(fn: (r: KeyValueChangeRecord<K, V>) => void): void;
    /**
     * Iterate over all keys for which values have changed.
     */
    forEachChangedItem(fn: (r: KeyValueChangeRecord<K, V>) => void): void;
    /**
     * Iterate over all added items.
     */
    forEachAddedItem(fn: (r: KeyValueChangeRecord<K, V>) => void): void;
    /**
     * Iterate over all removed items.
     */
    forEachRemovedItem(fn: (r: KeyValueChangeRecord<K, V>) => void): void;
}

/**
 * A differ that tracks changes made to an object over time.
 *
 * @publicApi
 */
export declare interface KeyValueDiffer<K, V> {
    /**
     * Compute a difference between the previous state and the new `object` state.
     *
     * @param object containing the new value.
     * @returns an object describing the difference. The return value is only valid until the next
     * `diff()` invocation.
     */
    diff(object: Map<K, V>): KeyValueChanges<K, V> | null;
    /**
     * Compute a difference between the previous state and the new `object` state.
     *
     * @param object containing the new value.
     * @returns an object describing the difference. The return value is only valid until the next
     * `diff()` invocation.
     */
    diff(object: {
        [key: string]: V;
    }): KeyValueChanges<string, V> | null;
}

/**
 * Provides a factory for {@link KeyValueDiffer}.
 *
 * @publicApi
 */
export declare interface KeyValueDifferFactory {
    /**
     * Test to see if the differ knows how to diff this kind of object.
     */
    supports(objects: any): boolean;
    /**
     * Create a `KeyValueDiffer`.
     */
    create<K, V>(): KeyValueDiffer<K, V>;
}

/**
 * A repository of different Map diffing strategies used by NgClass, NgStyle, and others.
 *
 * @publicApi
 */
export declare class KeyValueDiffers {
    /** @nocollapse */
    static ɵprov: never;
    /**
     * @deprecated v4.0.0 - Should be private.
     */
    factories: KeyValueDifferFactory[];
    constructor(factories: KeyValueDifferFactory[]);
    static create<S>(factories: KeyValueDifferFactory[], parent?: KeyValueDiffers): KeyValueDiffers;
    /**
     * Takes an array of {@link KeyValueDifferFactory} and returns a provider used to extend the
     * inherited {@link KeyValueDiffers} instance with the provided factories and return a new
     * {@link KeyValueDiffers} instance.
     *
     * @usageNotes
     * ### Example
     *
     * The following example shows how to extend an existing list of factories,
     * which will only be applied to the injector for this component and its children.
     * This step is all that's required to make a new {@link KeyValueDiffer} available.
     *
     * ```
     * @Component({
     *   viewProviders: [
     *     KeyValueDiffers.extend([new ImmutableMapDiffer()])
     *   ]
     * })
     * ```
     */
    static extend<S>(factories: KeyValueDifferFactory[]): StaticProvider;
    find(kv: any): KeyValueDifferFactory;
}

/**
 * The state associated with a container.
 *
 * This is an array so that its structure is closer to LView. This helps
 * when traversing the view tree (which is a mix of containers and component
 * views), so we can jump to viewOrContainer[NEXT] in the same way regardless
 * of type.
 */
declare interface LContainer extends Array<any> {
    /**
     * The host element of this LContainer.
     *
     * The host could be an LView if this container is on a component node.
     * In that case, the component LView is its HOST.
     */
    readonly [HOST]: RElement | RComment | ɵangular_packages_core_core_bp;
    /**
     * This is a type field which allows us to differentiate `LContainer` from `StylingContext` in an
     * efficient way. The value is always set to `true`
     */
    [TYPE]: true;
    /**
     * The next active index in the views array to read or write to. This helps us
     * keep track of where we are in the views array.
     * In the case the LContainer is created for a ViewContainerRef,
     * it is set to null to identify this scenario, as indices are "absolute" in that case,
     * i.e. provided directly by the user of the ViewContainerRef API.
     *
     * This is used by `ɵɵembeddedViewStart` to track which `LView` is currently active.
     * Because `ɵɵembeddedViewStart` is not generated by the compiler this feature is essentially
     * unused.
     *
     * The lowest bit signals that this `LContainer` has transplanted views which need to be change
     * detected as part of the declaration CD. (See `LView[DECLARATION_COMPONENT_VIEW]`)
     */
    [ACTIVE_INDEX]: ActiveIndexFlag;
    /**
     * Access to the parent view is necessary so we can propagate back
     * up from inside a container to parent[NEXT].
     */
    [PARENT]: ɵangular_packages_core_core_bp;
    /**
     * This allows us to jump from a container to a sibling container or component
     * view with the same parent, so we can remove listeners efficiently.
     */
    [NEXT]: ɵangular_packages_core_core_bp | LContainer | null;
    /**
     * A collection of views created based on the underlying `<ng-template>` element but inserted into
     * a different `LContainer`. We need to track views created from a given declaration point since
     * queries collect matches from the embedded view declaration point and _not_ the insertion point.
     */
    [MOVED_VIEWS]: ɵangular_packages_core_core_bp[] | null;
    /**
     * Pointer to the `TNode` which represents the host of the container.
     */
    [T_HOST]: TNode;
    /** The comment element that serves as an anchor for this LContainer. */
    readonly [NATIVE]: RComment;
    /**
     * Array of `ViewRef`s used by any `ViewContainerRef`s that point to this container.
     *
     * This is lazily initialized by `ViewContainerRef` when the first view is inserted.
     */
    [VIEW_REFS]: ViewRef[] | null;
}

/**
 *
 */
declare interface LFrame {
    /**
     * Parent LFrame.
     *
     * This is needed when `leaveView` is called to restore the previous state.
     */
    parent: LFrame;
    /**
     * Child LFrame.
     *
     * This is used to cache existing LFrames to relieve the memory pressure.
     */
    child: LFrame | null;
    /**
     * State of the current view being processed.
     *
     * An array of nodes (text, element, container, etc), pipes, their bindings, and
     * any local variables that need to be stored between invocations.
     */
    lView: ɵangular_packages_core_core_bp;
    /**
     * Current `TView` associated with the `LFrame.lView`.
     *
     * One can get `TView` from `lFrame[TVIEW]` however because it is so common it makes sense to
     * store it in `LFrame` for perf reasons.
     */
    tView: TView;
    /**
     * Used to set the parent property when nodes are created and track query results.
     *
     * This is used in conjunction with `isParent`.
     */
    previousOrParentTNode: TNode;
    /**
     * If `isParent` is:
     *  - `true`: then `previousOrParentTNode` points to a parent node.
     *  - `false`: then `previousOrParentTNode` points to previous node (sibling).
     */
    isParent: boolean;
    /**
     * Index of currently selected element in LView.
     *
     * Used by binding instructions. Updated as part of advance instruction.
     */
    selectedIndex: number;
    /**
     * Current pointer to the binding index.
     */
    bindingIndex: number;
    /**
     * The last viewData retrieved by nextContext().
     * Allows building nextContext() and reference() calls.
     *
     * e.g. const inner = x().$implicit; const outer = x().$implicit;
     */
    contextLView: ɵangular_packages_core_core_bp;
    /**
     * Store the element depth count. This is used to identify the root elements of the template
     * so that we can then attach patch data `LView` to only those elements. We know that those
     * are the only places where the patch data could change, this way we will save on number
     * of places where tha patching occurs.
     */
    elementDepthCount: number;
    /**
     * Current namespace to be used when creating elements
     */
    currentNamespace: string | null;
    /**
     * Current sanitizer
     */
    currentSanitizer: StyleSanitizeFn | null;
    /**
     * The root index from which pure function instructions should calculate their binding
     * indices. In component views, this is TView.bindingStartIndex. In a host binding
     * context, this is the TView.expandoStartIndex + any dirs/hostVars before the given dir.
     */
    bindingRootIndex: number;
    /**
     * Current index of a View or Content Query which needs to be processed next.
     * We iterate over the list of Queries and increment current query index at every step.
     */
    currentQueryIndex: number;
    /**
     * When host binding is executing this points to the directive index.
     * `TView.data[currentDirectiveIndex]` is `DirectiveDef`
     * `LView[currentDirectiveIndex]` is directive instance.
     */
    currentDirectiveIndex: number;
}

/**
 * Provide this token to set the locale of your application.
 * It is used for i18n extraction, by i18n pipes (DatePipe, I18nPluralPipe, CurrencyPipe,
 * DecimalPipe and PercentPipe) and by ICU expressions.
 *
 * See the [i18n guide](guide/i18n#setting-up-locale) for more information.
 *
 * @usageNotes
 * ### Example
 *
 * ```typescript
 * import { LOCALE_ID } from '@angular/core';
 * import { platformBrowserDynamic } from '@angular/platform-browser-dynamic';
 * import { AppModule } from './app/app.module';
 *
 * platformBrowserDynamic().bootstrapModule(AppModule, {
 *   providers: [{provide: LOCALE_ID, useValue: 'en-US' }]
 * });
 * ```
 *
 * @publicApi
 */
export declare const LOCALE_ID: InjectionToken<string>;

/**
 * Type for a function that extracts a value for a local refs.
 * Example:
 * - `<div #nativeDivEl>` - `nativeDivEl` should point to the native `<div>` element;
 * - `<ng-template #tplRef>` - `tplRef` should point to the `TemplateRef` instance;
 */
declare type LocalRefExtractor = (tNode: TNodeWithLocalRefs, currentView: ɵangular_packages_core_core_bp) => any;

/**
 * lQueries represent a collection of individual LQuery objects tracked in a given view.
 */
declare interface LQueries {
    /**
     * A collection of queries tracked in a given view.
     */
    queries: LQuery<any>[];
    /**
     * A method called when a new embedded view is created. As a result a set of LQueries applicable
     * for a new embedded view is instantiated (cloned) from the declaration view.
     * @param tView
     */
    createEmbeddedView(tView: TView): LQueries | null;
    /**
     * A method called when an embedded view is inserted into a container. As a result all impacted
     * `LQuery` objects (and associated `QueryList`) are marked as dirty.
     * @param tView
     */
    insertView(tView: TView): void;
    /**
     * A method called when an embedded view is detached from a container. As a result all impacted
     * `LQuery` objects (and associated `QueryList`) are marked as dirty.
     * @param tView
     */
    detachView(tView: TView): void;
}

/**
 * An interface that represents query-related information specific to a view instance. Most notably
 * it contains:
 * - materialized query matches;
 * - a pointer to a QueryList where materialized query results should be reported.
 */
declare interface LQuery<T> {
    /**
     * Materialized query matches for a given view only (!). Results are initialized lazily so the
     * array of matches is set to `null` initially.
     */
    matches: (T | null)[] | null;
    /**
     * A QueryList where materialized query results should be reported.
     */
    queryList: QueryList<T>;
    /**
     * Clones an LQuery for an embedded view. A cloned query shares the same `QueryList` but has a
     * separate collection of materialized matches.
     */
    clone(): LQuery<T>;
    /**
     * Called when an embedded view, impacting results of this query, is inserted or removed.
     */
    setDirty(): void;
}

/** Flags associated with an LView (saved in LView[FLAGS]) */
declare const enum LViewFlags {
    /** The state of the init phase on the first 2 bits */
    InitPhaseStateIncrementer = 1,
    InitPhaseStateMask = 3,
    /**
     * Whether or not the view is in creationMode.
     *
     * This must be stored in the view rather than using `data` as a marker so that
     * we can properly support embedded views. Otherwise, when exiting a child view
     * back into the parent view, `data` will be defined and `creationMode` will be
     * improperly reported as false.
     */
    CreationMode = 4,
    /**
     * Whether or not this LView instance is on its first processing pass.
     *
     * An LView instance is considered to be on its "first pass" until it
     * has completed one creation mode run and one update mode run. At this
     * time, the flag is turned off.
     */
    FirstLViewPass = 8,
    /** Whether this view has default change detection strategy (checks always) or onPush */
    CheckAlways = 16,
    /**
     * Whether or not manual change detection is turned on for onPush components.
     *
     * This is a special mode that only marks components dirty in two cases:
     * 1) There has been a change to an @Input property
     * 2) `markDirty()` has been called manually by the user
     *
     * Note that in this mode, the firing of events does NOT mark components
     * dirty automatically.
     *
     * Manual mode is turned off by default for backwards compatibility, as events
     * automatically mark OnPush components dirty in View Engine.
     *
     * TODO: Add a public API to ChangeDetectionStrategy to turn this mode on
     */
    ManualOnPush = 32,
    /** Whether or not this view is currently dirty (needing check) */
    Dirty = 64,
    /** Whether or not this view is currently attached to change detection tree. */
    Attached = 128,
    /** Whether or not this view is destroyed. */
    Destroyed = 256,
    /** Whether or not this view is the root view */
    IsRoot = 512,
    /**
     * Index of the current init phase on last 22 bits
     */
    IndexWithinInitPhaseIncrementer = 1024,
    IndexWithinInitPhaseShift = 10,
    IndexWithinInitPhaseReset = 1023
}

/**
 * Use this enum at bootstrap as an option of `bootstrapModule` to define the strategy
 * that the compiler should use in case of missing translations:
 * - Error: throw if you have missing translations.
 * - Warning (default): show a warning in the console and/or shell.
 * - Ignore: do nothing.
 *
 * See the [i18n guide](guide/i18n#missing-translation) for more information.
 *
 * @usageNotes
 * ### Example
 * ```typescript
 * import { MissingTranslationStrategy } from '@angular/core';
 * import { platformBrowserDynamic } from '@angular/platform-browser-dynamic';
 * import { AppModule } from './app/app.module';
 *
 * platformBrowserDynamic().bootstrapModule(AppModule, {
 *   missingTranslation: MissingTranslationStrategy.Error
 * });
 * ```
 *
 * @publicApi
 */
export declare enum MissingTranslationStrategy {
    Error = 0,
    Warning = 1,
    Ignore = 2
}

/**
 * Combination of NgModuleFactory and ComponentFactorys.
 *
 * @publicApi
 */
export declare class ModuleWithComponentFactories<T> {
    ngModuleFactory: NgModuleFactory<T>;
    componentFactories: ComponentFactory<any>[];
    constructor(ngModuleFactory: NgModuleFactory<T>, componentFactories: ComponentFactory<any>[]);
}

/**
 * A wrapper around an NgModule that associates it with the providers.
 *
 * @param T the module type. In Ivy applications, this must be explicitly
 * provided.
 *
 * Note that using ModuleWithProviders without a generic type is deprecated.
 * The generic will become required in a future version of Angular.
 *
 * @publicApi
 */
export declare interface ModuleWithProviders<T = any /** TODO(alxhub): remove default when callers pass explicit type param */> {
    ngModule: Type<T>;
    providers?: Provider[];
}

declare const MOVED_VIEWS = 5;

declare const NATIVE = 7;

declare const NEXT = 4;

declare interface NgContentDef {
    /**
     * this index is checked against NodeDef.ngContentIndex to find the nodes
     * that are matched by this ng-content.
     * Note that a NodeDef with an ng-content can be reprojected, i.e.
     * have a ngContentIndex on its own.
     */
    index: number;
}

/**
 * A type describing supported iterable types.
 *
 * @publicApi
 */
export declare type NgIterable<T> = Array<T> | Iterable<T>;

/**
 * Type of the NgModule metadata.
 *
 * @publicApi
 */
export declare interface NgModule {
    /**
     * The set of injectable objects that are available in the injector
     * of this module.
     *
     * @see [Dependency Injection guide](guide/dependency-injection)
     * @see [NgModule guide](guide/providers)
     *
     * @usageNotes
     *
     * Dependencies whose providers are listed here become available for injection
     * into any component, directive, pipe or service that is a child of this injector.
     * The NgModule used for bootstrapping uses the root injector, and can provide dependencies
     * to any part of the app.
     *
     * A lazy-loaded module has its own injector, typically a child of the app root injector.
     * Lazy-loaded services are scoped to the lazy-loaded module's injector.
     * If a lazy-loaded module also provides the `UserService`, any component created
     * within that module's context (such as by router navigation) gets the local instance
     * of the service, not the instance in the root injector.
     * Components in external modules continue to receive the instance provided by their injectors.
     *
     * ### Example
     *
     * The following example defines a class that is injected in
     * the HelloWorld NgModule:
     *
     * ```
     * class Greeter {
     *    greet(name:string) {
     *      return 'Hello ' + name + '!';
     *    }
     * }
     *
     * @NgModule({
     *   providers: [
     *     Greeter
     *   ]
     * })
     * class HelloWorld {
     *   greeter:Greeter;
     *
     *   constructor(greeter:Greeter) {
     *     this.greeter = greeter;
     *   }
     * }
     * ```
     */
    providers?: Provider[];
    /**
     * The set of components, directives, and pipes ([declarables](guide/glossary#declarable))
     * that belong to this module.
     *
     * @usageNotes
     *
     * The set of selectors that are available to a template include those declared here, and
     * those that are exported from imported NgModules.
     *
     * Declarables must belong to exactly one module.
     * The compiler emits an error if you try to declare the same class in more than one module.
     * Be careful not to declare a class that is imported from another module.
     *
     * ### Example
     *
     * The following example allows the CommonModule to use the `NgFor`
     * directive.
     *
     * ```javascript
     * @NgModule({
     *   declarations: [NgFor]
     * })
     * class CommonModule {
     * }
     * ```
     */
    declarations?: Array<Type<any> | any[]>;
    /**
     * The set of NgModules whose exported [declarables](guide/glossary#declarable)
     * are available to templates in this module.
     *
     * @usageNotes
     *
     * A template can use exported declarables from any
     * imported module, including those from modules that are imported indirectly
     * and re-exported.
     * For example, `ModuleA` imports `ModuleB`, and also exports
     * it, which makes the declarables from `ModuleB` available
     * wherever `ModuleA` is imported.
     *
     * ### Example
     *
     * The following example allows MainModule to use anything exported by
     * `CommonModule`:
     *
     * ```javascript
     * @NgModule({
     *   imports: [CommonModule]
     * })
     * class MainModule {
     * }
     * ```
     *
     */
    imports?: Array<Type<any> | ModuleWithProviders<{}> | any[]>;
    /**
     * The set of components, directives, and pipes declared in this
     * NgModule that can be used in the template of any component that is part of an
     * NgModule that imports this NgModule. Exported declarations are the module's public API.
     *
     * A declarable belongs to one and only one NgModule.
     * A module can list another module among its exports, in which case all of that module's
     * public declaration are exported.
     *
     * @usageNotes
     *
     * Declarations are private by default.
     * If this ModuleA does not export UserComponent, then only the components within this
     * ModuleA can use UserComponent.
     *
     * ModuleA can import ModuleB and also export it, making exports from ModuleB
     * available to an NgModule that imports ModuleA.
     *
     * ### Example
     *
     * The following example exports the `NgFor` directive from CommonModule.
     *
     * ```javascript
     * @NgModule({
     *   exports: [NgFor]
     * })
     * class CommonModule {
     * }
     * ```
     */
    exports?: Array<Type<any> | any[]>;
    /**
     * The set of components to compile when this NgModule is defined,
     * so that they can be dynamically loaded into the view.
     *
     * For each component listed here, Angular creates a `ComponentFactory`
     * and stores it in the `ComponentFactoryResolver`.
     *
     * Angular automatically adds components in the module's bootstrap
     * and route definitions into the `entryComponents` list. Use this
     * option to add components that are bootstrapped
     * using one of the imperative techniques, such as `ViewContainerRef.createComponent()`.
     *
     * @see [Entry Components](guide/entry-components)
     * @deprecated Since 9.0.0. With Ivy, this property is no longer necessary.
     */
    entryComponents?: Array<Type<any> | any[]>;
    /**
     * The set of components that are bootstrapped when
     * this module is bootstrapped. The components listed here
     * are automatically added to `entryComponents`.
     */
    bootstrap?: Array<Type<any> | any[]>;
    /**
     * The set of schemas that declare elements to be allowed in the NgModule.
     * Elements and properties that are neither Angular components nor directives
     * must be declared in a schema.
     *
     * Allowed value are `NO_ERRORS_SCHEMA` and `CUSTOM_ELEMENTS_SCHEMA`.
     *
     * @security When using one of `NO_ERRORS_SCHEMA` or `CUSTOM_ELEMENTS_SCHEMA`
     * you must ensure that allowed elements and properties securely escape inputs.
     */
    schemas?: Array<SchemaMetadata | any[]>;
    /**
     * A name or path that uniquely identifies this NgModule in `getModuleFactory`.
     * If left `undefined`, the NgModule is not registered with
     * `getModuleFactory`.
     */
    id?: string;
    /**
     * If true, this module will be skipped by the AOT compiler and so will always be compiled
     * using JIT.
     *
     * This exists to support future Ivy work and has no effect currently.
     */
    jit?: true;
}

/**
 * @Annotation
 * @publicApi
 */
export declare const NgModule: NgModuleDecorator;

/**
 * Type of the NgModule decorator / constructor function.
 *
 * @publicApi
 */
export declare interface NgModuleDecorator {
    /**
     * Decorator that marks a class as an NgModule and supplies configuration metadata.
     */
    (obj?: NgModule): TypeDecorator;
    new (obj?: NgModule): NgModule;
}

declare interface NgModuleDefinition extends Definition<NgModuleDefinitionFactory> {
    providers: NgModuleProviderDef[];
    providersByKey: {
        [tokenKey: string]: NgModuleProviderDef;
    };
    modules: any[];
    scope: 'root' | 'platform' | null;
}

declare interface NgModuleDefinitionFactory extends DefinitionFactory<NgModuleDefinition> {
}

/**
 * @publicApi
 */
export declare abstract class NgModuleFactory<T> {
    abstract get moduleType(): Type<T>;
    abstract create(parentInjector: Injector | null): NgModuleRef<T>;
}

/**
 * Used to load ng module factories.
 *
 * @publicApi
 * @deprecated the `string` form of `loadChildren` is deprecated, and `NgModuleFactoryLoader` is
 * part of its implementation. See `LoadChildren` for more details.
 */
export declare abstract class NgModuleFactoryLoader {
    abstract load(path: string): Promise<NgModuleFactory<any>>;
}

declare interface NgModuleProviderDef {
    flags: ɵNodeFlags;
    index: number;
    token: any;
    value: any;
    deps: DepDef[];
}

/**
 * Represents an instance of an NgModule created via a {@link NgModuleFactory}.
 *
 * `NgModuleRef` provides access to the NgModule Instance as well other objects related to this
 * NgModule Instance.
 *
 * @publicApi
 */
export declare abstract class NgModuleRef<T> {
    /**
     * The injector that contains all of the providers of the NgModule.
     */
    abstract get injector(): Injector;
    /**
     * The ComponentFactoryResolver to get hold of the ComponentFactories
     * declared in the `entryComponents` property of the module.
     */
    abstract get componentFactoryResolver(): ComponentFactoryResolver;
    /**
     * The NgModule instance.
     */
    abstract get instance(): T;
    /**
     * Destroys the module instance and all of the data structures associated with it.
     */
    abstract destroy(): void;
    /**
     * Allows to register a callback that will be called when the module is destroyed.
     */
    abstract onDestroy(callback: () => void): void;
}

/**
 * A token for third-party components that can register themselves with NgProbe.
 *
 * @publicApi
 */
export declare class NgProbeToken {
    name: string;
    token: any;
    constructor(name: string, token: any);
}

/**
 * An injectable service for executing work inside or outside of the Angular zone.
 *
 * The most common use of this service is to optimize performance when starting a work consisting of
 * one or more asynchronous tasks that don't require UI updates or error handling to be handled by
 * Angular. Such tasks can be kicked off via {@link #runOutsideAngular} and if needed, these tasks
 * can reenter the Angular zone via {@link #run}.
 *
 * <!-- TODO: add/fix links to:
 *   - docs explaining zones and the use of zones in Angular and change-detection
 *   - link to runOutsideAngular/run (throughout this file!)
 *   -->
 *
 * @usageNotes
 * ### Example
 *
 * ```
 * import {Component, NgZone} from '@angular/core';
 * import {NgIf} from '@angular/common';
 *
 * @Component({
 *   selector: 'ng-zone-demo',
 *   template: `
 *     <h2>Demo: NgZone</h2>
 *
 *     <p>Progress: {{progress}}%</p>
 *     <p *ngIf="progress >= 100">Done processing {{label}} of Angular zone!</p>
 *
 *     <button (click)="processWithinAngularZone()">Process within Angular zone</button>
 *     <button (click)="processOutsideOfAngularZone()">Process outside of Angular zone</button>
 *   `,
 * })
 * export class NgZoneDemo {
 *   progress: number = 0;
 *   label: string;
 *
 *   constructor(private _ngZone: NgZone) {}
 *
 *   // Loop inside the Angular zone
 *   // so the UI DOES refresh after each setTimeout cycle
 *   processWithinAngularZone() {
 *     this.label = 'inside';
 *     this.progress = 0;
 *     this._increaseProgress(() => console.log('Inside Done!'));
 *   }
 *
 *   // Loop outside of the Angular zone
 *   // so the UI DOES NOT refresh after each setTimeout cycle
 *   processOutsideOfAngularZone() {
 *     this.label = 'outside';
 *     this.progress = 0;
 *     this._ngZone.runOutsideAngular(() => {
 *       this._increaseProgress(() => {
 *         // reenter the Angular zone and display done
 *         this._ngZone.run(() => { console.log('Outside Done!'); });
 *       });
 *     });
 *   }
 *
 *   _increaseProgress(doneCallback: () => void) {
 *     this.progress += 1;
 *     console.log(`Current progress: ${this.progress}%`);
 *
 *     if (this.progress < 100) {
 *       window.setTimeout(() => this._increaseProgress(doneCallback), 10);
 *     } else {
 *       doneCallback();
 *     }
 *   }
 * }
 * ```
 *
 * @publicApi
 */
export declare class NgZone {
    readonly hasPendingMacrotasks: boolean;
    readonly hasPendingMicrotasks: boolean;
    /**
     * Whether there are no outstanding microtasks or macrotasks.
     */
    readonly isStable: boolean;
    /**
     * Notifies when code enters Angular Zone. This gets fired first on VM Turn.
     */
    readonly onUnstable: EventEmitter<any>;
    /**
     * Notifies when there is no more microtasks enqueued in the current VM Turn.
     * This is a hint for Angular to do change detection, which may enqueue more microtasks.
     * For this reason this event can fire multiple times per VM Turn.
     */
    readonly onMicrotaskEmpty: EventEmitter<any>;
    /**
     * Notifies when the last `onMicrotaskEmpty` has run and there are no more microtasks, which
     * implies we are about to relinquish VM turn.
     * This event gets called just once.
     */
    readonly onStable: EventEmitter<any>;
    /**
     * Notifies that an error has been delivered.
     */
    readonly onError: EventEmitter<any>;
    constructor({ enableLongStackTrace, shouldCoalesceEventChangeDetection }: {
        enableLongStackTrace?: boolean | undefined;
        shouldCoalesceEventChangeDetection?: boolean | undefined;
    });
    static isInAngularZone(): boolean;
    static assertInAngularZone(): void;
    static assertNotInAngularZone(): void;
    /**
     * Executes the `fn` function synchronously within the Angular zone and returns value returned by
     * the function.
     *
     * Running functions via `run` allows you to reenter Angular zone from a task that was executed
     * outside of the Angular zone (typically started via {@link #runOutsideAngular}).
     *
     * Any future tasks or microtasks scheduled from within this function will continue executing from
     * within the Angular zone.
     *
     * If a synchronous error happens it will be rethrown and not reported via `onError`.
     */
    run<T>(fn: (...args: any[]) => T, applyThis?: any, applyArgs?: any[]): T;
    /**
     * Executes the `fn` function synchronously within the Angular zone as a task and returns value
     * returned by the function.
     *
     * Running functions via `run` allows you to reenter Angular zone from a task that was executed
     * outside of the Angular zone (typically started via {@link #runOutsideAngular}).
     *
     * Any future tasks or microtasks scheduled from within this function will continue executing from
     * within the Angular zone.
     *
     * If a synchronous error happens it will be rethrown and not reported via `onError`.
     */
    runTask<T>(fn: (...args: any[]) => T, applyThis?: any, applyArgs?: any[], name?: string): T;
    /**
     * Same as `run`, except that synchronous errors are caught and forwarded via `onError` and not
     * rethrown.
     */
    runGuarded<T>(fn: (...args: any[]) => T, applyThis?: any, applyArgs?: any[]): T;
    /**
     * Executes the `fn` function synchronously in Angular's parent zone and returns value returned by
     * the function.
     *
     * Running functions via {@link #runOutsideAngular} allows you to escape Angular's zone and do
     * work that
     * doesn't trigger Angular change-detection or is subject to Angular's error handling.
     *
     * Any future tasks or microtasks scheduled from within this function will continue executing from
     * outside of the Angular zone.
     *
     * Use {@link #run} to reenter the Angular zone and do work that updates the application model.
     */
    runOutsideAngular<T>(fn: (...args: any[]) => T): T;
}

/**
 * Defines a schema that allows any property on any element.
 *
 * @publicApi
 */
export declare const NO_ERRORS_SCHEMA: SchemaMetadata;

declare interface NodeCheckFn {
    (view: ViewData, nodeIndex: number, argStyle: ɵArgumentType.Dynamic, values: any[]): any;
    (view: ViewData, nodeIndex: number, argStyle: ɵArgumentType.Inline, v0?: any, v1?: any, v2?: any, v3?: any, v4?: any, v5?: any, v6?: any, v7?: any, v8?: any, v9?: any): any;
}

/**
 * Node instance data.
 *
 * We have a separate type per NodeType to save memory
 * (TextData | ElementData | ProviderData | PureExpressionData | QueryList<any>)
 *
 * To keep our code monomorphic,
 * we prohibit using `NodeData` directly but enforce the use of accessors (`asElementData`, ...).
 * This way, no usage site can get a `NodeData` from view.nodes and then use it for different
 * purposes.
 */
declare class NodeData {
    private __brand;
}

/**
 * A node definition in the view.
 *
 * Note: We use one type for all nodes so that loops that loop over all nodes
 * of a ViewDefinition stay monomorphic!
 */
declare interface NodeDef {
    flags: ɵNodeFlags;
    nodeIndex: number;
    checkIndex: number;
    parent: NodeDef | null;
    renderParent: NodeDef | null;
    /** this is checked against NgContentDef.index to find matched nodes */
    ngContentIndex: number | null;
    /** number of transitive children */
    childCount: number;
    /** aggregated NodeFlags for all transitive children (does not include self) **/
    childFlags: ɵNodeFlags;
    /** aggregated NodeFlags for all direct children (does not include self) **/
    directChildFlags: ɵNodeFlags;
    bindingIndex: number;
    bindings: BindingDef[];
    bindingFlags: ɵBindingFlags;
    outputIndex: number;
    outputs: OutputDef[];
    /**
     * references that the user placed on the element
     */
    references: {
        [refId: string]: ɵQueryValueType;
    };
    /**
     * ids and value types of all queries that are matched by this node.
     */
    matchedQueries: {
        [queryId: number]: ɵQueryValueType;
    };
    /** Binary or of all matched query ids of this node. */
    matchedQueryIds: number;
    /**
     * Binary or of all query ids that are matched by one of the children.
     * This includes query ids from templates as well.
     * Used as a bloom filter.
     */
    childMatchedQueries: number;
    element: ElementDef | null;
    provider: ProviderDef | null;
    text: TextDef | null;
    query: QueryDef | null;
    ngContent: NgContentDef | null;
}

/**
 * Function to call console.error at the right source location. This is an indirection
 * via another function as browser will log the location that actually called
 * `console.error`.
 */
declare interface NodeLogger {
    (): () => void;
}

/**
 * Object Oriented style of API needed to create elements and text nodes.
 *
 * This is the native browser API style, e.g. operations are methods on individual objects
 * like HTMLElement. With this style, no additional code is needed as a facade
 * (reducing payload size).
 * */
declare interface ObjectOrientedRenderer3 {
    createComment(data: string): RComment;
    createElement(tagName: string): RElement;
    createElementNS(namespace: string, tagName: string): RElement;
    createTextNode(data: string): RText;
    querySelector(selectors: string): RElement | null;
}

/**
 * @description
 * A lifecycle hook that is called when any data-bound property of a directive changes.
 * Define an `ngOnChanges()` method to handle the changes.
 *
 * @see `DoCheck`
 * @see `OnInit`
 * @see [Lifecycle Hooks](guide/lifecycle-hooks#onchanges) guide
 *
 * @usageNotes
 * The following snippet shows how a component can implement this interface to
 * define an on-changes handler for an input property.
 *
 * {@example core/ts/metadata/lifecycle_hooks_spec.ts region='OnChanges'}
 *
 * @publicApi
 */
export declare interface OnChanges {
    /**
     * A callback method that is invoked immediately after the
     * default change detector has checked data-bound properties
     * if at least one has changed, and before the view and content
     * children are checked.
     * @param changes The changed properties.
     */
    ngOnChanges(changes: SimpleChanges): void;
}

/**
 * A lifecycle hook that is called when a directive, pipe, or service is destroyed.
 * Use for any custom cleanup that needs to occur when the
 * instance is destroyed.
 * @see [Lifecycle Hooks](guide/lifecycle-hooks#onchanges) guide
 *
 * @usageNotes
 * The following snippet shows how a component can implement this interface
 * to define its own custom clean-up method.
 *
 * {@example core/ts/metadata/lifecycle_hooks_spec.ts region='OnDestroy'}
 *
 * @publicApi
 */
export declare interface OnDestroy {
    /**
     * A callback method that performs custom clean-up, invoked immediately
     * before a directive, pipe, or service instance is destroyed.
     */
    ngOnDestroy(): void;
}

/**
 * @description
 * A lifecycle hook that is called after Angular has initialized
 * all data-bound properties of a directive.
 * Define an `ngOnInit()` method to handle any additional initialization tasks.
 *
 * @see `AfterContentInit`
 * @see [Lifecycle Hooks](guide/lifecycle-hooks#onchanges) guide
 *
 * @usageNotes
 * The following snippet shows how a component can implement this interface to
 * define its own initialization method.
 *
 * {@example core/ts/metadata/lifecycle_hooks_spec.ts region='OnInit'}
 *
 * @publicApi
 */
export declare interface OnInit {
    /**
     * A callback method that is invoked immediately after the
     * default change detector has checked the directive's
     * data-bound properties for the first time,
     * and before any of the view or content children have been checked.
     * It is invoked only once when the directive is instantiated.
     */
    ngOnInit(): void;
}

declare interface OpaqueViewState {
    '__brand__': 'Brand for OpaqueViewState that nothing will match';
}

/**
 * Type of the Optional metadata.
 *
 * @publicApi
 */
export declare interface Optional {
}

/**
 * Optional decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const Optional: OptionalDecorator;

/**
 * Type of the Optional decorator / constructor function.
 *
 * @publicApi
 */
export declare interface OptionalDecorator {
    /**
     * Parameter decorator to be used on constructor parameters,
     * which marks the parameter as being an optional dependency.
     * The DI framework provides null if the dependency is not found.
     *
     * Can be used together with other parameter decorators
     * that modify how dependency injection operates.
     *
     * Learn more in the ["Dependency Injection Guide"](guide/dependency-injection).
     *
     * @usageNotes
     *
     * The following code allows the possibility of a null result:
     *
     * <code-example path="core/di/ts/metadata_spec.ts" region="Optional">
     * </code-example>
     *
     */
    (): any;
    new (): Optional;
}

/**
 * Type of the Output metadata.
 *
 * @publicApi
 */
export declare interface Output {
    /**
    * The name of the DOM property to which the output property is bound.
    */
    bindingPropertyName?: string;
}

/**
 * @Annotation
 * @publicApi
 */
export declare const Output: OutputDecorator;

/**
 * Type of the Output decorator / constructor function.
 *
 * @publicApi
 */
export declare interface OutputDecorator {
    /**
    * Decorator that marks a class field as an output property and supplies configuration metadata.
    * The DOM property bound to the output property is automatically updated during change detection.
    *
    * @usageNotes
    *
    * You can supply an optional name to use in templates when the
    * component is instantiated, that maps to the
    * name of the bound property. By default, the original
    * name of the bound property is used for output binding.
    *
    * See `Input` decorator for an example of providing a binding name.
    *
    * @see [Input and Output properties](guide/template-syntax#input-and-output-properties)
    *
    */
    (bindingPropertyName?: string): any;
    new (bindingPropertyName?: string): any;
}

declare interface OutputDef {
    type: OutputType;
    target: 'window' | 'document' | 'body' | 'component' | null;
    eventName: string;
    propName: string | null;
}

declare const enum OutputType {
    ElementOutput = 0,
    DirectiveOutput = 1
}

/**
 * A token which indicates the root directory of the application
 * @publicApi
 */
export declare const PACKAGE_ROOT_URL: InjectionToken<string>;

declare const PARENT = 3;

/**
 * Type of the Pipe metadata.
 *
 * @publicApi
 */
export declare interface Pipe {
    /**
     * The pipe name to use in template bindings.
     * Typically uses [lowerCamelCase](guide/glossary#case-types)
     * because the name cannot contain hyphens.
     */
    name: string;
    /**
     * When true, the pipe is pure, meaning that the
     * `transform()` method is invoked only when its input arguments
     * change. Pipes are pure by default.
     *
     * If the pipe has internal state (that is, the result
     * depends on state other than its arguments), set `pure` to false.
     * In this case, the pipe is invoked on each change-detection cycle,
     * even if the arguments have not changed.
     */
    pure?: boolean;
}

/**
 * @Annotation
 * @publicApi
 */
export declare const Pipe: PipeDecorator;

/**
 * Type of the Pipe decorator / constructor function.
 *
 * @publicApi
 */
export declare interface PipeDecorator {
    /**
     *
     * Decorator that marks a class as pipe and supplies configuration metadata.
     *
     * A pipe class must implement the `PipeTransform` interface.
     * For example, if the name is "myPipe", use a template binding expression
     * such as the following:
     *
     * ```
     * {{ exp | myPipe }}
     * ```
     *
     * The result of the expression is passed to the pipe's `transform()` method.
     *
     * A pipe must belong to an NgModule in order for it to be available
     * to a template. To make it a member of an NgModule,
     * list it in the `declarations` field of the `NgModule` metadata.
     *
     * @see [Style Guide: Pipe Names](guide/styleguide#02-09)
     *
     */
    (obj: Pipe): TypeDecorator;
    /**
     * See the `Pipe` decorator.
     */
    new (obj: Pipe): Pipe;
}

declare type PipeDefList = ɵPipeDef<any>[];

/**
 * Type used for PipeDefs on component definition.
 *
 * The function is necessary to be able to support forward declarations.
 */
declare type PipeDefListOrFactory = (() => PipeDefList) | PipeDefList;


/**
 * An interface that is implemented by pipes in order to perform a transformation.
 * Angular invokes the `transform` method with the value of a binding
 * as the first argument, and any parameters as the second argument in list form.
 *
 * @usageNotes
 *
 * In the following example, `RepeatPipe` repeats a given value a given number of times.
 *
 * ```ts
 * import {Pipe, PipeTransform} from '@angular/core';
 *
 * @Pipe({name: 'repeat'})
 * export class RepeatPipe implements PipeTransform {
 *   transform(value: any, times: number) {
 *     return value.repeat(times);
 *   }
 * }
 * ```
 *
 * Invoking `{{ 'ok' | repeat:3 }}` in a template produces `okokok`.
 *
 * @publicApi
 */
export declare interface PipeTransform {
    transform(value: any, ...args: any[]): any;
}

/**
 * A subclass of `Type` which has a static `ɵpipe`:`PipeDef` field making it
 * consumable for rendering.
 */
declare interface PipeType<T> extends Type<T> {
    ɵpipe: never;
}

declare type PipeTypeList = (PipeType<any> | Type<any>)[];

declare type PipeTypesOrFactory = (() => PipeTypeList) | PipeTypeList;

/**
 * A token that indicates an opaque platform id.
 * @publicApi
 */
export declare const PLATFORM_ID: InjectionToken<Object>;

/**
 * A function that will be executed when a platform is initialized.
 * @publicApi
 */
export declare const PLATFORM_INITIALIZER: InjectionToken<(() => void)[]>;

/**
 * This platform has to be included in any other platform
 *
 * @publicApi
 */
export declare const platformCore: (extraProviders?: StaticProvider[] | undefined) => PlatformRef;

/**
 * The Angular platform is the entry point for Angular on a web page. Each page
 * has exactly one platform, and services (such as reflection) which are common
 * to every Angular application running on the page are bound in its scope.
 *
 * A page's platform is initialized implicitly when a platform is created via a platform factory
 * (e.g. {@link platformBrowser}), or explicitly by calling the {@link createPlatform} function.
 *
 * @publicApi
 */
export declare class PlatformRef {
    private _injector;
    private _modules;
    private _destroyListeners;
    private _destroyed;
    /**
     * Creates an instance of an `@NgModule` for the given platform
     * for offline compilation.
     *
     * @usageNotes
     * ### Simple Example
     *
     * ```typescript
     * my_module.ts:
     *
     * @NgModule({
     *   imports: [BrowserModule]
     * })
     * class MyModule {}
     *
     * main.ts:
     * import {MyModuleNgFactory} from './my_module.ngfactory';
     * import {platformBrowser} from '@angular/platform-browser';
     *
     * let moduleRef = platformBrowser().bootstrapModuleFactory(MyModuleNgFactory);
     * ```
     */
    bootstrapModuleFactory<M>(moduleFactory: NgModuleFactory<M>, options?: BootstrapOptions): Promise<NgModuleRef<M>>;
    /**
     * Creates an instance of an `@NgModule` for a given platform using the given runtime compiler.
     *
     * @usageNotes
     * ### Simple Example
     *
     * ```typescript
     * @NgModule({
     *   imports: [BrowserModule]
     * })
     * class MyModule {}
     *
     * let moduleRef = platformBrowser().bootstrapModule(MyModule);
     * ```
     *
     */
    bootstrapModule<M>(moduleType: Type<M>, compilerOptions?: (CompilerOptions & BootstrapOptions) | Array<CompilerOptions & BootstrapOptions>): Promise<NgModuleRef<M>>;
    private _moduleDoBootstrap;
    /**
     * Register a listener to be called when the platform is disposed.
     */
    onDestroy(callback: () => void): void;
    /**
     * Retrieve the platform {@link Injector}, which is the parent injector for
     * every Angular application on the page and provides singleton providers.
     */
    get injector(): Injector;
    /**
     * Destroy the Angular platform and all Angular applications on the page.
     */
    destroy(): void;
    get destroyed(): boolean;
    static ɵfac: ɵngcc0.ɵɵFactoryDef<PlatformRef>;
    static ɵprov: ɵngcc0.ɵɵInjectableDef<PlatformRef>;
}

declare interface PlatformReflectionCapabilities {
    isReflectionEnabled(): boolean;
    factory(type: Type<any>): Function;
    hasLifecycleHook(type: any, lcProperty: string): boolean;
    guards(type: any): {
        [key: string]: any;
    };
    /**
     * Return a list of annotations/types for constructor parameters
     */
    parameters(type: Type<any>): any[][];
    /**
     * Return a list of annotations declared on the class
     */
    annotations(type: Type<any>): any[];
    /**
     * Return a object literal which describes the annotations on Class fields/properties.
     */
    propMetadata(typeOrFunc: Type<any>): {
        [key: string]: any[];
    };
    getter(name: string): ɵGetterFn;
    setter(name: string): ɵSetterFn;
    method(name: string): ɵMethodFn;
    importUri(type: Type<any>): string;
    resourceUri(type: Type<any>): string;
    resolveIdentifier(name: string, moduleUrl: string, members: string[], runtime: any): any;
    resolveEnum(enumIdentifier: any, name: string): any;
}

/**
 * A boolean-valued function over a value, possibly including context information
 * regarding that value's position in an array.
 *
 * @publicApi
 */
export declare interface Predicate<T> {
    (value: T): boolean;
}

declare const PREORDER_HOOK_FLAGS = 18;

/** More flags associated with an LView (saved in LView[PREORDER_HOOK_FLAGS]) */
declare const enum PreOrderHookFlags {
    /** The index of the next pre-order hook to be called in the hooks array, on the first 16
       bits */
    IndexOfTheNextPreOrderHookMaskMask = 65535,
    /**
     * The number of init hooks that have already been called, on the last 16 bits
     */
    NumberOfInitHooksCalledIncrementer = 65536,
    NumberOfInitHooksCalledShift = 16,
    NumberOfInitHooksCalledMask = 4294901760
}

/**
 * Procedural style of API needed to create elements and text nodes.
 *
 * In non-native browser environments (e.g. platforms such as web-workers), this is the
 * facade that enables element manipulation. This also facilitates backwards compatibility
 * with Renderer2.
 */
declare interface ProceduralRenderer3 {
    destroy(): void;
    createComment(value: string): RComment;
    createElement(name: string, namespace?: string | null): RElement;
    createText(value: string): RText;
    /**
     * This property is allowed to be null / undefined,
     * in which case the view engine won't call it.
     * This is used as a performance optimization for production mode.
     */
    destroyNode?: ((node: RNode) => void) | null;
    appendChild(parent: RElement, newChild: RNode): void;
    insertBefore(parent: RNode, newChild: RNode, refChild: RNode | null): void;
    removeChild(parent: RElement, oldChild: RNode, isHostElement?: boolean): void;
    selectRootElement(selectorOrNode: string | any, preserveContent?: boolean): RElement;
    parentNode(node: RNode): RElement | null;
    nextSibling(node: RNode): RNode | null;
    setAttribute(el: RElement, name: string, value: string, namespace?: string | null): void;
    removeAttribute(el: RElement, name: string, namespace?: string | null): void;
    addClass(el: RElement, name: string): void;
    removeClass(el: RElement, name: string): void;
    setStyle(el: RElement, style: string, value: any, flags?: RendererStyleFlags2 | RendererStyleFlags3): void;
    removeStyle(el: RElement, style: string, flags?: RendererStyleFlags2 | RendererStyleFlags3): void;
    setProperty(el: RElement, name: string, value: any): void;
    setValue(node: RText | RComment, value: string): void;
    listen(target: GlobalTargetName | RNode, eventName: string, callback: (event: any) => boolean | void): () => void;
}

/**
 * Describes a function that is used to process provider lists (such as provider
 * overrides).
 */
declare type ProcessProvidersFunction = (providers: Provider[]) => Provider[];

/**
 * List of slots for a projection. A slot can be either based on a parsed CSS selector
 * which will be used to determine nodes which are projected into that slot.
 *
 * When set to "*", the slot is reserved and can be used for multi-slot projection
 * using {@link ViewContainerRef#createComponent}. The last slot that specifies the
 * wildcard selector will retrieve all projectable nodes which do not match any selector.
 */
declare type ProjectionSlots = (ɵCssSelectorList | '*')[];

/**
 * This mapping is necessary so we can set input properties and output listeners
 * properly at runtime when property names are minified or aliased.
 *
 * Key: unminified / public input or output name
 * Value: array containing minified / internal name and related directive index
 *
 * The value must be an array to support inputs and outputs with the same name
 * on the same node.
 */
declare type PropertyAliases = {
    [key: string]: PropertyAliasValue;
};

/**
 * Store the runtime input or output names for all the directives.
 *
 * i+0: directive instance index
 * i+1: privateName
 *
 * e.g. [0, 'change-minified']
 */
declare type PropertyAliasValue = (number | string)[];

/**
 * Describes how the `Injector` should be configured.
 * @see ["Dependency Injection Guide"](guide/dependency-injection).
 *
 * @see `StaticProvider`
 *
 * @publicApi
 */
export declare type Provider = TypeProvider | ValueProvider | ClassProvider | ConstructorProvider | ExistingProvider | FactoryProvider | any[];

declare interface ProviderDef {
    token: any;
    value: any;
    deps: DepDef[];
}

declare interface ProviderOverride {
    token: any;
    flags: ɵNodeFlags;
    value: any;
    deps: ([ɵDepFlags, any] | any)[];
    deprecatedBehavior: boolean;
}

/**
 * Testability API.
 * `declare` keyword causes tsickle to generate externs, so these methods are
 * not renamed by Closure Compiler.
 * @publicApi
 */
declare interface PublicTestability {
    isStable(): boolean;
    whenStable(callback: Function, timeout?: number, updateCallback?: Function): void;
    findProviders(using: any, provider: string, exactMatch: boolean): any[];
}

declare const QUERIES = 5;

/**
 * Type of the Query metadata.
 *
 * @publicApi
 */
export declare interface Query {
    descendants: boolean;
    first: boolean;
    read: any;
    isViewQuery: boolean;
    selector: any;
    static?: boolean;
}

/**
 * Base class for query metadata.
 *
 * @see `ContentChildren`.
 * @see `ContentChild`.
 * @see `ViewChildren`.
 * @see `ViewChild`.
 *
 * @publicApi
 */
export declare abstract class Query {
}

declare interface QueryBindingDef {
    propName: string;
    bindingType: ɵQueryBindingType;
}

declare interface QueryDef {
    id: number;
    filterId: number;
    bindings: QueryBindingDef[];
}

/**
 * An unmodifiable list of items that Angular keeps up to date when the state
 * of the application changes.
 *
 * The type of object that {@link ViewChildren}, {@link ContentChildren}, and {@link QueryList}
 * provide.
 *
 * Implements an iterable interface, therefore it can be used in both ES6
 * javascript `for (var i of items)` loops as well as in Angular templates with
 * `*ngFor="let i of myList"`.
 *
 * Changes can be observed by subscribing to the changes `Observable`.
 *
 * NOTE: In the future this class will implement an `Observable` interface.
 *
 * @usageNotes
 * ### Example
 * ```typescript
 * @Component({...})
 * class Container {
 *   @ViewChildren(Item) items:QueryList<Item>;
 * }
 * ```
 *
 * @publicApi
 */
export declare class QueryList<T> implements Iterable<T> {
    readonly dirty = true;
    private _results;
    readonly changes: Observable<any>;
    readonly length: number;
    readonly first: T;
    readonly last: T;
    constructor();
    /**
     * See
     * [Array.map](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/map)
     */
    map<U>(fn: (item: T, index: number, array: T[]) => U): U[];
    /**
     * See
     * [Array.filter](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/filter)
     */
    filter(fn: (item: T, index: number, array: T[]) => boolean): T[];
    /**
     * See
     * [Array.find](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/find)
     */
    find(fn: (item: T, index: number, array: T[]) => boolean): T | undefined;
    /**
     * See
     * [Array.reduce](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/reduce)
     */
    reduce<U>(fn: (prevValue: U, curValue: T, curIndex: number, array: T[]) => U, init: U): U;
    /**
     * See
     * [Array.forEach](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/forEach)
     */
    forEach(fn: (item: T, index: number, array: T[]) => void): void;
    /**
     * See
     * [Array.some](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/some)
     */
    some(fn: (value: T, index: number, array: T[]) => boolean): boolean;
    /**
     * Returns a copy of the internal results list as an Array.
     */
    toArray(): T[];
    toString(): string;
    /**
     * Updates the stored data of the query list, and resets the `dirty` flag to `false`, so that
     * on change detection, it will not notify of changes to the queries, unless a new change
     * occurs.
     *
     * @param resultsTree The query results to store
     */
    reset(resultsTree: Array<T | any[]>): void;
    /**
     * Triggers a change event by emitting on the `changes` {@link EventEmitter}.
     */
    notifyOnChanges(): void;
    /** internal */
    setDirty(): void;
    /** internal */
    destroy(): void;
    [Symbol.iterator]: () => Iterator<T>;
}

declare class R3Injector {
    readonly parent: Injector;
    /**
     * Map of tokens to records which contain the instances of those tokens.
     * - `null` value implies that we don't have the record. Used by tree-shakable injectors
     * to prevent further searches.
     */
    private records;
    /**
     * The transitive set of `InjectorType`s which define this injector.
     */
    private injectorDefTypes;
    /**
     * Set of values instantiated by this injector which contain `ngOnDestroy` lifecycle hooks.
     */
    private onDestroy;
    /**
     * Flag indicating this injector provides the APP_ROOT_SCOPE token, and thus counts as the
     * root scope.
     */
    private readonly scope;
    readonly source: string | null;
    /**
     * Flag indicating that this injector was previously destroyed.
     */
    get destroyed(): boolean;
    private _destroyed;
    constructor(def: InjectorType<any>, additionalProviders: StaticProvider[] | null, parent: Injector, source?: string | null);
    /**
     * Destroy the injector and release references to every instance or provider associated with it.
     *
     * Also calls the `OnDestroy` lifecycle hooks of every instance that was created for which a
     * hook was found.
     */
    destroy(): void;
    get<T>(token: Type<T> | InjectionToken<T>, notFoundValue?: any, flags?: InjectFlags): T;
    toString(): string;
    private assertNotDestroyed;
    /**
     * Add an `InjectorType` or `InjectorTypeWithProviders` and all of its transitive providers
     * to this injector.
     *
     * If an `InjectorTypeWithProviders` that declares providers besides the type is specified,
     * the function will return "true" to indicate that the providers of the type definition need
     * to be processed. This allows us to process providers of injector types after all imports of
     * an injector definition are processed. (following View Engine semantics: see FW-1349)
     */
    private processInjectorType;
    /**
     * Process a `SingleProvider` and add it.
     */
    private processProvider;
    private hydrate;
    private injectableDefInScope;
}

declare interface RComment extends RNode {
    textContent: string | null;
}

declare interface RCssStyleDeclaration {
    removeProperty(propertyName: string): string;
    setProperty(propertyName: string, value: string | null, priority?: string): void;
}

declare interface RDomTokenList {
    add(token: string): void;
    remove(token: string): void;
}

/**
 * A ReflectiveDependency injection container used for instantiating objects and resolving
 * dependencies.
 *
 * An `Injector` is a replacement for a `new` operator, which can automatically resolve the
 * constructor dependencies.
 *
 * In typical use, application code asks for the dependencies in the constructor and they are
 * resolved by the `Injector`.
 *
 * @usageNotes
 * ### Example
 *
 * The following example creates an `Injector` configured to create `Engine` and `Car`.
 *
 * ```typescript
 * @Injectable()
 * class Engine {
 * }
 *
 * @Injectable()
 * class Car {
 *   constructor(public engine:Engine) {}
 * }
 *
 * var injector = ReflectiveInjector.resolveAndCreate([Car, Engine]);
 * var car = injector.get(Car);
 * expect(car instanceof Car).toBe(true);
 * expect(car.engine instanceof Engine).toBe(true);
 * ```
 *
 * Notice, we don't use the `new` operator because we explicitly want to have the `Injector`
 * resolve all of the object's dependencies automatically.
 *
 * @deprecated from v5 - slow and brings in a lot of code, Use `Injector.create` instead.
 * @publicApi
 */
export declare abstract class ReflectiveInjector implements Injector {
    /**
     * Turns an array of provider definitions into an array of resolved providers.
     *
     * A resolution is a process of flattening multiple nested arrays and converting individual
     * providers into an array of `ResolvedReflectiveProvider`s.
     *
     * @usageNotes
     * ### Example
     *
     * ```typescript
     * @Injectable()
     * class Engine {
     * }
     *
     * @Injectable()
     * class Car {
     *   constructor(public engine:Engine) {}
     * }
     *
     * var providers = ReflectiveInjector.resolve([Car, [[Engine]]]);
     *
     * expect(providers.length).toEqual(2);
     *
     * expect(providers[0] instanceof ResolvedReflectiveProvider).toBe(true);
     * expect(providers[0].key.displayName).toBe("Car");
     * expect(providers[0].dependencies.length).toEqual(1);
     * expect(providers[0].factory).toBeDefined();
     *
     * expect(providers[1].key.displayName).toBe("Engine");
     * });
     * ```
     *
     */
    static resolve(providers: Provider[]): ResolvedReflectiveProvider[];
    /**
     * Resolves an array of providers and creates an injector from those providers.
     *
     * The passed-in providers can be an array of `Type`, `Provider`,
     * or a recursive array of more providers.
     *
     * @usageNotes
     * ### Example
     *
     * ```typescript
     * @Injectable()
     * class Engine {
     * }
     *
     * @Injectable()
     * class Car {
     *   constructor(public engine:Engine) {}
     * }
     *
     * var injector = ReflectiveInjector.resolveAndCreate([Car, Engine]);
     * expect(injector.get(Car) instanceof Car).toBe(true);
     * ```
     */
    static resolveAndCreate(providers: Provider[], parent?: Injector): ReflectiveInjector;
    /**
     * Creates an injector from previously resolved providers.
     *
     * This API is the recommended way to construct injectors in performance-sensitive parts.
     *
     * @usageNotes
     * ### Example
     *
     * ```typescript
     * @Injectable()
     * class Engine {
     * }
     *
     * @Injectable()
     * class Car {
     *   constructor(public engine:Engine) {}
     * }
     *
     * var providers = ReflectiveInjector.resolve([Car, Engine]);
     * var injector = ReflectiveInjector.fromResolvedProviders(providers);
     * expect(injector.get(Car) instanceof Car).toBe(true);
     * ```
     */
    static fromResolvedProviders(providers: ResolvedReflectiveProvider[], parent?: Injector): ReflectiveInjector;
    /**
     * Parent of this injector.
     *
     * <!-- TODO: Add a link to the section of the user guide talking about hierarchical injection.
     * -->
     */
    abstract get parent(): Injector | null;
    /**
     * Resolves an array of providers and creates a child injector from those providers.
     *
     * <!-- TODO: Add a link to the section of the user guide talking about hierarchical injection.
     * -->
     *
     * The passed-in providers can be an array of `Type`, `Provider`,
     * or a recursive array of more providers.
     *
     * @usageNotes
     * ### Example
     *
     * ```typescript
     * class ParentProvider {}
     * class ChildProvider {}
     *
     * var parent = ReflectiveInjector.resolveAndCreate([ParentProvider]);
     * var child = parent.resolveAndCreateChild([ChildProvider]);
     *
     * expect(child.get(ParentProvider) instanceof ParentProvider).toBe(true);
     * expect(child.get(ChildProvider) instanceof ChildProvider).toBe(true);
     * expect(child.get(ParentProvider)).toBe(parent.get(ParentProvider));
     * ```
     */
    abstract resolveAndCreateChild(providers: Provider[]): ReflectiveInjector;
    /**
     * Creates a child injector from previously resolved providers.
     *
     * <!-- TODO: Add a link to the section of the user guide talking about hierarchical injection.
     * -->
     *
     * This API is the recommended way to construct injectors in performance-sensitive parts.
     *
     * @usageNotes
     * ### Example
     *
     * ```typescript
     * class ParentProvider {}
     * class ChildProvider {}
     *
     * var parentProviders = ReflectiveInjector.resolve([ParentProvider]);
     * var childProviders = ReflectiveInjector.resolve([ChildProvider]);
     *
     * var parent = ReflectiveInjector.fromResolvedProviders(parentProviders);
     * var child = parent.createChildFromResolved(childProviders);
     *
     * expect(child.get(ParentProvider) instanceof ParentProvider).toBe(true);
     * expect(child.get(ChildProvider) instanceof ChildProvider).toBe(true);
     * expect(child.get(ParentProvider)).toBe(parent.get(ParentProvider));
     * ```
     */
    abstract createChildFromResolved(providers: ResolvedReflectiveProvider[]): ReflectiveInjector;
    /**
     * Resolves a provider and instantiates an object in the context of the injector.
     *
     * The created object does not get cached by the injector.
     *
     * @usageNotes
     * ### Example
     *
     * ```typescript
     * @Injectable()
     * class Engine {
     * }
     *
     * @Injectable()
     * class Car {
     *   constructor(public engine:Engine) {}
     * }
     *
     * var injector = ReflectiveInjector.resolveAndCreate([Engine]);
     *
     * var car = injector.resolveAndInstantiate(Car);
     * expect(car.engine).toBe(injector.get(Engine));
     * expect(car).not.toBe(injector.resolveAndInstantiate(Car));
     * ```
     */
    abstract resolveAndInstantiate(provider: Provider): any;
    /**
     * Instantiates an object using a resolved provider in the context of the injector.
     *
     * The created object does not get cached by the injector.
     *
     * @usageNotes
     * ### Example
     *
     * ```typescript
     * @Injectable()
     * class Engine {
     * }
     *
     * @Injectable()
     * class Car {
     *   constructor(public engine:Engine) {}
     * }
     *
     * var injector = ReflectiveInjector.resolveAndCreate([Engine]);
     * var carProvider = ReflectiveInjector.resolve([Car])[0];
     * var car = injector.instantiateResolved(carProvider);
     * expect(car.engine).toBe(injector.get(Engine));
     * expect(car).not.toBe(injector.instantiateResolved(carProvider));
     * ```
     */
    abstract instantiateResolved(provider: ResolvedReflectiveProvider): any;
    abstract get(token: any, notFoundValue?: any): any;
}


/**
 * A unique object used for retrieving items from the {@link ReflectiveInjector}.
 *
 * Keys have:
 * - a system-wide unique `id`.
 * - a `token`.
 *
 * `Key` is used internally by {@link ReflectiveInjector} because its system-wide unique `id` allows
 * the
 * injector to store created objects in a more efficient way.
 *
 * `Key` should not be created directly. {@link ReflectiveInjector} creates keys automatically when
 * resolving
 * providers.
 *
 * @deprecated No replacement
 * @publicApi
 */
export declare class ReflectiveKey {
    token: Object;
    id: number;
    readonly displayName: string;
    /**
     * Private
     */
    constructor(token: Object, id: number);
    /**
     * Retrieves a `Key` for a token.
     */
    static get(token: Object): ReflectiveKey;
    /**
     * @returns the number of keys registered in the system.
     */
    static get numberOfKeys(): number;
}

/**
 * Subset of API needed for writing attributes, properties, and setting up
 * listeners on Element.
 */
declare interface RElement extends RNode {
    style: RCssStyleDeclaration;
    classList: RDomTokenList;
    className: string;
    textContent: string | null;
    setAttribute(name: string, value: string): void;
    removeAttribute(name: string): void;
    setAttributeNS(namespaceURI: string, qualifiedName: string, value: string): void;
    addEventListener(type: string, listener: EventListener, useCapture?: boolean): void;
    removeEventListener(type: string, listener?: EventListener, options?: boolean): void;
    setProperty?(name: string, value: any): void;
}

declare const RENDERER = 11;

/**
 * Extend this base class to implement custom rendering. By default, Angular
 * renders a template into DOM. You can use custom rendering to intercept
 * rendering calls, or to render to something other than DOM.
 *
 * Create your custom renderer using `RendererFactory2`.
 *
 * Use a custom renderer to bypass Angular's templating and
 * make custom UI changes that can't be expressed declaratively.
 * For example if you need to set a property or an attribute whose name is
 * not statically known, use the `setProperty()` or
 * `setAttribute()` method.
 *
 * @publicApi
 */
export declare abstract class Renderer2 {
    /**
     * Use to store arbitrary developer-defined data on a renderer instance,
     * as an object containing key-value pairs.
     * This is useful for renderers that delegate to other renderers.
     */
    abstract get data(): {
        [key: string]: any;
    };
    /**
     * Implement this callback to destroy the renderer or the host element.
     */
    abstract destroy(): void;
    /**
     * Implement this callback to create an instance of the host element.
     * @param name An identifying name for the new element, unique within the namespace.
     * @param namespace The namespace for the new element.
     * @returns The new element.
     */
    abstract createElement(name: string, namespace?: string | null): any;
    /**
     * Implement this callback to add a comment to the DOM of the host element.
     * @param value The comment text.
     * @returns The modified element.
     */
    abstract createComment(value: string): any;
    /**
     * Implement this callback to add text to the DOM of the host element.
     * @param value The text string.
     * @returns The modified element.
     */
    abstract createText(value: string): any;
    /**
     * If null or undefined, the view engine won't call it.
     * This is used as a performance optimization for production mode.
     */
    destroyNode: ((node: any) => void) | null;
    /**
     * Appends a child to a given parent node in the host element DOM.
     * @param parent The parent node.
     * @param newChild The new child node.
     */
    abstract appendChild(parent: any, newChild: any): void;
    /**
     * Implement this callback to insert a child node at a given position in a parent node
     * in the host element DOM.
     * @param parent The parent node.
     * @param newChild The new child nodes.
     * @param refChild The existing child node before which `newChild` is inserted.
     */
    abstract insertBefore(parent: any, newChild: any, refChild: any): void;
    /**
     * Implement this callback to remove a child node from the host element's DOM.
     * @param parent The parent node.
     * @param oldChild The child node to remove.
     * @param isHostElement Optionally signal to the renderer whether this element is a host element
     * or not
     */
    abstract removeChild(parent: any, oldChild: any, isHostElement?: boolean): void;
    /**
     * Implement this callback to prepare an element to be bootstrapped
     * as a root element, and return the element instance.
     * @param selectorOrNode The DOM element.
     * @param preserveContent Whether the contents of the root element
     * should be preserved, or cleared upon bootstrap (default behavior).
     * Use with `ViewEncapsulation.ShadowDom` to allow simple native
     * content projection via `<slot>` elements.
     * @returns The root element.
     */
    abstract selectRootElement(selectorOrNode: string | any, preserveContent?: boolean): any;
    /**
     * Implement this callback to get the parent of a given node
     * in the host element's DOM.
     * @param node The child node to query.
     * @returns The parent node, or null if there is no parent.
     * For WebWorkers, always returns true.
     * This is because the check is synchronous,
     * and the caller can't rely on checking for null.
     */
    abstract parentNode(node: any): any;
    /**
     * Implement this callback to get the next sibling node of a given node
     * in the host element's DOM.
     * @returns The sibling node, or null if there is no sibling.
     * For WebWorkers, always returns a value.
     * This is because the check is synchronous,
     * and the caller can't rely on checking for null.
     */
    abstract nextSibling(node: any): any;
    /**
     * Implement this callback to set an attribute value for an element in the DOM.
     * @param el The element.
     * @param name The attribute name.
     * @param value The new value.
     * @param namespace The namespace.
     */
    abstract setAttribute(el: any, name: string, value: string, namespace?: string | null): void;
    /**
     * Implement this callback to remove an attribute from an element in the DOM.
     * @param el The element.
     * @param name The attribute name.
     * @param namespace The namespace.
     */
    abstract removeAttribute(el: any, name: string, namespace?: string | null): void;
    /**
     * Implement this callback to add a class to an element in the DOM.
     * @param el The element.
     * @param name The class name.
     */
    abstract addClass(el: any, name: string): void;
    /**
     * Implement this callback to remove a class from an element in the DOM.
     * @param el The element.
     * @param name The class name.
     */
    abstract removeClass(el: any, name: string): void;
    /**
     * Implement this callback to set a CSS style for an element in the DOM.
     * @param el The element.
     * @param style The name of the style.
     * @param value The new value.
     * @param flags Flags for style variations. No flags are set by default.
     */
    abstract setStyle(el: any, style: string, value: any, flags?: RendererStyleFlags2): void;
    /**
     * Implement this callback to remove the value from a CSS style for an element in the DOM.
     * @param el The element.
     * @param style The name of the style.
     * @param flags Flags for style variations to remove, if set. ???
     */
    abstract removeStyle(el: any, style: string, flags?: RendererStyleFlags2): void;
    /**
     * Implement this callback to set the value of a property of an element in the DOM.
     * @param el The element.
     * @param name The property name.
     * @param value The new value.
     */
    abstract setProperty(el: any, name: string, value: any): void;
    /**
     * Implement this callback to set the value of a node in the host element.
     * @param node The node.
     * @param value The new value.
     */
    abstract setValue(node: any, value: string): void;
    /**
     * Implement this callback to start an event listener.
     * @param target The context in which to listen for events. Can be
     * the entire window or document, the body of the document, or a specific
     * DOM element.
     * @param eventName The event to listen for.
     * @param callback A handler function to invoke when the event occurs.
     * @returns An "unlisten" function for disposing of this handler.
     */
    abstract listen(target: 'window' | 'document' | 'body' | any, eventName: string, callback: (event: any) => boolean | void): () => void;
}

declare type Renderer3 = ObjectOrientedRenderer3 | ProceduralRenderer3;

declare const RENDERER_FACTORY = 10;

/**
 * Creates and initializes a custom renderer that implements the `Renderer2` base class.
 *
 * @publicApi
 */
export declare abstract class RendererFactory2 {
    /**
     * Creates and initializes a custom renderer for a host DOM element.
     * @param hostElement The element to render.
     * @param type The base class to implement.
     * @returns The new custom renderer instance.
     */
    abstract createRenderer(hostElement: any, type: RendererType2 | null): Renderer2;
    /**
     * A callback invoked when rendering has begun.
     */
    abstract begin?(): void;
    /**
     * A callback invoked when rendering has completed.
     */
    abstract end?(): void;
    /**
     * Use with animations test-only mode. Notifies the test when rendering has completed.
     * @returns The asynchronous result of the developer-defined function.
     */
    abstract whenRenderingDone?(): Promise<any>;
}

declare interface RendererFactory3 {
    createRenderer(hostElement: RElement | null, rendererType: RendererType2 | null): Renderer3;
    begin?(): void;
    end?(): void;
}

/**
 * Flags for renderer-specific style modifiers.
 * @publicApi
 */
export declare enum RendererStyleFlags2 {
    /**
     * Marks a style as important.
     */
    Important = 1,
    /**
     * Marks a style as using dash case naming (this-is-dash-case).
     */
    DashCase = 2
}

declare enum RendererStyleFlags3 {
    Important = 1,
    DashCase = 2
}

/**
 * Used by `RendererFactory2` to associate custom rendering data and styles
 * with a rendering implementation.
 *  @publicApi
 */
export declare interface RendererType2 {
    /**
     * A unique identifying string for the new renderer, used when creating
     * unique styles for encapsulation.
     */
    id: string;
    /**
     * The view encapsulation type, which determines how styles are applied to
     * DOM elements. One of
     * - `Emulated` (default): Emulate native scoping of styles.
     * - `Native`: Use the native encapsulation mechanism of the renderer.
     * - `ShadowDom`: Use modern [Shadow
     * DOM](https://w3c.github.io/webcomponents/spec/shadow/) and
     * create a ShadowRoot for component's host element.
     * - `None`: Do not provide any template or style encapsulation.
     */
    encapsulation: ViewEncapsulation;
    /**
     * Defines CSS styles to be stored on a renderer instance.
     */
    styles: (string | any[])[];
    /**
     * Defines arbitrary developer-defined data to be stored on a renderer instance.
     * This is useful for renderers that delegate to other renderers.
     */
    data: {
        [kind: string]: any;
    };
}

/**
 * An internal resolved representation of a factory function created by resolving `Provider`.
 * @publicApi
 */
export declare class ResolvedReflectiveFactory {
    /**
     * Factory function which can return an instance of an object represented by a key.
     */
    factory: Function;
    /**
     * Arguments (dependencies) to the `factory` function.
     */
    dependencies: ɵangular_packages_core_core_e[];
    constructor(
    /**
     * Factory function which can return an instance of an object represented by a key.
     */
    factory: Function, 
    /**
     * Arguments (dependencies) to the `factory` function.
     */
    dependencies: ɵangular_packages_core_core_e[]);
}

/**
 * An internal resolved representation of a `Provider` used by the `Injector`.
 *
 * @usageNotes
 * This is usually created automatically by `Injector.resolveAndCreate`.
 *
 * It can be created manually, as follows:
 *
 * ### Example
 *
 * ```typescript
 * var resolvedProviders = Injector.resolve([{ provide: 'message', useValue: 'Hello' }]);
 * var injector = Injector.fromResolvedProviders(resolvedProviders);
 *
 * expect(injector.get('message')).toEqual('Hello');
 * ```
 *
 * @publicApi
 */
export declare interface ResolvedReflectiveProvider {
    /**
     * A key, usually a `Type<any>`.
     */
    key: ReflectiveKey;
    /**
     * Factory function which can return an instance of an object represented by a key.
     */
    resolvedFactories: ResolvedReflectiveFactory[];
    /**
     * Indicates if the provider is a multi-provider or a regular provider.
     */
    multiProvider: boolean;
}

/**
 * Lazily retrieves the reference value from a forwardRef.
 *
 * Acts as the identity function when given a non-forward-ref value.
 *
 * @usageNotes
 * ### Example
 *
 * {@example core/di/ts/forward_ref/forward_ref_spec.ts region='resolve_forward_ref'}
 *
 * @see `forwardRef`
 * @publicApi
 */
export declare function resolveForwardRef<T>(type: T): T;

/** Subset of API needed for appending elements and text nodes. */
declare interface RNode {
    /**
     * Returns the parent Element, Document, or DocumentFragment
     */
    parentNode: RNode | null;
    /**
     * Returns the parent Element if there is one
     */
    parentElement: RElement | null;
    /**
     * Gets the Node immediately following this one in the parent's childNodes
     */
    nextSibling: RNode | null;
    /**
     * Removes a child from the current node and returns the removed node
     * @param oldChild the child node to remove
     */
    removeChild(oldChild: RNode): RNode;
    /**
     * Insert a child node.
     *
     * Used exclusively for adding View root nodes into ViewAnchor location.
     */
    insertBefore(newChild: RNode, refChild: RNode | null, isViewRoot: boolean): void;
    /**
     * Append a child node.
     *
     * Used exclusively for building up DOM which are static (ie not View roots)
     */
    appendChild(newChild: RNode): RNode;
}

/**
 * RootContext contains information which is shared for all components which
 * were bootstrapped with {@link renderComponent}.
 */
declare interface RootContext {
    /**
     * A function used for scheduling change detection in the future. Usually
     * this is `requestAnimationFrame`.
     */
    scheduler: (workFn: () => void) => void;
    /**
     * A promise which is resolved when all components are considered clean (not dirty).
     *
     * This promise is overwritten every time a first call to {@link markDirty} is invoked.
     */
    clean: Promise<null>;
    /**
     * RootComponents - The components that were instantiated by the call to
     * {@link renderComponent}.
     */
    components: {}[];
    /**
     * The player flushing handler to kick off all animations
     */
    playerHandler: ɵPlayerHandler | null;
    /**
     * What render-related operations to run once a scheduler has been set
     */
    flags: RootContextFlags;
}

declare const enum RootContextFlags {
    Empty = 0,
    DetectChanges = 1,
    FlushPlayers = 2
}

declare interface RootData {
    injector: Injector;
    ngModule: NgModuleRef<any>;
    projectableNodes: any[][];
    selectorOrNode: any;
    renderer: Renderer2;
    rendererFactory: RendererFactory2;
    errorHandler: ErrorHandler;
    sanitizer: Sanitizer;
}

declare interface RText extends RNode {
    textContent: string | null;
}

declare const SANITIZER = 12;

/**
 * Sanitizer is used by the views to sanitize potentially dangerous values.
 *
 * @publicApi
 */
export declare abstract class Sanitizer {
    abstract sanitize(context: SecurityContext, value: {} | string | null): string | null;
    /** @nocollapse */
    static ɵprov: never;
}


/**
 * Function used to sanitize the value before writing it into the renderer.
 */
declare type SanitizerFn = (value: any, tagName?: string, propName?: string) => string;


/**
 * A schema definition associated with an NgModule.
 *
 * @see `@NgModule`, `CUSTOM_ELEMENTS_SCHEMA`, `NO_ERRORS_SCHEMA`
 *
 * @param name The name of a defined schema.
 *
 * @publicApi
 */
export declare interface SchemaMetadata {
    name: string;
}


/**
 * A SecurityContext marks a location that has dangerous security implications, e.g. a DOM property
 * like `innerHTML` that could cause Cross Site Scripting (XSS) security bugs when improperly
 * handled.
 *
 * See DomSanitizer for more details on security in Angular applications.
 *
 * @publicApi
 */
export declare enum SecurityContext {
    NONE = 0,
    HTML = 1,
    STYLE = 2,
    SCRIPT = 3,
    URL = 4,
    RESOURCE_URL = 5
}

/** Flags used to build up CssSelectors */
declare const enum SelectorFlags {
    /** Indicates this is the beginning of a new negative selector */
    NOT = 1,
    /** Mode for matching attributes */
    ATTRIBUTE = 2,
    /** Mode for matching tag names */
    ELEMENT = 4,
    /** Mode for matching class names */
    CLASS = 8
}

/**
 * Type of the Self metadata.
 *
 * @publicApi
 */
export declare interface Self {
}

/**
 * Self decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const Self: SelfDecorator;

/**
 * Type of the Self decorator / constructor function.
 *
 * @publicApi
 */
export declare interface SelfDecorator {
    /**
     * Parameter decorator to be used on constructor parameters,
     * which tells the DI framework to start dependency resolution from the local injector.
     *
     * Resolution works upward through the injector hierarchy, so the children
     * of this class must configure their own providers or be prepared for a null result.
     *
     * @usageNotes
     *
     * In the following example, the dependency can be resolved
     * by the local injector when instantiating the class itself, but not
     * when instantiating a child.
     *
     * <code-example path="core/di/ts/metadata_spec.ts" region="Self">
     * </code-example>
     *
     *
     * @see `SkipSelf`
     * @see `Optional`
     *
     */
    (): any;
    new (): Self;
}

/**
 * Set the {@link GetTestability} implementation used by the Angular testing framework.
 * @publicApi
 */
export declare function setTestabilityGetter(getter: GetTestability): void;


/**
 * Represents a basic change from a previous to a new value for a single
 * property on a directive instance. Passed as a value in a
 * {@link SimpleChanges} object to the `ngOnChanges` hook.
 *
 * @see `OnChanges`
 *
 * @publicApi
 */
export declare class SimpleChange {
    previousValue: any;
    currentValue: any;
    firstChange: boolean;
    constructor(previousValue: any, currentValue: any, firstChange: boolean);
    /**
     * Check whether the new value is the first value assigned.
     */
    isFirstChange(): boolean;
}

/**
 * A hashtable of changes represented by {@link SimpleChange} objects stored
 * at the declared property name they belong to on a Directive or Component. This is
 * the type passed to the `ngOnChanges` hook.
 *
 * @see `OnChanges`
 *
 * @publicApi
 */
export declare interface SimpleChanges {
    [propName: string]: SimpleChange;
}

/**
 * Type of the SkipSelf metadata.
 *
 * @publicApi
 */
export declare interface SkipSelf {
}

/**
 * SkipSelf decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const SkipSelf: SkipSelfDecorator;

/**
 * Type of the SkipSelf decorator / constructor function.
 *
 * @publicApi
 */
export declare interface SkipSelfDecorator {
    /**
     * Parameter decorator to be used on constructor parameters,
     * which tells the DI framework to start dependency resolution from the parent injector.
     * Resolution works upward through the injector hierarchy, so the local injector
     * is not checked for a provider.
     *
     * @usageNotes
     *
     * In the following example, the dependency can be resolved when
     * instantiating a child, but not when instantiating the class itself.
     *
     * <code-example path="core/di/ts/metadata_spec.ts" region="SkipSelf">
     * </code-example>
     *
     * Learn more in the
     * [Dependency Injection guide](guide/dependency-injection-in-action#skip).
     *
     * @see `Self`
     * @see `Optional`
     *
     */
    (): any;
    new (): SkipSelf;
}

/**
 * Configures the `Injector` to return an instance of `useClass` for a token.
 * @see ["Dependency Injection Guide"](guide/dependency-injection).
 *
 * @usageNotes
 *
 * {@example core/di/ts/provider_spec.ts region='StaticClassProvider'}
 *
 * Note that following two providers are not equal:
 *
 * {@example core/di/ts/provider_spec.ts region='StaticClassProviderDifference'}
 *
 * ### Multi-value example
 *
 * {@example core/di/ts/provider_spec.ts region='MultiProviderAspect'}
 *
 * @publicApi
 */
export declare interface StaticClassProvider extends StaticClassSansProvider {
    /**
     * An injection token. Typically an instance of `Type` or `InjectionToken`, but can be `any`.
     */
    provide: any;
    /**
     * When true, injector returns an array of instances. This is useful to allow multiple
     * providers spread across many files to provide configuration information to a common token.
     */
    multi?: boolean;
}

/**
 * Configures the `Injector` to return an instance of `useClass` for a token.
 * Base for `StaticClassProvider` decorator.
 *
 * @publicApi
 */
export declare interface StaticClassSansProvider {
    /**
     * An optional class to instantiate for the `token`. By default, the `provide`
     * class is instantiated.
     */
    useClass: Type<any>;
    /**
     * A list of `token`s to be resolved by the injector. The list of values is then
     * used as arguments to the `useClass` constructor.
     */
    deps: any[];
}

/**
 * Describes how an `Injector` should be configured as static (that is, without reflection).
 * A static provider provides tokens to an injector for various types of dependencies.
 *
 * @see [Injector.create()](/api/core/Injector#create).
 * @see ["Dependency Injection Guide"](guide/dependency-injection-providers).
 *
 * @publicApi
 */
export declare type StaticProvider = ValueProvider | ExistingProvider | StaticClassProvider | ConstructorProvider | FactoryProvider | any[];

/**
 * Used to intercept and sanitize style values before they are written to the renderer.
 *
 * This function is designed to be called in two modes. When a value is not provided
 * then the function will return a boolean whether a property will be sanitized later.
 * If a value is provided then the sanitized version of that will be returned.
 */
declare interface StyleSanitizeFn {
    (prop: string, value: string | ɵSafeValue | null, mode?: StyleSanitizeMode): any;
}

/**
 * A series of flags to instruct a style sanitizer to either validate
 * or sanitize a value.
 *
 * Because sanitization is dependent on the style property (i.e. style
 * sanitization for `width` is much different than for `background-image`)
 * the sanitization function (e.g. `StyleSanitizerFn`) needs to check a
 * property value first before it actually sanitizes any values.
 *
 * This enum exist to allow a style sanitization function to either only
 * do validation (check the property to see whether a value will be
 * sanitized or not) or to sanitize the value (or both).
 *
 * @publicApi
 */
declare const enum StyleSanitizeMode {
    /** Just check to see if the property is required to be sanitized or not */
    ValidateProperty = 1,
    /** Skip checking the property; just sanitize the value */
    SanitizeOnly = 2,
    /** Check the property and (if true) then sanitize the value */
    ValidateAndSanitize = 3
}

/**
 * NgModuleFactoryLoader that uses SystemJS to load NgModuleFactory
 * @publicApi
 * @deprecated the `string` form of `loadChildren` is deprecated, and `SystemJsNgModuleLoader` is
 * part of its implementation. See `LoadChildren` for more details.
 */
export declare class SystemJsNgModuleLoader implements NgModuleFactoryLoader {
    private _compiler;
    private _config;
    constructor(_compiler: Compiler, config?: SystemJsNgModuleLoaderConfig);
    load(path: string): Promise<NgModuleFactory<any>>;
    private loadAndCompile;
    private loadFactory;
    static ɵfac: ɵngcc0.ɵɵFactoryDef<SystemJsNgModuleLoader>;
    static ɵprov: ɵngcc0.ɵɵInjectableDef<SystemJsNgModuleLoader>;
}

/**
 * Configuration for SystemJsNgModuleLoader.
 * token.
 *
 * @publicApi
 * @deprecated the `string` form of `loadChildren` is deprecated, and `SystemJsNgModuleLoaderConfig`
 * is part of its implementation. See `LoadChildren` for more details.
 */
export declare abstract class SystemJsNgModuleLoaderConfig {
    /**
     * Prefix to add when computing the name of the factory module for a given module name.
     */
    factoryPathPrefix: string;
    /**
     * Suffix to add when computing the name of the factory module for a given module name.
     */
    factoryPathSuffix: string;
}

declare const T_HOST = 6;

/**
 * A combination of:
 * - Attribute names and values.
 * - Special markers acting as flags to alter attributes processing.
 * - Parsed ngProjectAs selectors.
 */
declare type TAttributes = (string | ɵAttributeMarker | CssSelector)[];

/**
 * Constants that are associated with a view. Includes:
 * - Attribute arrays.
 * - Local definition arrays.
 */
declare type TConstants = (TAttributes | string)[];

/** Static data for an LContainer */
declare interface TContainerNode extends TNode {
    /**
     * Index in the data[] array.
     *
     * If it's -1, this is a dynamically created container node that isn't stored in
     * data[] (e.g. when you inject ViewContainerRef) .
     */
    index: number;
    child: null;
    /**
     * Container nodes will have parents unless:
     *
     * - They are the first node of a component or embedded view
     * - They are dynamically created
     */
    parent: ɵangular_packages_core_core_be | TElementContainerNode | null;
    tViews: TView | TView[] | null;
    projection: null;
}

/**
 * Static data that corresponds to the instance-specific data array on an LView.
 *
 * Each node's static data is stored in tData at the same index that it's stored
 * in the data array.  Any nodes that do not have static data store a null value in
 * tData to avoid a sparse array.
 *
 * Each pipe's definition is stored here at the same index as its pipe instance in
 * the data array.
 *
 * Each host property's name is stored here at the same index as its value in the
 * data array.
 *
 * Each property binding name is stored here at the same index as its value in
 * the data array. If the binding is an interpolation, the static string values
 * are stored parallel to the dynamic values. Example:
 *
 * id="prefix {{ v0 }} a {{ v1 }} b {{ v2 }} suffix"
 *
 * LView       |   TView.data
 *------------------------
 *  v0 value   |   'a'
 *  v1 value   |   'b'
 *  v2 value   |   id � prefix � suffix
 *
 * Injector bloom filters are also stored here.
 */
declare type TData = (TNode | ɵPipeDef<any> | ɵDirectiveDef<any> | ɵComponentDef<any> | number | TStylingRange | TStylingKey | Type<any> | InjectionToken<any> | TI18n | I18nUpdateOpCodes | null | string)[];

/** Static data for an <ng-container> */
declare interface TElementContainerNode extends TNode {
    /** Index in the LView[] array. */
    index: number;
    child: ɵangular_packages_core_core_be | TTextNode | TContainerNode | TElementContainerNode | TProjectionNode | null;
    parent: ɵangular_packages_core_core_be | TElementContainerNode | null;
    tViews: null;
    projection: null;
}

/**
 * Represents an embedded template that can be used to instantiate embedded views.
 * To instantiate embedded views based on a template, use the `ViewContainerRef`
 * method `createEmbeddedView()`.
 *
 * Access a `TemplateRef` instance by placing a directive on an `<ng-template>`
 * element (or directive prefixed with `*`). The `TemplateRef` for the embedded view
 * is injected into the constructor of the directive,
 * using the `TemplateRef` token.
 *
 * You can also use a `Query` to find a `TemplateRef` associated with
 * a component or a directive.
 *
 * @see `ViewContainerRef`
 * @see [Navigate the Component Tree with DI](guide/dependency-injection-navtree)
 *
 * @publicApi
 */
export declare abstract class TemplateRef<C> {
    /**
     * The anchor element in the parent view for this embedded view.
     *
     * The data-binding and injection contexts of embedded views created from this `TemplateRef`
     * inherit from the contexts of this location.
     *
     * Typically new embedded views are attached to the view container of this location, but in
     * advanced use-cases, the view can be attached to a different container while keeping the
     * data-binding and injection context from the original location.
     *
     */
    abstract get elementRef(): ElementRef;
    /**
     * Instantiates an embedded view based on this template,
     * and attaches it to the view container.
     * @param context The data-binding context of the embedded view, as declared
     * in the `<ng-template>` usage.
     * @returns The new embedded view object.
     */
    abstract createEmbeddedView(context: C): EmbeddedViewRef<C>;
}

/**
 * The Testability service provides testing hooks that can be accessed from
 * the browser and by services such as Protractor. Each bootstrapped Angular
 * application on the page will have an instance of Testability.
 * @publicApi
 */
export declare class Testability implements PublicTestability {
    private _ngZone;
    private _pendingCount;
    private _isZoneStable;
    private _callbacks;
    private taskTrackingZone;
    constructor(_ngZone: NgZone);
    private _watchAngularEvents;
    /**
     * Increases the number of pending request
     * @deprecated pending requests are now tracked with zones.
     */
    increasePendingRequestCount(): number;
    /**
     * Decreases the number of pending request
     * @deprecated pending requests are now tracked with zones
     */
    decreasePendingRequestCount(): number;
    /**
     * Whether an associated application is stable
     */
    isStable(): boolean;
    private _runCallbacksIfReady;
    private getPendingTasks;
    private addCallback;
    /**
     * Wait for the application to be stable with a timeout. If the timeout is reached before that
     * happens, the callback receives a list of the macro tasks that were pending, otherwise null.
     *
     * @param doneCb The callback to invoke when Angular is stable or the timeout expires
     *    whichever comes first.
     * @param timeout Optional. The maximum time to wait for Angular to become stable. If not
     *    specified, whenStable() will wait forever.
     * @param updateCb Optional. If specified, this callback will be invoked whenever the set of
     *    pending macrotasks changes. If this callback returns true doneCb will not be invoked
     *    and no further updates will be issued.
     */
    whenStable(doneCb: Function, timeout?: number, updateCb?: Function): void;
    /**
     * Get the number of pending requests
     * @deprecated pending requests are now tracked with zones
     */
    getPendingRequestCount(): number;
    /**
     * Find providers by name
     * @param using The root element to search from
     * @param provider The name of binding variable
     * @param exactMatch Whether using exactMatch
     */
    findProviders(using: any, provider: string, exactMatch: boolean): any[];
    static ɵfac: ɵngcc0.ɵɵFactoryDef<Testability>;
    static ɵprov: ɵngcc0.ɵɵInjectableDef<Testability>;
}

/**
 * A global registry of {@link Testability} instances for specific elements.
 * @publicApi
 */
export declare class TestabilityRegistry {
    constructor();
    /**
     * Registers an application with a testability hook so that it can be tracked
     * @param token token of application, root element
     * @param testability Testability hook
     */
    registerApplication(token: any, testability: Testability): void;
    /**
     * Unregisters an application.
     * @param token token of application, root element
     */
    unregisterApplication(token: any): void;
    /**
     * Unregisters all applications
     */
    unregisterAllApplications(): void;
    /**
     * Get a testability hook associated with the application
     * @param elem root element
     */
    getTestability(elem: any): Testability | null;
    /**
     * Get all registered testabilities
     */
    getAllTestabilities(): Testability[];
    /**
     * Get all registered applications(root elements)
     */
    getAllRootElements(): any[];
    /**
     * Find testability of a node in the Tree
     * @param elem node
     * @param findInAncestors whether finding testability in ancestors if testability was not found in
     * current node
     */
    findTestabilityInTree(elem: Node, findInAncestors?: boolean): Testability | null;
    static ɵfac: ɵngcc0.ɵɵFactoryDef<TestabilityRegistry>;
    static ɵprov: ɵngcc0.ɵɵInjectableDef<TestabilityRegistry>;
}

declare interface TextDef {
    prefix: string;
}

/**
 * Store information for the i18n translation block.
 */
declare interface TI18n {
    /**
     * Number of slots to allocate in expando.
     *
     * This is the max number of DOM elements which will be created by this i18n + ICU blocks. When
     * the DOM elements are being created they are stored in the EXPANDO, so that update OpCodes can
     * write into them.
     */
    vars: number;
    /**
     * A set of OpCodes which will create the Text Nodes and ICU anchors for the translation blocks.
     *
     * NOTE: The ICU anchors are filled in with ICU Update OpCode.
     */
    create: I18nMutateOpCodes;
    /**
     * A set of OpCodes which will be executed on each change detection to determine if any changes to
     * DOM are required.
     */
    update: I18nUpdateOpCodes;
    /**
     * A list of ICUs in a translation block (or `null` if block has no ICUs).
     *
     * Example:
     * Given: `<div i18n>You have {count, plural, ...} and {state, switch, ...}</div>`
     * There would be 2 ICUs in this array.
     *   1. `{count, plural, ...}`
     *   2. `{state, switch, ...}`
     */
    icus: TIcu[] | null;
}

declare interface TIcu {
    /**
     * Defines the ICU type of `select` or `plural`
     */
    type: IcuType;
    /**
     * Number of slots to allocate in expando for each case.
     *
     * This is the max number of DOM elements which will be created by this i18n + ICU blocks. When
     * the DOM elements are being created they are stored in the EXPANDO, so that update OpCodes can
     * write into them.
     */
    vars: number[];
    /**
     * An optional array of child/sub ICUs.
     *
     * In case of nested ICUs such as:
     * ```
     * {�0�, plural,
     *   =0 {zero}
     *   other {�0� {�1�, select,
     *                     cat {cats}
     *                     dog {dogs}
     *                     other {animals}
     *                   }!
     *   }
     * }
     * ```
     * When the parent ICU is changing it must clean up child ICUs as well. For this reason it needs
     * to know which child ICUs to run clean up for as well.
     *
     * In the above example this would be:
     * ```ts
     * [
     *   [],   // `=0` has no sub ICUs
     *   [1],  // `other` has one subICU at `1`st index.
     * ]
     * ```
     *
     * The reason why it is Array of Arrays is because first array represents the case, and second
     * represents the child ICUs to clean up. There may be more than one child ICUs per case.
     */
    childIcus: number[][];
    /**
     * A list of case values which the current ICU will try to match.
     *
     * The last value is `other`
     */
    cases: any[];
    /**
     * A set of OpCodes to apply in order to build up the DOM render tree for the ICU
     */
    create: I18nMutateOpCodes[];
    /**
     * A set of OpCodes to apply in order to destroy the DOM render tree for the ICU.
     */
    remove: I18nMutateOpCodes[];
    /**
     * A set of OpCodes to apply in order to update the DOM render tree for the ICU bindings.
     */
    update: I18nUpdateOpCodes[];
}

/**
 * Binding data (flyweight) for a particular node that is shared between all templates
 * of a specific type.
 *
 * If a property is:
 *    - PropertyAliases: that property's data was generated and this is it
 *    - Null: that property's data was already generated and nothing was found.
 *    - Undefined: that property's data has not yet been generated
 *
 * see: https://en.wikipedia.org/wiki/Flyweight_pattern for more on the Flyweight pattern
 */
declare interface TNode {
    /** The type of the TNode. See TNodeType. */
    type: TNodeType;
    /**
     * Index of the TNode in TView.data and corresponding native element in LView.
     *
     * This is necessary to get from any TNode to its corresponding native element when
     * traversing the node tree.
     *
     * If index is -1, this is a dynamically created container node or embedded view node.
     */
    index: number;
    /**
     * The index of the closest injector in this node's LView.
     *
     * If the index === -1, there is no injector on this node or any ancestor node in this view.
     *
     * If the index !== -1, it is the index of this node's injector OR the index of a parent injector
     * in the same view. We pass the parent injector index down the node tree of a view so it's
     * possible to find the parent injector without walking a potentially deep node tree. Injector
     * indices are not set across view boundaries because there could be multiple component hosts.
     *
     * If tNode.injectorIndex === tNode.parent.injectorIndex, then the index belongs to a parent
     * injector.
     */
    injectorIndex: number;
    /**
     * Stores starting index of the directives.
     */
    directiveStart: number;
    /**
     * Stores final exclusive index of the directives.
     */
    directiveEnd: number;
    /**
     * Stores the last directive which had a styling instruction.
     *
     * Initial value of this is `-1` which means that no `hostBindings` styling instruction has
     * executed. As `hostBindings` instructions execute they set the value to the index of the
     * `DirectiveDef` which contained the last `hostBindings` styling instruction.
     *
     * Valid values are:
     * - `-1` No `hostBindings` instruction has executed.
     * - `directiveStart <= directiveStylingLast < directiveEnd`: Points to the `DirectiveDef` of the
     *   last styling instruction which executed in the `hostBindings`.
     *
     * This data is needed so that styling instructions know which static styling data needs to be
     * collected from the `DirectiveDef.hostAttrs`. A styling instruction needs to collect all data
     * since last styling instruction.
     */
    directiveStylingLast: number;
    /**
     * Stores indexes of property bindings. This field is only set in the ngDevMode and holds indexes
     * of property bindings so TestBed can get bound property metadata for a given node.
     */
    propertyBindings: number[] | null;
    /**
     * Stores if Node isComponent, isProjected, hasContentQuery, hasClassInput and hasStyleInput etc.
     */
    flags: TNodeFlags;
    /**
     * This number stores two values using its bits:
     *
     * - the index of the first provider on that node (first 16 bits)
     * - the count of view providers from the component on this node (last 16 bits)
     */
    providerIndexes: TNodeProviderIndexes;
    /** The tag name associated with this node. */
    tagName: string | null;
    /**
     * Attributes associated with an element. We need to store attributes to support various use-cases
     * (attribute injection, content projection with selectors, directives matching).
     * Attributes are stored statically because reading them from the DOM would be way too slow for
     * content projection and queries.
     *
     * Since attrs will always be calculated first, they will never need to be marked undefined by
     * other instructions.
     *
     * For regular attributes a name of an attribute and its value alternate in the array.
     * e.g. ['role', 'checkbox']
     * This array can contain flags that will indicate "special attributes" (attributes with
     * namespaces, attributes extracted from bindings and outputs).
     */
    attrs: TAttributes | null;
    /**
     * Same as `TNode.attrs` but contains merged data across all directive host bindings.
     *
     * We need to keep `attrs` as unmerged so that it can be used for attribute selectors.
     * We merge attrs here so that it can be used in a performant way for initial rendering.
     *
     * The `attrs` are merged in first pass in following order:
     * - Component's `hostAttrs`
     * - Directives' `hostAttrs`
     * - Template `TNode.attrs` associated with the current `TNode`.
     */
    mergedAttrs: TAttributes | null;
    /**
     * A set of local names under which a given element is exported in a template and
     * visible to queries. An entry in this array can be created for different reasons:
     * - an element itself is referenced, ex.: `<div #foo>`
     * - a component is referenced, ex.: `<my-cmpt #foo>`
     * - a directive is referenced, ex.: `<my-cmpt #foo="directiveExportAs">`.
     *
     * A given element might have different local names and those names can be associated
     * with a directive. We store local names at even indexes while odd indexes are reserved
     * for directive index in a view (or `-1` if there is no associated directive).
     *
     * Some examples:
     * - `<div #foo>` => `["foo", -1]`
     * - `<my-cmpt #foo>` => `["foo", myCmptIdx]`
     * - `<my-cmpt #foo #bar="directiveExportAs">` => `["foo", myCmptIdx, "bar", directiveIdx]`
     * - `<div #foo #bar="directiveExportAs">` => `["foo", -1, "bar", directiveIdx]`
     */
    localNames: (string | number)[] | null;
    /** Information about input properties that need to be set once from attribute data. */
    initialInputs: InitialInputData | null | undefined;
    /**
     * Input data for all directives on this node. `null` means that there are no directives with
     * inputs on this node.
     */
    inputs: PropertyAliases | null;
    /**
     * Output data for all directives on this node. `null` means that there are no directives with
     * outputs on this node.
     */
    outputs: PropertyAliases | null;
    /**
     * The TView or TViews attached to this node.
     *
     * If this TNode corresponds to an LContainer with inline views, the container will
     * need to store separate static data for each of its view blocks (TView[]). Otherwise,
     * nodes in inline views with the same index as nodes in their parent views will overwrite
     * each other, as they are in the same template.
     *
     * Each index in this array corresponds to the static data for a certain
     * view. So if you had V(0) and V(1) in a container, you might have:
     *
     * [
     *   [{tagName: 'div', attrs: ...}, null],     // V(0) TView
     *   [{tagName: 'button', attrs ...}, null]    // V(1) TView
     *
     * If this TNode corresponds to an LContainer with a template (e.g. structural
     * directive), the template's TView will be stored here.
     *
     * If this TNode corresponds to an element, tViews will be null .
     */
    tViews: TView | TView[] | null;
    /**
     * The next sibling node. Necessary so we can propagate through the root nodes of a view
     * to insert them or remove them from the DOM.
     */
    next: TNode | null;
    /**
     * The next projected sibling. Since in Angular content projection works on the node-by-node basis
     * the act of projecting nodes might change nodes relationship at the insertion point (target
     * view). At the same time we need to keep initial relationship between nodes as expressed in
     * content view.
     */
    projectionNext: TNode | null;
    /**
     * First child of the current node.
     *
     * For component nodes, the child will always be a ContentChild (in same view).
     * For embedded view nodes, the child will be in their child view.
     */
    child: TNode | null;
    /**
     * Parent node (in the same view only).
     *
     * We need a reference to a node's parent so we can append the node to its parent's native
     * element at the appropriate time.
     *
     * If the parent would be in a different view (e.g. component host), this property will be null.
     * It's important that we don't try to cross component boundaries when retrieving the parent
     * because the parent will change (e.g. index, attrs) depending on where the component was
     * used (and thus shouldn't be stored on TNode). In these cases, we retrieve the parent through
     * LView.node instead (which will be instance-specific).
     *
     * If this is an inline view node (V), the parent will be its container.
     */
    parent: ɵangular_packages_core_core_be | TContainerNode | null;
    /**
     * List of projected TNodes for a given component host element OR index into the said nodes.
     *
     * For easier discussion assume this example:
     * `<parent>`'s view definition:
     * ```
     * <child id="c1">content1</child>
     * <child id="c2"><span>content2</span></child>
     * ```
     * `<child>`'s view definition:
     * ```
     * <ng-content id="cont1"></ng-content>
     * ```
     *
     * If `Array.isArray(projection)` then `TNode` is a host element:
     * - `projection` stores the content nodes which are to be projected.
     *    - The nodes represent categories defined by the selector: For example:
     *      `<ng-content/><ng-content select="abc"/>` would represent the heads for `<ng-content/>`
     *      and `<ng-content select="abc"/>` respectively.
     *    - The nodes we store in `projection` are heads only, we used `.next` to get their
     *      siblings.
     *    - The nodes `.next` is sorted/rewritten as part of the projection setup.
     *    - `projection` size is equal to the number of projections `<ng-content>`. The size of
     *      `c1` will be `1` because `<child>` has only one `<ng-content>`.
     * - we store `projection` with the host (`c1`, `c2`) rather than the `<ng-content>` (`cont1`)
     *   because the same component (`<child>`) can be used in multiple locations (`c1`, `c2`) and as
     *   a result have different set of nodes to project.
     * - without `projection` it would be difficult to efficiently traverse nodes to be projected.
     *
     * If `typeof projection == 'number'` then `TNode` is a `<ng-content>` element:
     * - `projection` is an index of the host's `projection`Nodes.
     *   - This would return the first head node to project:
     *     `getHost(currentTNode).projection[currentTNode.projection]`.
     * - When projecting nodes the parent node retrieved may be a `<ng-content>` node, in which case
     *   the process is recursive in nature.
     *
     * If `projection` is of type `RNode[][]` than we have a collection of native nodes passed as
     * projectable nodes during dynamic component creation.
     */
    projection: (TNode | RNode[])[] | number | null;
    /**
     * A collection of all style static values for an element.
     *
     * This field will be populated if and when:
     *
     * - There are one or more initial styles on an element (e.g. `<div style="width:200px">`)
     */
    styles: string | null;
    /**
     * A `KeyValueArray` version of residual `styles`.
     *
     * When there are styling instructions than each instruction stores the static styling
     * which is of lower priority than itself. This means that there may be a higher priority styling
     * than the instruction.
     *
     * Imagine:
     * ```
     * <div style="color: highest;" my-dir>
     *
     * @Directive({
     *   host: {
     *     style: 'color: lowest; ',
     *     '[styles.color]': 'exp' // ɵɵstyleProp('color', ctx.exp);
     *   }
     * })
     * ```
     *
     * In the above case:
     * - `color: lowest` is stored with `ɵɵstyleProp('color', ctx.exp);` instruction
     * -  `color: highest` is the residual and is stored here.
     *
     * - `undefined': not initialized.
     * - `null`: initialized but `styles` is `null`
     * - `KeyValueArray`: parsed version of `styles`.
     */
    residualStyles: KeyValueArray<any> | undefined | null;
    /**
     * A collection of all class static values for an element.
     *
     * This field will be populated if and when:
     *
     * - There are one or more initial classes on an element (e.g. `<div class="one two three">`)
     */
    classes: string | null;
    /**
     * A `KeyValueArray` version of residual `classes`.
     *
     * Same as `TNode.residualStyles` but for classes.
     *
     * - `undefined': not initialized.
     * - `null`: initialized but `classes` is `null`
     * - `KeyValueArray`: parsed version of `classes`.
     */
    residualClasses: KeyValueArray<any> | undefined | null;
    /**
     * Stores the head/tail index of the class bindings.
     *
     * - If no bindings, the head and tail will both be 0.
     * - If there are template bindings, stores the head/tail of the class bindings in the template.
     * - If no template bindings but there are host bindings, the head value will point to the last
     *   host binding for "class" (not the head of the linked list), tail will be 0.
     *
     * See: `style_binding_list.ts` for details.
     *
     * This is used by `insertTStylingBinding` to know where the next styling binding should be
     * inserted so that they can be sorted in priority order.
     */
    classBindings: TStylingRange;
    /**
     * Stores the head/tail index of the class bindings.
     *
     * - If no bindings, the head and tail will both be 0.
     * - If there are template bindings, stores the head/tail of the style bindings in the template.
     * - If no template bindings but there are host bindings, the head value will point to the last
     *   host binding for "style" (not the head of the linked list), tail will be 0.
     *
     * See: `style_binding_list.ts` for details.
     *
     * This is used by `insertTStylingBinding` to know where the next styling binding should be
     * inserted so that they can be sorted in priority order.
     */
    styleBindings: TStylingRange;
}

/**
 * Corresponds to the TNode.flags property.
 */
declare const enum TNodeFlags {
    /** Bit #1 - This bit is set if the node is a host for any directive (including a component) */
    isDirectiveHost = 1,
    /**
     * Bit #2 - This bit is set if the node is a host for a component.
     *
     * Setting this bit implies that the `isDirectiveHost` bit is set as well.
     * */
    isComponentHost = 2,
    /** Bit #3 - This bit is set if the node has been projected */
    isProjected = 4,
    /** Bit #4 - This bit is set if any directive on this node has content queries */
    hasContentQuery = 8,
    /** Bit #5 - This bit is set if the node has any "class" inputs */
    hasClassInput = 16,
    /** Bit #6 - This bit is set if the node has any "style" inputs */
    hasStyleInput = 32,
    /** Bit #7 This bit is set if the node has been detached by i18n */
    isDetached = 64,
    /**
     * Bit #8 - This bit is set if the node has directives with host bindings.
     *
     * This flags allows us to guard host-binding logic and invoke it only on nodes
     * that actually have directives with host bindings.
     */
    hasHostBindings = 128
}

/**
 * Corresponds to the TNode.providerIndexes property.
 */
declare const enum TNodeProviderIndexes {
    /** The index of the first provider on this node is encoded on the least significant bits */
    ProvidersStartIndexMask = 65535,
    /** The count of view providers from the component on this node is encoded on the 16 most
       significant bits */
    CptViewProvidersCountShift = 16,
    CptViewProvidersCountShifter = 65536
}

/**
 * TNodeType corresponds to the {@link TNode} `type` property.
 */
declare const enum TNodeType {
    /**
     * The TNode contains information about an {@link LContainer} for embedded views.
     */
    Container = 0,
    /**
     * The TNode contains information about an `<ng-content>` projection
     */
    Projection = 1,
    /**
     * The TNode contains information about an {@link LView}
     */
    View = 2,
    /**
     * The TNode contains information about a DOM element aka {@link RNode}.
     */
    Element = 3,
    /**
     * The TNode contains information about an `<ng-container>` element {@link RNode}.
     */
    ElementContainer = 4,
    /**
     * The TNode contains information about an ICU comment used in `i18n`.
     */
    IcuContainer = 5
}

/**
 * Type representing a set of TNodes that can have local refs (`#foo`) placed on them.
 */
declare type TNodeWithLocalRefs = TContainerNode | ɵangular_packages_core_core_be | TElementContainerNode;

/** Static data for an LProjectionNode  */
declare interface TProjectionNode extends TNode {
    /** Index in the data[] array */
    child: null;
    /**
     * Projection nodes will have parents unless they are the first node of a component
     * or embedded view (which means their parent is in a different view and must be
     * retrieved using LView.node).
     */
    parent: ɵangular_packages_core_core_be | TElementContainerNode | null;
    tViews: null;
    /** Index of the projection node. (See TNode.projection for more info.) */
    projection: number;
}

/**
 * TQueries represent a collection of individual TQuery objects tracked in a given view. Most of the
 * methods on this interface are simple proxy methods to the corresponding functionality on TQuery.
 */
declare interface TQueries {
    /**
     * Adds a new TQuery to a collection of queries tracked in a given view.
     * @param tQuery
     */
    track(tQuery: TQuery): void;
    /**
     * Returns a TQuery instance for at the given index  in the queries array.
     * @param index
     */
    getByIndex(index: number): TQuery;
    /**
     * Returns the number of queries tracked in a given view.
     */
    length: number;
    /**
     * A proxy method that iterates over all the TQueries in a given TView and calls the corresponding
     * `elementStart` on each and every TQuery.
     * @param tView
     * @param tNode
     */
    elementStart(tView: TView, tNode: TNode): void;
    /**
     * A proxy method that iterates over all the TQueries in a given TView and calls the corresponding
     * `elementEnd` on each and every TQuery.
     * @param tNode
     */
    elementEnd(tNode: TNode): void;
    /**
     * A proxy method that iterates over all the TQueries in a given TView and calls the corresponding
     * `template` on each and every TQuery.
     * @param tView
     * @param tNode
     */
    template(tView: TView, tNode: TNode): void;
    /**
    * A proxy method that iterates over all the TQueries in a given TView and calls the corresponding
     * `embeddedTView` on each and every TQuery.
     * @param tNode
     */
    embeddedTView(tNode: TNode): TQueries | null;
}

/**
 * TQuery objects represent all the query-related data that remain the same from one view instance
 * to another and can be determined on the very first template pass. Most notably TQuery holds all
 * the matches for a given view.
 */
declare interface TQuery {
    /**
     * Query metadata extracted from query annotations.
     */
    metadata: TQueryMetadata;
    /**
     * Index of a query in a declaration view in case of queries propagated to en embedded view, -1
     * for queries declared in a given view. We are storing this index so we can find a parent query
     * to clone for an embedded view (when an embedded view is created).
     */
    indexInDeclarationView: number;
    /**
     * Matches collected on the first template pass. Each match is a pair of:
     * - TNode index;
     * - match index;
     *
     * A TNode index can be either:
     * - a positive number (the most common case) to indicate a matching TNode;
     * - a negative number to indicate that a given query is crossing a <ng-template> element and
     * results from views created based on TemplateRef should be inserted at this place.
     *
     * A match index is a number used to find an actual value (for a given node) when query results
     * are materialized. This index can have one of the following values:
     * - -2 - indicates that we need to read a special token (TemplateRef, ViewContainerRef etc.);
     * - -1 - indicates that we need to read a default value based on the node type (TemplateRef for
     * ng-template and ElementRef for other elements);
     * - a positive number - index of an injectable to be read from the element injector.
     */
    matches: number[] | null;
    /**
     * A flag indicating if a given query crosses an <ng-template> element. This flag exists for
     * performance reasons: we can notice that queries not crossing any <ng-template> elements will
     * have matches from a given view only (and adapt processing accordingly).
     */
    crossesNgTemplate: boolean;
    /**
     * A method call when a given query is crossing an element (or element container). This is where a
     * given TNode is matched against a query predicate.
     * @param tView
     * @param tNode
     */
    elementStart(tView: TView, tNode: TNode): void;
    /**
     * A method called when processing the elementEnd instruction - this is mostly useful to determine
     * if a given content query should match any nodes past this point.
     * @param tNode
     */
    elementEnd(tNode: TNode): void;
    /**
     * A method called when processing the template instruction. This is where a
     * given TContainerNode is matched against a query predicate.
     * @param tView
     * @param tNode
     */
    template(tView: TView, tNode: TNode): void;
    /**
     * A query-related method called when an embedded TView is created based on the content of a
     * <ng-template> element. We call this method to determine if a given query should be propagated
     * to the embedded view and if so - return a cloned TQuery for this embedded view.
     * @param tNode
     * @param childQueryIndex
     */
    embeddedTView(tNode: TNode, childQueryIndex: number): TQuery | null;
}

/**
 * An object representing query metadata extracted from query annotations.
 */
declare interface TQueryMetadata {
    predicate: Type<any> | string[];
    descendants: boolean;
    read: any;
    isStatic: boolean;
}

/**
 * An optional function passed into the `NgForOf` directive that defines how to track
 * changes for items in an iterable.
 * The function takes the iteration index and item ID.
 * When supplied, Angular tracks changes by the return value of the function.
 *
 * @publicApi
 */
export declare interface TrackByFunction<T> {
    (index: number, item: T): any;
}

/**
 * Use this token at bootstrap to provide the content of your translation file (`xtb`,
 * `xlf` or `xlf2`) when you want to translate your application in another language.
 *
 * See the [i18n guide](guide/i18n#merge) for more information.
 *
 * @usageNotes
 * ### Example
 *
 * ```typescript
 * import { TRANSLATIONS } from '@angular/core';
 * import { platformBrowserDynamic } from '@angular/platform-browser-dynamic';
 * import { AppModule } from './app/app.module';
 *
 * // content of your translation file
 * const translations = '....';
 *
 * platformBrowserDynamic().bootstrapModule(AppModule, {
 *   providers: [{provide: TRANSLATIONS, useValue: translations }]
 * });
 * ```
 *
 * @publicApi
 */
export declare const TRANSLATIONS: InjectionToken<string>;

/**
 * Provide this token at bootstrap to set the format of your {@link TRANSLATIONS}: `xtb`,
 * `xlf` or `xlf2`.
 *
 * See the [i18n guide](guide/i18n#merge) for more information.
 *
 * @usageNotes
 * ### Example
 *
 * ```typescript
 * import { TRANSLATIONS_FORMAT } from '@angular/core';
 * import { platformBrowserDynamic } from '@angular/platform-browser-dynamic';
 * import { AppModule } from './app/app.module';
 *
 * platformBrowserDynamic().bootstrapModule(AppModule, {
 *   providers: [{provide: TRANSLATIONS_FORMAT, useValue: 'xlf' }]
 * });
 * ```
 *
 * @publicApi
 */
export declare const TRANSLATIONS_FORMAT: InjectionToken<string>;

/**
 * Value stored in the `TData` which is needed to re-concatenate the styling.
 *
 * See: `TStylingKeyPrimitive` and `TStylingStatic`
 */
declare type TStylingKey = TStylingKeyPrimitive | TStylingStatic;

/**
 * The primitive portion (`TStylingStatic` removed) of the value stored in the `TData` which is
 * needed to re-concatenate the styling.
 *
 * - `string`: Stores the property name. Used with `ɵɵstyleProp`/`ɵɵclassProp` instruction.
 * - `null`: Represents map, so there is no name. Used with `ɵɵstyleMap`/`ɵɵclassMap`.
 * - `false`: Represents an ignore case. This happens when `ɵɵstyleProp`/`ɵɵclassProp` instruction
 *   is combined with directive which shadows its input `@Input('class')`. That way the binding
 *   should not participate in the styling resolution.
 */
declare type TStylingKeyPrimitive = string | null | false;

/**
 * This is a branded number which contains previous and next index.
 *
 * When we come across styling instructions we need to store the `TStylingKey` in the correct
 * order so that we can re-concatenate the styling value in the desired priority.
 *
 * The insertion can happen either at the:
 * - end of template as in the case of coming across additional styling instruction in the template
 * - in front of the template in the case of coming across additional instruction in the
 *   `hostBindings`.
 *
 * We use `TStylingRange` to store the previous and next index into the `TData` where the template
 * bindings can be found.
 *
 * - bit 0 is used to mark that the previous index has a duplicate for current value.
 * - bit 1 is used to mark that the next index has a duplicate for the current value.
 * - bits 2-16 are used to encode the next/tail of the template.
 * - bits 17-32 are used to encode the previous/head of template.
 *
 * NODE: *duplicate* false implies that it is statically known that this binding will not collide
 * with other bindings and therefore there is no need to check other bindings. For example the
 * bindings in `<div [style.color]="exp" [style.width]="exp">` will never collide and will have
 * their bits set accordingly. Previous duplicate means that we may need to check previous if the
 * current binding is `null`. Next duplicate means that we may need to check next bindings if the
 * current binding is not `null`.
 *
 * NOTE: `0` has special significance and represents `null` as in no additional pointer.
 */
declare interface TStylingRange {
    __brand__: 'TStylingRange';
}

/**
 * Store the static values for the styling binding.
 *
 * The `TStylingStatic` is just `KeyValueArray` where key `""` (stored at location 0) contains the
 * `TStylingKey` (stored at location 1). In other words this wraps the `TStylingKey` such that the
 * `""` contains the wrapped value.
 *
 * When instructions are resolving styling they may need to look forward or backwards in the linked
 * list to resolve the value. For this reason we have to make sure that he linked list also contains
 * the static values. However the list only has space for one item per styling instruction. For this
 * reason we store the static values here as part of the `TStylingKey`. This means that the
 * resolution function when looking for a value needs to first look at the binding value, and than
 * at `TStylingKey` (if it exists).
 *
 * Imagine we have:
 *
 * ```
 * <div class="TEMPLATE" my-dir>
 *
 * @Directive({
 *   host: {
 *     class: 'DIR',
 *     '[class.dynamic]': 'exp' // ɵɵclassProp('dynamic', ctx.exp);
 *   }
 * })
 * ```
 *
 * In the above case the linked list will contain one item:
 *
 * ```
 *   // assume binding location: 10 for `ɵɵclassProp('dynamic', ctx.exp);`
 *   tData[10] = <TStylingStatic>[
 *     '': 'dynamic', // This is the wrapped value of `TStylingKey`
 *     'DIR': true,   // This is the default static value of directive binding.
 *   ];
 *   tData[10 + 1] = 0; // We don't have prev/next.
 *
 *   lView[10] = undefined;     // assume `ctx.exp` is `undefined`
 *   lView[10 + 1] = undefined; // Just normalized `lView[10]`
 * ```
 *
 * So when the function is resolving styling value, it first needs to look into the linked list
 * (there is none) and than into the static `TStylingStatic` too see if there is a default value for
 * `dynamic` (there is not). Therefore it is safe to remove it.
 *
 * If setting `true` case:
 * ```
 *   lView[10] = true;     // assume `ctx.exp` is `true`
 *   lView[10 + 1] = true; // Just normalized `lView[10]`
 * ```
 * So when the function is resolving styling value, it first needs to look into the linked list
 * (there is none) and than into `TNode.residualClass` (TNode.residualStyle) which contains
 * ```
 *   tNode.residualClass = [
 *     'TEMPLATE': true,
 *   ];
 * ```
 *
 * This means that it is safe to add class.
 */
declare interface TStylingStatic extends KeyValueArray<any> {
}

/** Static data for a text node */
declare interface TTextNode extends TNode {
    /** Index in the data[] array */
    index: number;
    child: null;
    /**
     * Text nodes will have parents unless they are the first node of a component or
     * embedded view (which means their parent is in a different view and must be
     * retrieved using LView.node).
     */
    parent: ɵangular_packages_core_core_be | TElementContainerNode | null;
    tViews: null;
    projection: null;
}

declare const TVIEW = 1;

/**
 * The static data for an LView (shared between all templates of a
 * given type).
 *
 * Stored on the `ComponentDef.tView`.
 */
declare interface TView {
    /**
     * Type of `TView` (`Root`|`Component`|`Embedded`).
     */
    type: TViewType;
    /**
     * ID for inline views to determine whether a view is the same as the previous view
     * in a certain position. If it's not, we know the new view needs to be inserted
     * and the one that exists needs to be removed (e.g. if/else statements)
     *
     * If this is -1, then this is a component view or a dynamically created view.
     */
    readonly id: number;
    /**
     * This is a blueprint used to generate LView instances for this TView. Copying this
     * blueprint is faster than creating a new LView from scratch.
     */
    blueprint: ɵangular_packages_core_core_bp;
    /**
     * The template function used to refresh the view of dynamically created views
     * and components. Will be null for inline views.
     */
    template: ComponentTemplate<{}> | null;
    /**
     * A function containing query-related instructions.
     */
    viewQuery: ViewQueriesFunction<{}> | null;
    /**
     * Pointer to the host `TNode` (not part of this TView).
     *
     * If this is a `TViewNode` for an `LViewNode`, this is an embedded view of a container.
     * We need this pointer to be able to efficiently find this node when inserting the view
     * into an anchor.
     *
     * If this is a `TElementNode`, this is the view of a root component. It has exactly one
     * root TNode.
     *
     * If this is null, this is the view of a component that is not at root. We do not store
     * the host TNodes for child component views because they can potentially have several
     * different host TNodes, depending on where the component is being used. These host
     * TNodes cannot be shared (due to different indices, etc).
     */
    node: TViewNode | ɵangular_packages_core_core_be | null;
    /** Whether or not this template has been processed in creation mode. */
    firstCreatePass: boolean;
    /**
     *  Whether or not this template has been processed in update mode (e.g. change detected)
     *
     * `firstUpdatePass` is used by styling to set up `TData` to contain metadata about the styling
     * instructions. (Mainly to build up a linked list of styling priority order.)
     *
     * Typically this function gets cleared after first execution. If exception is thrown then this
     * flag can remain turned un until there is first successful (no exception) pass. This means that
     * individual styling instructions keep track of if they have already been added to the linked
     * list to prevent double adding.
     */
    firstUpdatePass: boolean;
    /** Static data equivalent of LView.data[]. Contains TNodes, PipeDefInternal or TI18n. */
    data: TData;
    /**
     * The binding start index is the index at which the data array
     * starts to store bindings only. Saving this value ensures that we
     * will begin reading bindings at the correct point in the array when
     * we are in update mode.
     *
     * -1 means that it has not been initialized.
     */
    bindingStartIndex: number;
    /**
     * The index where the "expando" section of `LView` begins. The expando
     * section contains injectors, directive instances, and host binding values.
     * Unlike the "decls" and "vars" sections of `LView`, the length of this
     * section cannot be calculated at compile-time because directives are matched
     * at runtime to preserve locality.
     *
     * We store this start index so we know where to start checking host bindings
     * in `setHostBindings`.
     */
    expandoStartIndex: number;
    /**
     * Whether or not there are any static view queries tracked on this view.
     *
     * We store this so we know whether or not we should do a view query
     * refresh after creation mode to collect static query results.
     */
    staticViewQueries: boolean;
    /**
     * Whether or not there are any static content queries tracked on this view.
     *
     * We store this so we know whether or not we should do a content query
     * refresh after creation mode to collect static query results.
     */
    staticContentQueries: boolean;
    /**
     * A reference to the first child node located in the view.
     */
    firstChild: TNode | null;
    /**
     * Set of instructions used to process host bindings efficiently.
     *
     * See VIEW_DATA.md for more information.
     */
    expandoInstructions: ExpandoInstructions | null;
    /**
     * Full registry of directives and components that may be found in this view.
     *
     * It's necessary to keep a copy of the full def list on the TView so it's possible
     * to render template functions without a host component.
     */
    directiveRegistry: DirectiveDefList | null;
    /**
     * Full registry of pipes that may be found in this view.
     *
     * The property is either an array of `PipeDefs`s or a function which returns the array of
     * `PipeDefs`s. The function is necessary to be able to support forward declarations.
     *
     * It's necessary to keep a copy of the full def list on the TView so it's possible
     * to render template functions without a host component.
     */
    pipeRegistry: PipeDefList | null;
    /**
     * Array of ngOnInit, ngOnChanges and ngDoCheck hooks that should be executed for this view in
     * creation mode.
     *
     * Even indices: Directive index
     * Odd indices: Hook function
     */
    preOrderHooks: HookData | null;
    /**
     * Array of ngOnChanges and ngDoCheck hooks that should be executed for this view in update mode.
     *
     * Even indices: Directive index
     * Odd indices: Hook function
     */
    preOrderCheckHooks: HookData | null;
    /**
     * Array of ngAfterContentInit and ngAfterContentChecked hooks that should be executed
     * for this view in creation mode.
     *
     * Even indices: Directive index
     * Odd indices: Hook function
     */
    contentHooks: HookData | null;
    /**
     * Array of ngAfterContentChecked hooks that should be executed for this view in update
     * mode.
     *
     * Even indices: Directive index
     * Odd indices: Hook function
     */
    contentCheckHooks: HookData | null;
    /**
     * Array of ngAfterViewInit and ngAfterViewChecked hooks that should be executed for
     * this view in creation mode.
     *
     * Even indices: Directive index
     * Odd indices: Hook function
     */
    viewHooks: HookData | null;
    /**
     * Array of ngAfterViewChecked hooks that should be executed for this view in
     * update mode.
     *
     * Even indices: Directive index
     * Odd indices: Hook function
     */
    viewCheckHooks: HookData | null;
    /**
     * Array of ngOnDestroy hooks that should be executed when this view is destroyed.
     *
     * Even indices: Directive index
     * Odd indices: Hook function
     */
    destroyHooks: HookData | null;
    /**
     * When a view is destroyed, listeners need to be released and outputs need to be
     * unsubscribed. This cleanup array stores both listener data (in chunks of 4)
     * and output data (in chunks of 2) for a particular view. Combining the arrays
     * saves on memory (70 bytes per array) and on a few bytes of code size (for two
     * separate for loops).
     *
     * If it's a native DOM listener or output subscription being stored:
     * 1st index is: event name  `name = tView.cleanup[i+0]`
     * 2nd index is: index of native element or a function that retrieves global target (window,
     *               document or body) reference based on the native element:
     *    `typeof idxOrTargetGetter === 'function'`: global target getter function
     *    `typeof idxOrTargetGetter === 'number'`: index of native element
     *
     * 3rd index is: index of listener function `listener = lView[CLEANUP][tView.cleanup[i+2]]`
     * 4th index is: `useCaptureOrIndx = tView.cleanup[i+3]`
     *    `typeof useCaptureOrIndx == 'boolean' : useCapture boolean
     *    `typeof useCaptureOrIndx == 'number':
     *         `useCaptureOrIndx >= 0` `removeListener = LView[CLEANUP][useCaptureOrIndx]`
     *         `useCaptureOrIndx <  0` `subscription = LView[CLEANUP][-useCaptureOrIndx]`
     *
     * If it's an output subscription or query list destroy hook:
     * 1st index is: output unsubscribe function / query list destroy function
     * 2nd index is: index of function context in LView.cleanupInstances[]
     *               `tView.cleanup[i+0].call(lView[CLEANUP][tView.cleanup[i+1]])`
     */
    cleanup: any[] | null;
    /**
     * A list of element indices for child components that will need to be
     * refreshed when the current view has finished its check. These indices have
     * already been adjusted for the HEADER_OFFSET.
     *
     */
    components: number[] | null;
    /**
     * A collection of queries tracked in a given view.
     */
    queries: TQueries | null;
    /**
     * An array of indices pointing to directives with content queries alongside with the
     * corresponding
     * query index. Each entry in this array is a tuple of:
     * - index of the first content query index declared by a given directive;
     * - index of a directive.
     *
     * We are storing those indexes so we can refresh content queries as part of a view refresh
     * process.
     */
    contentQueries: number[] | null;
    /**
     * Set of schemas that declare elements to be allowed inside the view.
     */
    schemas: SchemaMetadata[] | null;
    /**
     * Array of constants for the view. Includes attribute arrays, local definition arrays etc.
     * Used for directive matching, attribute bindings, local definitions and more.
     */
    consts: TConstants | null;
}

/** Static data for a view  */
declare interface TViewNode extends TNode {
    /** If -1, it's a dynamically created view. Otherwise, it is the view block ID. */
    index: number;
    child: ɵangular_packages_core_core_be | TTextNode | TElementContainerNode | TContainerNode | TProjectionNode | null;
    parent: TContainerNode | null;
    tViews: null;
    projection: null;
}

/**
 * Explicitly marks `TView` as a specific type in `ngDevMode`
 *
 * It is useful to know conceptually what time of `TView` we are dealing with when
 * debugging an application (even if the runtime does not need it.) For this reason
 * we store this information in the `ngDevMode` `TView` and than use it for
 * better debugging experience.
 */
declare const enum TViewType {
    /**
     * Root `TView` is the used to bootstrap components into. It is used in conjunction with
     * `LView` which takes an existing DOM node not owned by Angular and wraps it in `TView`/`LView`
     * so that other components can be loaded into it.
     */
    Root = 0,
    /**
     * `TView` associated with a Component. This would be the `TView` directly associated with the
     * component view (as opposed an `Embedded` `TView` which would be a child of `Component` `TView`)
     */
    Component = 1,
    /**
     * `TView` associated with a template. Such as `*ngIf`, `<ng-template>` etc... A `Component`
     * can have zero or more `Embedede` `TView`s.
     */
    Embedded = 2
}

/**
 * Special location which allows easy identification of type. If we have an array which was
 * retrieved from the `LView` and that array has `true` at `TYPE` location, we know it is
 * `LContainer`.
 */
declare const TYPE = 1;

/**
 * @description
 *
 * Represents a type that a Component or other object is instances of.
 *
 * An example of a `Type` is `MyCustomComponent` class, which in JavaScript is be represented by
 * the `MyCustomComponent` constructor function.
 *
 * @publicApi
 */
export declare const Type: FunctionConstructor;

export declare interface Type<T> extends Function {
    new (...args: any[]): T;
}

/**
 * An interface implemented by all Angular type decorators, which allows them to be used as
 * decorators as well as Angular syntax.
 *
 * ```
 * @ng.Component({...})
 * class MyClass {...}
 * ```
 *
 * @publicApi
 */
export declare interface TypeDecorator {
    /**
     * Invoke as decorator.
     */
    <T extends Type<any>>(type: T): T;
    (target: Object, propertyKey?: string | symbol, parameterIndex?: number): void;
}

/**
 * Configures the `Injector` to return an instance of `Type` when `Type' is used as the token.
 *
 * Create an instance by invoking the `new` operator and supplying additional arguments.
 * This form is a short form of `TypeProvider`;
 *
 * For more details, see the ["Dependency Injection Guide"](guide/dependency-injection).
 *
 * @usageNotes
 *
 * {@example core/di/ts/provider_spec.ts region='TypeProvider'}
 *
 * @publicApi
 */
export declare interface TypeProvider extends Type<any> {
}

/**
 * Configures the `Injector` to return a value for a token.
 * @see ["Dependency Injection Guide"](guide/dependency-injection).
 *
 * @usageNotes
 *
 * ### Example
 *
 * {@example core/di/ts/provider_spec.ts region='ValueProvider'}
 *
 * ### Multi-value example
 *
 * {@example core/di/ts/provider_spec.ts region='MultiProviderAspect'}
 *
 * @publicApi
 */
export declare interface ValueProvider extends ValueSansProvider {
    /**
     * An injection token. Typically an instance of `Type` or `InjectionToken`, but can be `any`.
     */
    provide: any;
    /**
     * When true, injector returns an array of instances. This is useful to allow multiple
     * providers spread across many files to provide configuration information to a common token.
     */
    multi?: boolean;
}

/**
 * Configures the `Injector` to return a value for a token.
 * Base for `ValueProvider` decorator.
 *
 * @publicApi
 */
export declare interface ValueSansProvider {
    /**
     * The value to inject.
     */
    useValue: any;
}

/**
 * @publicApi
 */
export declare const VERSION: Version;


/**
 * @description Represents the version of Angular
 *
 * @publicApi
 */
export declare class Version {
    full: string;
    readonly major: string;
    readonly minor: string;
    readonly patch: string;
    constructor(full: string);
}

declare const VIEW_REFS = 8;

/**
 * Type of the ViewChild metadata.
 *
 * @publicApi
 */
export declare type ViewChild = Query;

/**
 * ViewChild decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const ViewChild: ViewChildDecorator;

/**
 * Type of the ViewChild decorator / constructor function.
 *
 * @see `ViewChild`.
 * @publicApi
 */
export declare interface ViewChildDecorator {
    /**
     * @description
     * Property decorator that configures a view query.
     * The change detector looks for the first element or the directive matching the selector
     * in the view DOM. If the view DOM changes, and a new child matches the selector,
     * the property is updated.
     *
     * View queries are set before the `ngAfterViewInit` callback is called.
     *
     * **Metadata Properties**:
     *
     * * **selector** - The directive type or the name used for querying.
     * * **read** - True to read a different token from the queried elements.
     * * **static** - True to resolve query results before change detection runs,
     * false to resolve after change detection. Defaults to false.
     *
     *
     * The following selectors are supported.
     *   * Any class with the `@Component` or `@Directive` decorator
     *   * A template reference variable as a string (e.g. query `<my-component #cmp></my-component>`
     * with `@ViewChild('cmp')`)
     *   * Any provider defined in the child component tree of the current component (e.g.
     * `@ViewChild(SomeService) someService: SomeService`)
     *   * Any provider defined through a string token (e.g. `@ViewChild('someToken') someTokenVal:
     * any`)
     *   * A `TemplateRef` (e.g. query `<ng-template></ng-template>` with `@ViewChild(TemplateRef)
     * template;`)
     *
     * @usageNotes
     *
     * {@example core/di/ts/viewChild/view_child_example.ts region='Component'}
     *
     * ### Example 2
     *
     * {@example core/di/ts/viewChild/view_child_howto.ts region='HowTo'}
     *
     * @Annotation
     */
    (selector: Type<any> | Function | string, opts?: {
        read?: any;
        static?: boolean;
    }): any;
    new (selector: Type<any> | Function | string, opts?: {
        read?: any;
        static?: boolean;
    }): ViewChild;
}

/**
 * Type of the ViewChildren metadata.
 *
 * @publicApi
 */
export declare type ViewChildren = Query;

/**
 * ViewChildren decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const ViewChildren: ViewChildrenDecorator;

/**
 * Type of the ViewChildren decorator / constructor function.
 *
 * @see `ViewChildren`.
 *
 * @publicApi
 */
export declare interface ViewChildrenDecorator {
    /**
     * Parameter decorator that configures a view query.
     *
     * Use to get the `QueryList` of elements or directives from the view DOM.
     * Any time a child element is added, removed, or moved, the query list will be updated,
     * and the changes observable of the query list will emit a new value.
     *
     * View queries are set before the `ngAfterViewInit` callback is called.
     *
     * **Metadata Properties**:
     *
     * * **selector** - The directive type or the name used for querying.
     * * **read** - True to read a different token from the queried elements.
     *
     * @usageNotes
     *
     * {@example core/di/ts/viewChildren/view_children_howto.ts region='HowTo'}
     *
     * ### Another example
     *
     * {@example core/di/ts/viewChildren/view_children_example.ts region='Component'}
     *
     * @Annotation
     */
    (selector: Type<any> | Function | string, opts?: {
        read?: any;
    }): any;
    new (selector: Type<any> | Function | string, opts?: {
        read?: any;
    }): ViewChildren;
}

/**
 * Represents a container where one or more views can be attached to a component.
 *
 * Can contain *host views* (created by instantiating a
 * component with the `createComponent()` method), and *embedded views*
 * (created by instantiating a `TemplateRef` with the `createEmbeddedView()` method).
 *
 * A view container instance can contain other view containers,
 * creating a [view hierarchy](guide/glossary#view-tree).
 *
 * @see `ComponentRef`
 * @see `EmbeddedViewRef`
 *
 * @publicApi
 */
export declare abstract class ViewContainerRef {
    /**
     * Anchor element that specifies the location of this container in the containing view.
     * Each view container can have only one anchor element, and each anchor element
     * can have only a single view container.
     *
     * Root elements of views attached to this container become siblings of the anchor element in
     * the rendered view.
     *
     * Access the `ViewContainerRef` of an element by placing a `Directive` injected
     * with `ViewContainerRef` on the element, or use a `ViewChild` query.
     *
     * <!-- TODO: rename to anchorElement -->
     */
    abstract get element(): ElementRef;
    /**
     * The [dependency injector](guide/glossary#injector) for this view container.
     */
    abstract get injector(): Injector;
    /** @deprecated No replacement */
    abstract get parentInjector(): Injector;
    /**
     * Destroys all views in this container.
     */
    abstract clear(): void;
    /**
     * Retrieves a view from this container.
     * @param index The 0-based index of the view to retrieve.
     * @returns The `ViewRef` instance, or null if the index is out of range.
     */
    abstract get(index: number): ViewRef | null;
    /**
     * Reports how many views are currently attached to this container.
     * @returns The number of views.
     */
    abstract get length(): number;
    /**
     * Instantiates an embedded view and inserts it
     * into this container.
     * @param templateRef The HTML template that defines the view.
     * @param index The 0-based index at which to insert the new view into this container.
     * If not specified, appends the new view as the last entry.
     *
     * @returns The `ViewRef` instance for the newly created view.
     */
    abstract createEmbeddedView<C>(templateRef: TemplateRef<C>, context?: C, index?: number): EmbeddedViewRef<C>;
    /**
     * Instantiates a single component and inserts its host view into this container.
     *
     * @param componentFactory The factory to use.
     * @param index The index at which to insert the new component's host view into this container.
     * If not specified, appends the new view as the last entry.
     * @param injector The injector to use as the parent for the new component.
     * @param projectableNodes
     * @param ngModule
     *
     * @returns The new component instance, containing the host view.
     *
     */
    abstract createComponent<C>(componentFactory: ComponentFactory<C>, index?: number, injector?: Injector, projectableNodes?: any[][], ngModule?: NgModuleRef<any>): ComponentRef<C>;
    /**
     * Inserts a view into this container.
     * @param viewRef The view to insert.
     * @param index The 0-based index at which to insert the view.
     * If not specified, appends the new view as the last entry.
     * @returns The inserted `ViewRef` instance.
     *
     */
    abstract insert(viewRef: ViewRef, index?: number): ViewRef;
    /**
     * Moves a view to a new location in this container.
     * @param viewRef The view to move.
     * @param index The 0-based index of the new location.
     * @returns The moved `ViewRef` instance.
     */
    abstract move(viewRef: ViewRef, currentIndex: number): ViewRef;
    /**
     * Returns the index of a view within the current container.
     * @param viewRef The view to query.
     * @returns The 0-based index of the view's position in this container,
     * or `-1` if this container doesn't contain the view.
     */
    abstract indexOf(viewRef: ViewRef): number;
    /**
     * Destroys a view attached to this container
     * @param index The 0-based index of the view to destroy.
     * If not specified, the last view in the container is removed.
     */
    abstract remove(index?: number): void;
    /**
     * Detaches a view from this container without destroying it.
     * Use along with `insert()` to move a view within the current container.
     * @param index The 0-based index of the view to detach.
     * If not specified, the last view in the container is detached.
     */
    abstract detach(index?: number): ViewRef | null;
}

/**
 * View instance data.
 * Attention: Adding fields to this is performance sensitive!
 */
declare interface ViewData {
    def: ɵViewDefinition;
    root: RootData;
    renderer: Renderer2;
    parentNodeDef: NodeDef | null;
    parent: ViewData | null;
    viewContainerParent: ViewData | null;
    component: any;
    context: any;
    nodes: {
        [key: number]: NodeData;
    };
    state: ViewState;
    oldValues: any[];
    disposables: DisposableFn[] | null;
    initIndex: number;
}

declare interface ViewDefinitionFactory extends DefinitionFactory<ɵViewDefinition> {
}


/**
 * Defines template and style encapsulation options available for Component's {@link Component}.
 *
 * See {@link Component#encapsulation encapsulation}.
 *
 * @usageNotes
 * ### Example
 *
 * {@example core/ts/metadata/encapsulation.ts region='longform'}
 *
 * @publicApi
 */
export declare enum ViewEncapsulation {
    /**
     * Emulate `Native` scoping of styles by adding an attribute containing surrogate id to the Host
     * Element and pre-processing the style rules provided via {@link Component#styles styles} or
     * {@link Component#styleUrls styleUrls}, and adding the new Host Element attribute to all
     * selectors.
     *
     * This is the default option.
     */
    Emulated = 0,
    /**
     * @deprecated v6.1.0 - use {ViewEncapsulation.ShadowDom} instead.
     * Use the native encapsulation mechanism of the renderer.
     *
     * For the DOM this means using the deprecated [Shadow DOM
     * v0](https://w3c.github.io/webcomponents/spec/shadow/) and
     * creating a ShadowRoot for Component's Host Element.
     */
    Native = 1,
    /**
     * Don't provide any template or style encapsulation.
     */
    None = 2,
    /**
     * Use Shadow DOM to encapsulate styles.
     *
     * For the DOM this means using modern [Shadow
     * DOM](https://w3c.github.io/webcomponents/spec/shadow/) and
     * creating a ShadowRoot for Component's Host Element.
     */
    ShadowDom = 3
}

declare interface viewEngine_ChangeDetectorRef_interface extends ChangeDetectorRef {
}

declare interface ViewHandleEventFn {
    (view: ViewData, nodeIndex: number, eventName: string, event: any): boolean;
}

/**
 * Definition of what a view queries function should look like.
 */
declare type ViewQueriesFunction<T> = <U extends T>(rf: ɵRenderFlags, ctx: U) => void;

/**
 * Represents an Angular [view](guide/glossary#view),
 * specifically the [host view](guide/glossary#view-tree) that is defined by a component.
 * Also serves as the base class
 * that adds destroy methods for [embedded views](guide/glossary#view-tree).
 *
 * @see `EmbeddedViewRef`
 *
 * @publicApi
 */
export declare abstract class ViewRef extends ChangeDetectorRef {
    /**
     * Destroys this view and all of the data structures associated with it.
     */
    abstract destroy(): void;
    /**
     * Reports whether this view has been destroyed.
     * @returns True after the `destroy()` method has been called, false otherwise.
     */
    abstract get destroyed(): boolean;
    /**
     * A lifecycle hook that provides additional developer-defined cleanup
     * functionality for views.
     * @param callback A handler function that cleans up developer-defined data
     * associated with a view. Called when the `destroy()` method is invoked.
     */
    abstract onDestroy(callback: Function): any /** TODO #9100 */;
}

declare class ViewRef_2<T> implements EmbeddedViewRef<T>, InternalViewRef, viewEngine_ChangeDetectorRef_interface {
    /**
     * This represents the `LView` associated with the point where `ChangeDetectorRef` was
     * requested.
     *
     * This may be different from `_lView` if the `_cdRefInjectingView` is an embedded view.
     */
    private _cdRefInjectingView?;
    private _appRef;
    private _viewContainerRef;
    get rootNodes(): any[];
    constructor(
    /**
     * This represents `LView` associated with the component when ViewRef is a ChangeDetectorRef.
     *
     * When ViewRef is created for a dynamic component, this also represents the `LView` for the
     * component.
     *
     * For a "regular" ViewRef created for an embedded view, this is the `LView` for the embedded
     * view.
     *
     * @internal
     */
    _lView: ɵangular_packages_core_core_bp, 
    /**
     * This represents the `LView` associated with the point where `ChangeDetectorRef` was
     * requested.
     *
     * This may be different from `_lView` if the `_cdRefInjectingView` is an embedded view.
     */
    _cdRefInjectingView?: ɵangular_packages_core_core_bp | undefined);
    get context(): T;
    get destroyed(): boolean;
    destroy(): void;
    onDestroy(callback: Function): void;
    /**
     * Marks a view and all of its ancestors dirty.
     *
     * It also triggers change detection by calling `scheduleTick` internally, which coalesces
     * multiple `markForCheck` calls to into one change detection run.
     *
     * This can be used to ensure an {@link ChangeDetectionStrategy#OnPush OnPush} component is
     * checked when it needs to be re-rendered but the two normal triggers haven't marked it
     * dirty (i.e. inputs haven't changed and events haven't fired in the view).
     *
     * <!-- TODO: Add a link to a chapter on OnPush components -->
     *
     * @usageNotes
     * ### Example
     *
     * ```typescript
     * @Component({
     *   selector: 'my-app',
     *   template: `Number of ticks: {{numberOfTicks}}`
     *   changeDetection: ChangeDetectionStrategy.OnPush,
     * })
     * class AppComponent {
     *   numberOfTicks = 0;
     *
     *   constructor(private ref: ChangeDetectorRef) {
     *     setInterval(() => {
     *       this.numberOfTicks++;
     *       // the following is required, otherwise the view will not be updated
     *       this.ref.markForCheck();
     *     }, 1000);
     *   }
     * }
     * ```
     */
    markForCheck(): void;
    /**
     * Detaches the view from the change detection tree.
     *
     * Detached views will not be checked during change detection runs until they are
     * re-attached, even if they are dirty. `detach` can be used in combination with
     * {@link ChangeDetectorRef#detectChanges detectChanges} to implement local change
     * detection checks.
     *
     * <!-- TODO: Add a link to a chapter on detach/reattach/local digest -->
     * <!-- TODO: Add a live demo once ref.detectChanges is merged into master -->
     *
     * @usageNotes
     * ### Example
     *
     * The following example defines a component with a large list of readonly data.
     * Imagine the data changes constantly, many times per second. For performance reasons,
     * we want to check and update the list every five seconds. We can do that by detaching
     * the component's change detector and doing a local check every five seconds.
     *
     * ```typescript
     * class DataProvider {
     *   // in a real application the returned data will be different every time
     *   get data() {
     *     return [1,2,3,4,5];
     *   }
     * }
     *
     * @Component({
     *   selector: 'giant-list',
     *   template: `
     *     <li *ngFor="let d of dataProvider.data">Data {{d}}</li>
     *   `,
     * })
     * class GiantList {
     *   constructor(private ref: ChangeDetectorRef, private dataProvider: DataProvider) {
     *     ref.detach();
     *     setInterval(() => {
     *       this.ref.detectChanges();
     *     }, 5000);
     *   }
     * }
     *
     * @Component({
     *   selector: 'app',
     *   providers: [DataProvider],
     *   template: `
     *     <giant-list><giant-list>
     *   `,
     * })
     * class App {
     * }
     * ```
     */
    detach(): void;
    /**
     * Re-attaches a view to the change detection tree.
     *
     * This can be used to re-attach views that were previously detached from the tree
     * using {@link ChangeDetectorRef#detach detach}. Views are attached to the tree by default.
     *
     * <!-- TODO: Add a link to a chapter on detach/reattach/local digest -->
     *
     * @usageNotes
     * ### Example
     *
     * The following example creates a component displaying `live` data. The component will detach
     * its change detector from the main change detector tree when the component's live property
     * is set to false.
     *
     * ```typescript
     * class DataProvider {
     *   data = 1;
     *
     *   constructor() {
     *     setInterval(() => {
     *       this.data = this.data * 2;
     *     }, 500);
     *   }
     * }
     *
     * @Component({
     *   selector: 'live-data',
     *   inputs: ['live'],
     *   template: 'Data: {{dataProvider.data}}'
     * })
     * class LiveData {
     *   constructor(private ref: ChangeDetectorRef, private dataProvider: DataProvider) {}
     *
     *   set live(value) {
     *     if (value) {
     *       this.ref.reattach();
     *     } else {
     *       this.ref.detach();
     *     }
     *   }
     * }
     *
     * @Component({
     *   selector: 'my-app',
     *   providers: [DataProvider],
     *   template: `
     *     Live Update: <input type="checkbox" [(ngModel)]="live">
     *     <live-data [live]="live"><live-data>
     *   `,
     * })
     * class AppComponent {
     *   live = true;
     * }
     * ```
     */
    reattach(): void;
    /**
     * Checks the view and its children.
     *
     * This can also be used in combination with {@link ChangeDetectorRef#detach detach} to implement
     * local change detection checks.
     *
     * <!-- TODO: Add a link to a chapter on detach/reattach/local digest -->
     * <!-- TODO: Add a live demo once ref.detectChanges is merged into master -->
     *
     * @usageNotes
     * ### Example
     *
     * The following example defines a component with a large list of readonly data.
     * Imagine, the data changes constantly, many times per second. For performance reasons,
     * we want to check and update the list every five seconds.
     *
     * We can do that by detaching the component's change detector and doing a local change detection
     * check every five seconds.
     *
     * See {@link ChangeDetectorRef#detach detach} for more information.
     */
    detectChanges(): void;
    /**
     * Checks the change detector and its children, and throws if any changes are detected.
     *
     * This is used in development mode to verify that running change detection doesn't
     * introduce other changes.
     */
    checkNoChanges(): void;
    attachToViewContainerRef(vcRef: ViewContainerRef): void;
    detachFromAppRef(): void;
    attachToAppRef(appRef: ApplicationRef): void;
}

/**
 * Bitmask of states
 */
declare const enum ViewState {
    BeforeFirstCheck = 1,
    FirstCheck = 2,
    Attached = 4,
    ChecksEnabled = 8,
    IsProjectedView = 16,
    CheckProjectedView = 32,
    CheckProjectedViews = 64,
    Destroyed = 128,
    InitState_Mask = 1792,
    InitState_BeforeInit = 0,
    InitState_CallingOnInit = 256,
    InitState_CallingAfterContentInit = 512,
    InitState_CallingAfterViewInit = 768,
    InitState_AfterInit = 1024,
    CatDetectChanges = 12,
    CatInit = 13
}

declare interface ViewUpdateFn {
    (check: NodeCheckFn, view: ViewData): void;
}

/**
 * Indicates that the result of a {@link Pipe} transformation has changed even though the
 * reference has not changed.
 *
 * Wrapped values are unwrapped automatically during the change detection, and the unwrapped value
 * is stored.
 *
 * Example:
 *
 * ```
 * if (this._latestValue === this._latestReturnedValue) {
 *    return this._latestReturnedValue;
 *  } else {
 *    this._latestReturnedValue = this._latestValue;
 *    return WrappedValue.wrap(this._latestValue); // this will force update
 *  }
 * ```
 *
 * @publicApi
 */
export declare class WrappedValue {
    /** @deprecated from 5.3, use `unwrap()` instead - will switch to protected */
    wrapped: any;
    constructor(value: any);
    /** Creates a wrapped value. */
    static wrap(value: any): WrappedValue;
    /**
     * Returns the underlying value of a wrapped value.
     * Returns the given `value` when it is not wrapped.
     **/
    static unwrap(value: any): any;
    /** Returns true if `value` is a wrapped value. */
    static isWrapped(value: any): value is WrappedValue;
}

/**
 * Sanitizes the given unsafe, untrusted HTML fragment, and returns HTML text that is safe to add to
 * the DOM in a browser environment.
 */
export declare function ɵ_sanitizeHtml(defaultDoc: any, unsafeHtmlInput: string): string;

/**
 * Sanitizes the given untrusted CSS style property value (i.e. not an entire object, just a single
 * value) and returns a value that is safe to use in a browser environment.
 */
export declare function ɵ_sanitizeStyle(value: string): string;


export declare function ɵ_sanitizeUrl(url: string): string;

export declare const ɵALLOW_MULTIPLE_PLATFORMS: InjectionToken<boolean>;

export declare function ɵallowSanitizationBypassAndThrow(value: any, type: ɵBypassType.Html): value is ɵSafeHtml;

export declare function ɵallowSanitizationBypassAndThrow(value: any, type: ɵBypassType.ResourceUrl): value is ɵSafeResourceUrl;

export declare function ɵallowSanitizationBypassAndThrow(value: any, type: ɵBypassType.Script): value is ɵSafeScript;

export declare function ɵallowSanitizationBypassAndThrow(value: any, type: ɵBypassType.Style): value is ɵSafeStyle;

export declare function ɵallowSanitizationBypassAndThrow(value: any, type: ɵBypassType.Url): value is ɵSafeUrl;

export declare function ɵallowSanitizationBypassAndThrow(value: any, type: ɵBypassType): boolean;

export declare function ɵand(flags: ɵNodeFlags, matchedQueriesDsl: null | [string | number, ɵQueryValueType][], ngContentIndex: null | number, childCount: number, handleEvent?: null | ElementHandleEventFn, templateFactory?: ViewDefinitionFactory): NodeDef;

/** Checks whether a function is wrapped by a `forwardRef`. */
export declare function ɵangular_packages_core_core_a(fn: any): fn is () => any;

export declare function ɵangular_packages_core_core_b<T>(token: Type<T> | InjectionToken<T>): T;

export declare function ɵangular_packages_core_core_b<T>(token: Type<T> | InjectionToken<T>, flags?: InjectFlags): T | null;

/**
 * Return the current `LView`.
 */
export declare function ɵangular_packages_core_core_ba(): ɵangular_packages_core_core_bp;

export declare function ɵangular_packages_core_core_bb(): TNode;

export declare function ɵangular_packages_core_core_bc(): number;

export declare function ɵangular_packages_core_core_bd<T = any>(level: number): T;

/** Static data for an element  */
export declare interface ɵangular_packages_core_core_be extends TNode {
    /** Index in the data[] array */
    index: number;
    child: ɵangular_packages_core_core_be | TTextNode | TElementContainerNode | TContainerNode | TProjectionNode | null;
    /**
     * Element nodes will have parents unless they are the first node of a component or
     * embedded view (which means their parent is in a different view and must be
     * retrieved using viewData[HOST_NODE]).
     */
    parent: ɵangular_packages_core_core_be | TElementContainerNode | null;
    tViews: null;
    /**
     * If this is a component TNode with projection, this will be an array of projected
     * TNodes or native nodes (see TNode.projection for more info). If it's a regular element node or
     * a component without projection, it will be null.
     */
    projection: (TNode | RNode[])[] | null;
}

/**
 * If the value of the provided exp has changed, calls the pure function to return
 * an updated value. Or if the value has not changed, returns cached value.
 *
 * @param lView LView in which the function is being executed.
 * @param bindingRoot Binding root index.
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn Function that returns an updated value
 * @param exp Updated expression value
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 */
export declare function ɵangular_packages_core_core_bf(lView: ɵangular_packages_core_core_bp, bindingRoot: number, slotOffset: number, pureFn: (v: any) => any, exp: any, thisArg?: any): any;

/**
 * If the value of any provided exp has changed, calls the pure function to return
 * an updated value. Or if no values have changed, returns cached value.
 *
 * @param lView LView in which the function is being executed.
 * @param bindingRoot Binding root index.
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn
 * @param exp1
 * @param exp2
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 */
export declare function ɵangular_packages_core_core_bg(lView: ɵangular_packages_core_core_bp, bindingRoot: number, slotOffset: number, pureFn: (v1: any, v2: any) => any, exp1: any, exp2: any, thisArg?: any): any;

/**
 * If the value of any provided exp has changed, calls the pure function to return
 * an updated value. Or if no values have changed, returns cached value.
 *
 * @param lView LView in which the function is being executed.
 * @param bindingRoot Binding root index.
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn
 * @param exp1
 * @param exp2
 * @param exp3
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 */
export declare function ɵangular_packages_core_core_bh(lView: ɵangular_packages_core_core_bp, bindingRoot: number, slotOffset: number, pureFn: (v1: any, v2: any, v3: any) => any, exp1: any, exp2: any, exp3: any, thisArg?: any): any;

/**
 * If the value of any provided exp has changed, calls the pure function to return
 * an updated value. Or if no values have changed, returns cached value.
 *
 * @param lView LView in which the function is being executed.
 * @param bindingRoot Binding root index.
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn
 * @param exp1
 * @param exp2
 * @param exp3
 * @param exp4
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 *
 */
export declare function ɵangular_packages_core_core_bi(lView: ɵangular_packages_core_core_bp, bindingRoot: number, slotOffset: number, pureFn: (v1: any, v2: any, v3: any, v4: any) => any, exp1: any, exp2: any, exp3: any, exp4: any, thisArg?: any): any;

/**
 * pureFunction instruction that can support any number of bindings.
 *
 * If the value of any provided exp has changed, calls the pure function to return
 * an updated value. Or if no values have changed, returns cached value.
 *
 * @param lView LView in which the function is being executed.
 * @param bindingRoot Binding root index.
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn A pure function that takes binding values and builds an object or array
 * containing those values.
 * @param exps An array of binding values
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 */
export declare function ɵangular_packages_core_core_bj(lView: ɵangular_packages_core_core_bp, bindingRoot: number, slotOffset: number, pureFn: (...v: any[]) => any, exps: any[], thisArg?: any): any;

/**
 * Detects which sanitizer to use for URL property, based on tag name and prop name.
 *
 * The rules are based on the RESOURCE_URL context config from
 * `packages/compiler/src/schema/dom_security_schema.ts`.
 * If tag and prop names don't match Resource URL schema, use URL sanitizer.
 */
export declare function ɵangular_packages_core_core_bk(tag: string, prop: string): typeof ɵɵsanitizeResourceUrl;

export declare function ɵangular_packages_core_core_bl(name: string, props?: (...args: any[]) => any, parentClass?: any): any;

export declare function ɵangular_packages_core_core_bm(name: string, props?: (...args: any[]) => any, parentClass?: any, additionalProcessing?: (target: any, name: string, ...args: any[]) => void): any;


export declare function ɵangular_packages_core_core_bn<T>(objWithPropertyToExtract: T): string;


/**
 * Convince closure compiler that the wrapped function has no side-effects.
 *
 * Closure compiler always assumes that `toString` has no side-effects. We use this quirk to
 * allow us to execute a function but have closure compiler mark the call as no-side-effects.
 * It is important that the return value for the `noSideEffects` function be assigned
 * to something which is retained otherwise the call to `noSideEffects` will be removed by closure
 * compiler.
 */
export declare function ɵangular_packages_core_core_bo<T>(fn: () => T): T;

/**
 * `LView` stores all of the information needed to process the instructions as
 * they are invoked from the template. Each embedded view and component view has its
 * own `LView`. When processing a particular view, we set the `viewData` to that
 * `LView`. When that view is done processing, the `viewData` is set back to
 * whatever the original `viewData` was before (the parent `LView`).
 *
 * Keeping separate state for each view facilities view insertion / deletion, so we
 * don't have to edit the data array based on which views are present.
 */
export declare interface ɵangular_packages_core_core_bp extends Array<any> {
    /**
     * The host node for this LView instance, if this is a component view.
     * If this is an embedded view, HOST will be null.
     */
    [HOST]: RElement | null;
    /**
     * The static data for this view. We need a reference to this so we can easily walk up the
     * node tree in DI and get the TView.data array associated with a node (where the
     * directive defs are stored).
     */
    readonly [TVIEW]: TView;
    /** Flags for this view. See LViewFlags for more info. */
    [FLAGS]: LViewFlags;
    /**
     * This may store an {@link LView} or {@link LContainer}.
     *
     * `LView` - The parent view. This is needed when we exit the view and must restore the previous
     * LView. Without this, the render method would have to keep a stack of
     * views as it is recursively rendering templates.
     *
     * `LContainer` - The current view is part of a container, and is an embedded view.
     */
    [PARENT]: ɵangular_packages_core_core_bp | LContainer | null;
    /**
     *
     * The next sibling LView or LContainer.
     *
     * Allows us to propagate between sibling view states that aren't in the same
     * container. Embedded views already have a node.next, but it is only set for
     * views in the same container. We need a way to link component views and views
     * across containers as well.
     */
    [NEXT]: ɵangular_packages_core_core_bp | LContainer | null;
    /** Queries active for this view - nodes from a view are reported to those queries. */
    [QUERIES]: LQueries | null;
    /**
     * Pointer to the `TViewNode` or `TElementNode` which represents the root of the view.
     *
     * If `TViewNode`, this is an embedded view of a container. We need this to be able to
     * efficiently find the `LViewNode` when inserting the view into an anchor.
     *
     * If `TElementNode`, this is the LView of a component.
     *
     * If null, this is the root view of an application (root component is in this view).
     */
    [T_HOST]: TViewNode | ɵangular_packages_core_core_be | null;
    /**
     * When a view is destroyed, listeners need to be released and outputs need to be
     * unsubscribed. This context array stores both listener functions wrapped with
     * their context and output subscription instances for a particular view.
     *
     * These change per LView instance, so they cannot be stored on TView. Instead,
     * TView.cleanup saves an index to the necessary context in this array.
     */
    [CLEANUP]: any[] | null;
    /**
     * - For dynamic views, this is the context with which to render the template (e.g.
     *   `NgForContext`), or `{}` if not defined explicitly.
     * - For root view of the root component the context contains change detection data.
     * - For non-root components, the context is the component instance,
     * - For inline views, the context is null.
     */
    [CONTEXT]: {} | RootContext | null;
    /** An optional Module Injector to be used as fall back after Element Injectors are consulted. */
    readonly [INJECTOR_2]: Injector | null;
    /** Factory to be used for creating Renderer. */
    [RENDERER_FACTORY]: RendererFactory3;
    /** Renderer to be used for this view. */
    [RENDERER]: Renderer3;
    /** An optional custom sanitizer. */
    [SANITIZER]: Sanitizer | null;
    /**
     * Reference to the first LView or LContainer beneath this LView in
     * the hierarchy.
     *
     * Necessary to store this so views can traverse through their nested views
     * to remove listeners and call onDestroy callbacks.
     */
    [CHILD_HEAD]: ɵangular_packages_core_core_bp | LContainer | null;
    /**
     * The last LView or LContainer beneath this LView in the hierarchy.
     *
     * The tail allows us to quickly add a new state to the end of the view list
     * without having to propagate starting from the first child.
     */
    [CHILD_TAIL]: ɵangular_packages_core_core_bp | LContainer | null;
    /**
     * View where this view's template was declared.
     *
     * Only applicable for dynamically created views. Will be null for inline/component views.
     *
     * The template for a dynamically created view may be declared in a different view than
     * it is inserted. We already track the "insertion view" (view where the template was
     * inserted) in LView[PARENT], but we also need access to the "declaration view"
     * (view where the template was declared). Otherwise, we wouldn't be able to call the
     * view's template function with the proper contexts. Context should be inherited from
     * the declaration view tree, not the insertion view tree.
     *
     * Example (AppComponent template):
     *
     * <ng-template #foo></ng-template>       <-- declared here -->
     * <some-comp [tpl]="foo"></some-comp>    <-- inserted inside this component -->
     *
     * The <ng-template> above is declared in the AppComponent template, but it will be passed into
     * SomeComp and inserted there. In this case, the declaration view would be the AppComponent,
     * but the insertion view would be SomeComp. When we are removing views, we would want to
     * traverse through the insertion view to clean up listeners. When we are calling the
     * template function during change detection, we need the declaration view to get inherited
     * context.
     */
    [DECLARATION_VIEW]: ɵangular_packages_core_core_bp | null;
    /**
     * Points to the declaration component view, used to track transplanted `LView`s.
     *
     * See: `DECLARATION_VIEW` which points to the actual `LView` where it was declared, whereas
     * `DECLARATION_COMPONENT_VIEW` points to the component which may not be same as
     * `DECLARATION_VIEW`.
     *
     * Example:
     * ```
     * <#VIEW #myComp>
     *  <div *ngIf="true">
     *   <ng-template #myTmpl>...</ng-template>
     *  </div>
     * </#VIEW>
     * ```
     * In the above case `DECLARATION_VIEW` for `myTmpl` points to the `LView` of `ngIf` whereas
     * `DECLARATION_COMPONENT_VIEW` points to `LView` of the `myComp` which owns the template.
     *
     * The reason for this is that all embedded views are always check-always whereas the component
     * view can be check-always or on-push. When we have a transplanted view it is important to
     * determine if we have transplanted a view from check-always declaration to on-push insertion
     * point. In such a case the transplanted view needs to be added to the `LContainer` in the
     * declared `LView` and CD during the declared view CD (in addition to the CD at the insertion
     * point.) (Any transplanted views which are intra Component are of no interest because the CD
     * strategy of declaration and insertion will always be the same, because it is the same
     * component.)
     *
     * Queries already track moved views in `LView[DECLARATION_LCONTAINER]` and
     * `LContainer[MOVED_VIEWS]`. However the queries also track `LView`s which moved within the same
     * component `LView`. Transplanted views are a subset of moved views, and we use
     * `DECLARATION_COMPONENT_VIEW` to differentiate them. As in this example.
     *
     * Example showing intra component `LView` movement.
     * ```
     * <#VIEW #myComp>
     *   <div *ngIf="condition; then thenBlock else elseBlock"></div>
     *   <ng-template #thenBlock>Content to render when condition is true.</ng-template>
     *   <ng-template #elseBlock>Content to render when condition is false.</ng-template>
     * </#VIEW>
     * ```
     * The `thenBlock` and `elseBlock` is moved but not transplanted.
     *
     * Example showing inter component `LView` movement (transplanted view).
     * ```
     * <#VIEW #myComp>
     *   <ng-template #myTmpl>...</ng-template>
     *   <insertion-component [template]="myTmpl"></insertion-component>
     * </#VIEW>
     * ```
     * In the above example `myTmpl` is passed into a different component. If `insertion-component`
     * instantiates `myTmpl` and `insertion-component` is on-push then the `LContainer` needs to be
     * marked as containing transplanted views and those views need to be CD as part of the
     * declaration CD.
     *
     *
     * When change detection runs, it iterates over `[MOVED_VIEWS]` and CDs any child `LView`s where
     * the `DECLARATION_COMPONENT_VIEW` of the current component and the child `LView` does not match
     * (it has been transplanted across components.)
     *
     * Note: `[DECLARATION_COMPONENT_VIEW]` points to itself if the LView is a component view (the
     *       simplest / most common case).
     *
     * see also:
     *   - https://hackmd.io/@mhevery/rJUJsvv9H write up of the problem
     *   - `LContainer[ACTIVE_INDEX]` for flag which marks which `LContainer` has transplanted views.
     *   - `LContainer[TRANSPLANT_HEAD]` and `LContainer[TRANSPLANT_TAIL]` storage for transplanted
     *   - `LView[DECLARATION_LCONTAINER]` similar problem for queries
     *   - `LContainer[MOVED_VIEWS]` similar problem for queries
     */
    [DECLARATION_COMPONENT_VIEW]: ɵangular_packages_core_core_bp;
    /**
     * A declaration point of embedded views (ones instantiated based on the content of a
     * <ng-template>), null for other types of views.
     *
     * We need to track all embedded views created from a given declaration point so we can prepare
     * query matches in a proper order (query matches are ordered based on their declaration point and
     * _not_ the insertion point).
     */
    [DECLARATION_LCONTAINER]: LContainer | null;
    /**
     * More flags for this view. See PreOrderHookFlags for more info.
     */
    [PREORDER_HOOK_FLAGS]: PreOrderHookFlags;
}

/**
 * Returns the `RootContext` instance that is associated with
 * the application where the target is situated. It does this by walking the parent views until it
 * gets to the root view, then getting the context off of that.
 *
 * @param viewOrComponent the `LView` or component to get the root context for.
 */
export declare function ɵangular_packages_core_core_bq(viewOrComponent: ɵangular_packages_core_core_bp | {}): RootContext;

export declare class ɵangular_packages_core_core_c implements Injector {
    get(token: any, notFoundValue?: any): any;
}

export declare class ɵangular_packages_core_core_d implements ReflectiveInjector {
    private static INJECTOR_KEY;
    readonly parent: Injector | null;
    keyIds: number[];
    objs: any[];
    /**
     * Private
     */
    constructor(_providers: ResolvedReflectiveProvider[], _parent?: Injector);
    get(token: any, notFoundValue?: any): any;
    resolveAndCreateChild(providers: Provider[]): ReflectiveInjector;
    createChildFromResolved(providers: ResolvedReflectiveProvider[]): ReflectiveInjector;
    resolveAndInstantiate(provider: Provider): any;
    instantiateResolved(provider: ResolvedReflectiveProvider): any;
    getProviderAtIndex(index: number): ResolvedReflectiveProvider;
    private _getMaxNumberOfObjects;
    private _instantiateProvider;
    private _instantiate;
    private _getByReflectiveDependency;
    private _getByKey;
    private _getObjByKeyId;
    get displayName(): string;
    toString(): string;
}

/**
 * `Dependency` is used by the framework to extend DI.
 * This is internal to Angular and should not be used directly.
 */
export declare class ɵangular_packages_core_core_e {
    key: ReflectiveKey;
    optional: boolean;
    visibility: Self | SkipSelf | null;
    constructor(key: ReflectiveKey, optional: boolean, visibility: Self | SkipSelf | null);
    static fromKey(key: ReflectiveKey): ɵangular_packages_core_core_e;
}

/**
 * Resolve a list of Providers.
 */
export declare function ɵangular_packages_core_core_f(providers: Provider[]): ResolvedReflectiveProvider[];

export declare function ɵangular_packages_core_core_g(): string;

/**
 * Creates an ElementRef given a node.
 *
 * @param ElementRefToken The ElementRef type
 * @param tNode The node for which you'd like an ElementRef
 * @param view The view to which the node belongs
 * @returns The ElementRef instance to use
 */
export declare function ɵangular_packages_core_core_h(ElementRefToken: typeof ElementRef, tNode: TNode, view: ɵangular_packages_core_core_bp): ElementRef;

/**
 * Creates a TemplateRef and stores it on the injector.
 *
 * @param TemplateRefToken The TemplateRef type
 * @param ElementRefToken The ElementRef type
 * @param hostTNode The node on which a TemplateRef is requested
 * @param hostView The view to which the node belongs
 * @returns The TemplateRef instance or null if we can't create a TemplateRef on a given node type
 */
export declare function ɵangular_packages_core_core_i<T>(TemplateRefToken: typeof TemplateRef, ElementRefToken: typeof ElementRef, hostTNode: TNode, hostView: ɵangular_packages_core_core_bp): TemplateRef<T> | null;

export declare function ɵangular_packages_core_core_j(id: string): NgModuleFactory<any>;

export declare class ɵangular_packages_core_core_k {
    readonly listeners: DebugEventListener[];
    readonly parent: DebugElement | null;
    readonly nativeNode: any;
    private readonly _debugContext;
    constructor(nativeNode: any, parent: DebugNode | null, _debugContext: ɵangular_packages_core_core_w);
    get injector(): Injector;
    get componentInstance(): any;
    get context(): any;
    get references(): {
        [key: string]: any;
    };
    get providerTokens(): any[];
}

export declare class ɵangular_packages_core_core_l extends ɵangular_packages_core_core_k implements DebugElement {
    readonly name: string;
    readonly properties: {
        [key: string]: any;
    };
    readonly attributes: {
        [key: string]: string | null;
    };
    readonly classes: {
        [key: string]: boolean;
    };
    readonly styles: {
        [key: string]: string | null;
    };
    readonly childNodes: DebugNode[];
    readonly nativeElement: any;
    constructor(nativeNode: any, parent: any, _debugContext: ɵangular_packages_core_core_w);
    addChild(child: DebugNode): void;
    removeChild(child: DebugNode): void;
    insertChildrenAfter(child: DebugNode, newChildren: DebugNode[]): void;
    insertBefore(refChild: DebugNode, newChild: DebugNode): void;
    query(predicate: Predicate<DebugElement>): DebugElement;
    queryAll(predicate: Predicate<DebugElement>): DebugElement[];
    queryAllNodes(predicate: Predicate<DebugNode>): DebugNode[];
    get children(): DebugElement[];
    triggerEventHandler(eventName: string, eventObj: any): void;
}

export declare function ɵangular_packages_core_core_m(nativeNode: any): DebugNode | null;

export declare class ɵangular_packages_core_core_n implements IterableDifferFactory {
    constructor();
    supports(obj: Object | null | undefined): boolean;
    create<V>(trackByFn?: TrackByFunction<V>): DefaultIterableDiffer<V>;
}

export declare class ɵangular_packages_core_core_o<K, V> implements KeyValueDifferFactory {
    constructor();
    supports(obj: any): boolean;
    create<K, V>(): KeyValueDiffer<K, V>;
}

export declare function ɵangular_packages_core_core_p(): IterableDiffers;

export declare function ɵangular_packages_core_core_q(): KeyValueDiffers;

export declare function ɵangular_packages_core_core_r(locale?: string): string;

/**
 * A built-in [dependency injection token](guide/glossary#di-token)
 * that is used to configure the root injector for bootstrapping.
 */
export declare const ɵangular_packages_core_core_s: StaticProvider[];

/**
 * Schedule work at next available slot.
 *
 * In Ivy this is just `requestAnimationFrame`. For compatibility reasons when bootstrapped
 * using `platformRef.bootstrap` we need to use `NgZone.onStable` as the scheduling mechanism.
 * This overrides the scheduling mechanism in Ivy to `NgZone.onStable`.
 *
 * @param ngZone NgZone to use for scheduling.
 */
export declare function ɵangular_packages_core_core_t(ngZone: NgZone): (fn: () => void) => void;

/**
 * USD currency code that the application uses by default for CurrencyPipe when no
 * DEFAULT_CURRENCY_CODE is provided.
 */
export declare const ɵangular_packages_core_core_u = "USD";

export declare function ɵangular_packages_core_core_v(checkIndex: number, flags: ɵNodeFlags, matchedQueriesDsl: [string | number, ɵQueryValueType][] | null, childCount: number, token: any, value: any, deps: ([ɵDepFlags, any] | any)[], bindings?: BindingDef[], outputs?: OutputDef[]): NodeDef;

export declare abstract class ɵangular_packages_core_core_w {
    abstract get view(): ViewData;
    abstract get nodeIndex(): number | null;
    abstract get injector(): Injector;
    abstract get component(): any;
    abstract get providerTokens(): any[];
    abstract get references(): {
        [key: string]: any;
    };
    abstract get context(): any;
    abstract get componentRenderElement(): any;
    abstract get renderNode(): any;
    abstract logError(console: Console, ...values: any[]): void;
}

/**
 * A change detection scheduler token for {@link RootContext}. This token is the default value used
 * for the default `RootContext` found in the {@link ROOT_CONTEXT} token.
 */
export declare const ɵangular_packages_core_core_x: InjectionToken<(fn: () => void) => void>;

/**
 * Inject static attribute value into directive constructor.
 *
 * This method is used with `factory` functions which are generated as part of
 * `defineDirective` or `defineComponent`. The method retrieves the static value
 * of an attribute. (Dynamic attributes are not supported since they are not resolved
 *  at the time of injection and can change over time.)
 *
 * # Example
 * Given:
 * ```
 * @Component(...)
 * class MyComponent {
 *   constructor(@Attribute('title') title: string) { ... }
 * }
 * ```
 * When instantiated with
 * ```
 * <my-component title="Hello"></my-component>
 * ```
 *
 * Then factory method generated is:
 * ```
 * MyComponent.ɵcmp = defineComponent({
 *   factory: () => new MyComponent(injectAttribute('title'))
 *   ...
 * })
 * ```
 *
 * @publicApi
 */
export declare function ɵangular_packages_core_core_y(tNode: TNode, attrNameToInject: string): string | null;

export declare const ɵangular_packages_core_core_z: InstructionState;

/**
 * Providers that will generate a random APP_ID_TOKEN.
 * @publicApi
 */
export declare const ɵAPP_ID_RANDOM_PROVIDER: {
    provide: InjectionToken<string>;
    useFactory: typeof ɵangular_packages_core_core_g;
    deps: any[];
};

export declare const enum ɵArgumentType {
    Inline = 0,
    Dynamic = 1
}

/**
 * A set of marker values to be used in the attributes arrays. These markers indicate that some
 * items are not regular attributes and the processing should be adapted accordingly.
 */
export declare const enum ɵAttributeMarker {
    /**
     * An implicit marker which indicates that the value in the array are of `attributeKey`,
     * `attributeValue` format.
     *
     * NOTE: This is implicit as it is the type when no marker is present in array. We indicate that
     * it should not be present at runtime by the negative number.
     */
    ImplicitAttributes = -1,
    /**
     * Marker indicates that the following 3 values in the attributes array are:
     * namespaceUri, attributeName, attributeValue
     * in that order.
     */
    NamespaceURI = 0,
    /**
      * Signals class declaration.
      *
      * Each value following `Classes` designates a class name to include on the element.
      * ## Example:
      *
      * Given:
      * ```
      * <div class="foo bar baz">...<d/vi>
      * ```
      *
      * the generated code is:
      * ```
      * var _c1 = [AttributeMarker.Classes, 'foo', 'bar', 'baz'];
      * ```
      */
    Classes = 1,
    /**
     * Signals style declaration.
     *
     * Each pair of values following `Styles` designates a style name and value to include on the
     * element.
     * ## Example:
     *
     * Given:
     * ```
     * <div style="width:100px; height:200px; color:red">...</div>
     * ```
     *
     * the generated code is:
     * ```
     * var _c1 = [AttributeMarker.Styles, 'width', '100px', 'height'. '200px', 'color', 'red'];
     * ```
     */
    Styles = 2,
    /**
     * Signals that the following attribute names were extracted from input or output bindings.
     *
     * For example, given the following HTML:
     *
     * ```
     * <div moo="car" [foo]="exp" (bar)="doSth()">
     * ```
     *
     * the generated code is:
     *
     * ```
     * var _c1 = ['moo', 'car', AttributeMarker.Bindings, 'foo', 'bar'];
     * ```
     */
    Bindings = 3,
    /**
     * Signals that the following attribute names were hoisted from an inline-template declaration.
     *
     * For example, given the following HTML:
     *
     * ```
     * <div *ngFor="let value of values; trackBy:trackBy" dirA [dirB]="value">
     * ```
     *
     * the generated code for the `template()` instruction would include:
     *
     * ```
     * ['dirA', '', AttributeMarker.Bindings, 'dirB', AttributeMarker.Template, 'ngFor', 'ngForOf',
     * 'ngForTrackBy', 'let-value']
     * ```
     *
     * while the generated code for the `element()` instruction inside the template function would
     * include:
     *
     * ```
     * ['dirA', '', AttributeMarker.Bindings, 'dirB']
     * ```
     */
    Template = 4,
    /**
     * Signals that the following attribute is `ngProjectAs` and its value is a parsed `CssSelector`.
     *
     * For example, given the following HTML:
     *
     * ```
     * <h1 attr="value" ngProjectAs="[title]">
     * ```
     *
     * the generated code for the `element()` instruction would include:
     *
     * ```
     * ['attr', 'value', AttributeMarker.ProjectAs, ['', 'title', '']]
     * ```
     */
    ProjectAs = 5,
    /**
     * Signals that the following attribute will be translated by runtime i18n
     *
     * For example, given the following HTML:
     *
     * ```
     * <div moo="car" foo="value" i18n-foo [bar]="binding" i18n-bar>
     * ```
     *
     * the generated code is:
     *
     * ```
     * var _c1 = ['moo', 'car', AttributeMarker.I18n, 'foo', 'bar'];
     */
    I18n = 6
}

export declare const enum ɵBindingFlags {
    TypeElementAttribute = 1,
    TypeElementClass = 2,
    TypeElementStyle = 4,
    TypeProperty = 8,
    SyntheticProperty = 16,
    SyntheticHostProperty = 32,
    CatSyntheticProperty = 48,
    Types = 15
}

/**
 * Mark `html` string as trusted.
 *
 * This function wraps the trusted string in `String` and brands it in a way which makes it
 * recognizable to {@link htmlSanitizer} to be trusted implicitly.
 *
 * @param trustedHtml `html` string which needs to be implicitly trusted.
 * @returns a `html` which has been branded to be implicitly trusted.
 */
export declare function ɵbypassSanitizationTrustHtml(trustedHtml: string): ɵSafeHtml;

/**
 * Mark `url` string as trusted.
 *
 * This function wraps the trusted string in `String` and brands it in a way which makes it
 * recognizable to {@link resourceUrlSanitizer} to be trusted implicitly.
 *
 * @param trustedResourceUrl `url` string which needs to be implicitly trusted.
 * @returns a `url` which has been branded to be implicitly trusted.
 */
export declare function ɵbypassSanitizationTrustResourceUrl(trustedResourceUrl: string): ɵSafeResourceUrl;

/**
 * Mark `script` string as trusted.
 *
 * This function wraps the trusted string in `String` and brands it in a way which makes it
 * recognizable to {@link scriptSanitizer} to be trusted implicitly.
 *
 * @param trustedScript `script` string which needs to be implicitly trusted.
 * @returns a `script` which has been branded to be implicitly trusted.
 */
export declare function ɵbypassSanitizationTrustScript(trustedScript: string): ɵSafeScript;

/**
 * Mark `style` string as trusted.
 *
 * This function wraps the trusted string in `String` and brands it in a way which makes it
 * recognizable to {@link styleSanitizer} to be trusted implicitly.
 *
 * @param trustedStyle `style` string which needs to be implicitly trusted.
 * @returns a `style` hich has been branded to be implicitly trusted.
 */
export declare function ɵbypassSanitizationTrustStyle(trustedStyle: string): ɵSafeStyle;

/**
 * Mark `url` string as trusted.
 *
 * This function wraps the trusted string in `String` and brands it in a way which makes it
 * recognizable to {@link urlSanitizer} to be trusted implicitly.
 *
 * @param trustedUrl `url` string which needs to be implicitly trusted.
 * @returns a `url`  which has been branded to be implicitly trusted.
 */
export declare function ɵbypassSanitizationTrustUrl(trustedUrl: string): ɵSafeUrl;


export declare const enum ɵBypassType {
    Url = "URL",
    Html = "HTML",
    ResourceUrl = "ResourceURL",
    Script = "Script",
    Style = "Style"
}

export declare function ɵccf(selector: string, componentType: Type<any>, viewDefFactory: ViewDefinitionFactory, inputs: {
    [propName: string]: string;
} | null, outputs: {
    [propName: string]: string;
}, ngContentSelectors: string[]): ComponentFactory<any>;

/**
 * Defines the possible states of the default change detector.
 * @see `ChangeDetectorRef`
 */
export declare enum ɵChangeDetectorStatus {
    /**
     * A state in which, after calling `detectChanges()`, the change detector
     * state becomes `Checked`, and must be explicitly invoked or reactivated.
     */
    CheckOnce = 0,
    /**
     * A state in which change detection is skipped until the change detector mode
     * becomes `CheckOnce`.
     */
    Checked = 1,
    /**
     * A state in which change detection continues automatically until explicitly
     * deactivated.
     */
    CheckAlways = 2,
    /**
     * A state in which a change detector sub tree is not a part of the main tree and
     * should be skipped.
     */
    Detached = 3,
    /**
     * Indicates that the change detector encountered an error checking a binding
     * or calling a directive lifecycle method and is now in an inconsistent state. Change
     * detectors in this state do not detect changes.
     */
    Errored = 4,
    /**
     * Indicates that the change detector has been destroyed.
     */
    Destroyed = 5
}

export declare function ɵclearOverrides(): void;

export declare function ɵclearResolutionOfComponentResourcesQueue(): Map<Type<any>, Component>;

export declare function ɵcmf(ngModuleType: Type<any>, bootstrapComponents: Type<any>[], defFactory: NgModuleDefinitionFactory): NgModuleFactory<any>;

export declare class ɵCodegenComponentFactoryResolver implements ComponentFactoryResolver {
    private _parent;
    private _ngModule;
    private _factories;
    constructor(factories: ComponentFactory<any>[], _parent: ComponentFactoryResolver, _ngModule: NgModuleRef<any>);
    resolveComponentFactory<T>(component: {
        new (...args: any[]): T;
    }): ComponentFactory<T>;
}

/**
 * Compile an Angular component according to its decorator metadata, and patch the resulting
 * component def (ɵcmp) onto the component type.
 *
 * Compilation may be asynchronous (due to the need to resolve URLs for the component template or
 * other resources, for example). In the event that compilation is not immediate, `compileComponent`
 * will enqueue resource resolution into a global queue and will fail to return the `ɵcmp`
 * until the global queue has been resolved with a call to `resolveComponentResources`.
 */
export declare function ɵcompileComponent(type: Type<any>, metadata: Component): void;

/**
 * Compile an Angular directive according to its decorator metadata, and patch the resulting
 * directive def onto the component type.
 *
 * In the event that compilation is not immediate, `compileDirective` will return a `Promise` which
 * will resolve when compilation completes and the directive becomes usable.
 */
export declare function ɵcompileDirective(type: Type<any>, directive: Directive | null): void;

/**
 * Compiles a module in JIT mode.
 *
 * This function automatically gets called when a class has a `@NgModule` decorator.
 */
export declare function ɵcompileNgModule(moduleType: Type<any>, ngModule?: NgModule): void;

/**
 * Compiles and adds the `ɵmod` and `ɵinj` properties to the module class.
 *
 * It's possible to compile a module via this API which will allow duplicate declarations in its
 * root.
 */
export declare function ɵcompileNgModuleDefs(moduleType: ɵNgModuleType, ngModule: NgModule, allowDuplicateDeclarationsInRoot?: boolean): void;

export declare function ɵcompileNgModuleFactory__POST_R3__<M>(injector: Injector, options: CompilerOptions, moduleType: Type<M>): Promise<NgModuleFactory<M>>;

export declare function ɵcompilePipe(type: Type<any>, meta: Pipe): void;

export declare const ɵCompiler_compileModuleAndAllComponentsAsync__POST_R3__: <T>(moduleType: Type<T>) => Promise<ModuleWithComponentFactories<T>>;

export declare const ɵCompiler_compileModuleAndAllComponentsSync__POST_R3__: <T>(moduleType: Type<T>) => ModuleWithComponentFactories<T>;

export declare const ɵCompiler_compileModuleAsync__POST_R3__: <T>(moduleType: Type<T>) => Promise<NgModuleFactory<T>>;

export declare const ɵCompiler_compileModuleSync__POST_R3__: <T>(moduleType: Type<T>) => NgModuleFactory<T>;

/**
 * Runtime link information for Components.
 *
 * This is an internal data structure used by the render to link
 * components into templates.
 *
 * NOTE: Always use `defineComponent` function to create this object,
 * never create the object directly since the shape of this object
 * can change between versions.
 *
 * See: {@link defineComponent}
 */
export declare interface ɵComponentDef<T> extends ɵDirectiveDef<T> {
    /**
     * Runtime unique component ID.
     */
    readonly id: string;
    /**
     * The View template of the component.
     */
    readonly template: ComponentTemplate<T>;
    /** Constants associated with the component's view. */
    readonly consts: TConstants | null;
    /**
     * An array of `ngContent[selector]` values that were found in the template.
     */
    readonly ngContentSelectors?: string[];
    /**
     * A set of styles that the component needs to be present for component to render correctly.
     */
    readonly styles: string[];
    /**
     * The number of nodes, local refs, and pipes in this component template.
     *
     * Used to calculate the length of the component's LView array, so we
     * can pre-fill the array and set the binding start index.
     */
    readonly decls: number;
    /**
     * The number of bindings in this component template (including pure fn bindings).
     *
     * Used to calculate the length of the component's LView array, so we
     * can pre-fill the array and set the host binding start index.
     */
    readonly vars: number;
    /**
     * Query-related instructions for a component.
     */
    viewQuery: ViewQueriesFunction<T> | null;
    /**
     * The view encapsulation type, which determines how styles are applied to
     * DOM elements. One of
     * - `Emulated` (default): Emulate native scoping of styles.
     * - `Native`: Use the native encapsulation mechanism of the renderer.
     * - `ShadowDom`: Use modern [ShadowDOM](https://w3c.github.io/webcomponents/spec/shadow/) and
     *   create a ShadowRoot for component's host element.
     * - `None`: Do not provide any template or style encapsulation.
     */
    readonly encapsulation: ViewEncapsulation;
    /**
     * Defines arbitrary developer-defined data to be stored on a renderer instance.
     * This is useful for renderers that delegate to other renderers.
     */
    readonly data: {
        [kind: string]: any;
    };
    /** Whether or not this component's ChangeDetectionStrategy is OnPush */
    readonly onPush: boolean;
    /**
     * Registry of directives and components that may be found in this view.
     *
     * The property is either an array of `DirectiveDef`s or a function which returns the array of
     * `DirectiveDef`s. The function is necessary to be able to support forward declarations.
     */
    directiveDefs: DirectiveDefListOrFactory | null;
    /**
     * Registry of pipes that may be found in this view.
     *
     * The property is either an array of `PipeDefs`s or a function which returns the array of
     * `PipeDefs`s. The function is necessary to be able to support forward declarations.
     */
    pipeDefs: PipeDefListOrFactory | null;
    /**
     * The set of schemas that declare elements to be allowed in the component's template.
     */
    schemas: SchemaMetadata[] | null;
    /**
     * Ivy runtime uses this place to store the computed tView for the component. This gets filled on
     * the first run of component.
     */
    tView: TView | null;
    /**
     * Used to store the result of `noSideEffects` function so that it is not removed by closure
     * compiler. The property should never be read.
     */
    readonly _?: never;
}

/**
 * A subclass of `Type` which has a static `ɵcmp`:`ComponentDef` field making it
 * consumable for rendering.
 */
export declare interface ɵComponentType<T> extends Type<T> {
    ɵcmp: never;
}


export declare class ɵConsole {
    log(message: string): void;
    warn(message: string): void;
    static ɵfac: ɵngcc0.ɵɵFactoryDef<ɵConsole>;
    static ɵprov: ɵngcc0.ɵɵInjectableDef<ɵConsole>;
}

/**
 * Create a new `Injector` which is configured using a `defType` of `InjectorType<any>`s.
 *
 * @publicApi
 */
export declare function ɵcreateInjector(defType: any, parent?: Injector | null, additionalProviders?: StaticProvider[] | null, name?: string): Injector;

export declare function ɵcrt(values: {
    styles: (string | any[])[];
    encapsulation: ViewEncapsulation;
    data: {
        [kind: string]: any[];
    };
}): RendererType2;

/**
 * A list of CssSelectors.
 *
 * A directive or component can have multiple selectors. This type is used for
 * directive defs so any of the selectors in the list will match that directive.
 *
 * Original: 'form, [ngForm]'
 * Parsed: [['form'], ['', 'ngForm', '']]
 */
export declare type ɵCssSelectorList = CssSelector[];

/**
 * Index of each value in currency data (used to describe CURRENCIES_EN in currencies.ts)
 */
export declare const enum ɵCurrencyIndex {
    Symbol = 0,
    SymbolNarrow = 1,
    NbOfDigits = 2
}

/**
 * The locale id that the application is using by default (for translations and ICU expressions).
 */
export declare const ɵDEFAULT_LOCALE_ID = "en-US";

export declare const ɵdefaultIterableDiffers: IterableDiffers;

export declare const ɵdefaultKeyValueDiffers: KeyValueDiffers;

/**
 * Bitmask for DI flags
 */
export declare const enum ɵDepFlags {
    None = 0,
    SkipSelf = 1,
    Optional = 2,
    Self = 4,
    Value = 8
}


/**
 * Synchronously perform change detection on a component (and possibly its sub-components).
 *
 * This function triggers change detection in a synchronous way on a component.
 *
 * @param component The component which the change detection should be performed on.
 */
export declare function ɵdetectChanges(component: {}): void;


export declare function ɵdevModeEqual(a: any, b: any): boolean;

export declare function ɵdid(checkIndex: number, flags: ɵNodeFlags, matchedQueries: null | [string | number, ɵQueryValueType][], childCount: number, ctor: any, deps: ([ɵDepFlags, any] | any)[], props?: null | {
    [name: string]: [number, string];
}, outputs?: null | {
    [name: string]: string;
}): NodeDef;

/**
 * Runtime link information for Directives.
 *
 * This is an internal data structure used by the render to link
 * directives into templates.
 *
 * NOTE: Always use `defineDirective` function to create this object,
 * never create the object directly since the shape of this object
 * can change between versions.
 *
 * @param Selector type metadata specifying the selector of the directive or component
 *
 * See: {@link defineDirective}
 */
export declare interface ɵDirectiveDef<T> {
    /**
     * A dictionary mapping the inputs' minified property names to their public API names, which
     * are their aliases if any, or their original unminified property names
     * (as in `@Input('alias') propertyName: any;`).
     */
    readonly inputs: {
        [P in keyof T]: string;
    };
    /**
     * @deprecated This is only here because `NgOnChanges` incorrectly uses declared name instead of
     * public or minified name.
     */
    readonly declaredInputs: {
        [P in keyof T]: string;
    };
    /**
     * A dictionary mapping the outputs' minified property names to their public API names, which
     * are their aliases if any, or their original unminified property names
     * (as in `@Output('alias') propertyName: any;`).
     */
    readonly outputs: {
        [P in keyof T]: string;
    };
    /**
     * Function to create and refresh content queries associated with a given directive.
     */
    contentQueries: ContentQueriesFunction<T> | null;
    /**
     * Query-related instructions for a directive. Note that while directives don't have a
     * view and as such view queries won't necessarily do anything, there might be
     * components that extend the directive.
     */
    viewQuery: ViewQueriesFunction<T> | null;
    /**
     * Refreshes host bindings on the associated directive.
     */
    readonly hostBindings: HostBindingsFunction<T> | null;
    /**
     * The number of bindings in this directive `hostBindings` (including pure fn bindings).
     *
     * Used to calculate the length of the component's LView array, so we
     * can pre-fill the array and set the host binding start index.
     */
    readonly hostVars: number;
    /**
     * Assign static attribute values to a host element.
     *
     * This property will assign static attribute values as well as class and style
     * values to a host element. Since attribute values can consist of different types of values, the
     * `hostAttrs` array must include the values in the following format:
     *
     * attrs = [
     *   // static attributes (like `title`, `name`, `id`...)
     *   attr1, value1, attr2, value,
     *
     *   // a single namespace value (like `x:id`)
     *   NAMESPACE_MARKER, namespaceUri1, name1, value1,
     *
     *   // another single namespace value (like `x:name`)
     *   NAMESPACE_MARKER, namespaceUri2, name2, value2,
     *
     *   // a series of CSS classes that will be applied to the element (no spaces)
     *   CLASSES_MARKER, class1, class2, class3,
     *
     *   // a series of CSS styles (property + value) that will be applied to the element
     *   STYLES_MARKER, prop1, value1, prop2, value2
     * ]
     *
     * All non-class and non-style attributes must be defined at the start of the list
     * first before all class and style values are set. When there is a change in value
     * type (like when classes and styles are introduced) a marker must be used to separate
     * the entries. The marker values themselves are set via entries found in the
     * [AttributeMarker] enum.
     */
    readonly hostAttrs: TAttributes | null;
    /** Token representing the directive. Used by DI. */
    readonly type: Type<T>;
    /** Function that resolves providers and publishes them into the DI system. */
    providersResolver: (<U extends T>(def: ɵDirectiveDef<U>, processProvidersFn?: ProcessProvidersFunction) => void) | null;
    /** The selectors that will be used to match nodes to this directive. */
    readonly selectors: ɵCssSelectorList;
    /**
     * Name under which the directive is exported (for use with local references in template)
     */
    readonly exportAs: string[] | null;
    /**
     * Factory function used to create a new directive instance. Will be null initially.
     * Populated when the factory is first requested by directive instantiation logic.
     */
    readonly factory: FactoryFn<T> | null;
    readonly onChanges: (() => void) | null;
    readonly onInit: (() => void) | null;
    readonly doCheck: (() => void) | null;
    readonly afterContentInit: (() => void) | null;
    readonly afterContentChecked: (() => void) | null;
    readonly afterViewInit: (() => void) | null;
    readonly afterViewChecked: (() => void) | null;
    readonly onDestroy: (() => void) | null;
    /**
     * The features applied to this directive
     */
    readonly features: DirectiveDefFeature[] | null;
    setInput: (<U extends T>(this: ɵDirectiveDef<U>, instance: U, value: any, publicName: string, privateName: string) => void) | null;
}

/**
 * A subclass of `Type` which has a static `ɵdir`:`DirectiveDef` field making it
 * consumable for rendering.
 */
export declare interface ɵDirectiveType<T> extends Type<T> {
    ɵdir: never;
    ɵfac: () => T;
}

export declare function ɵeld(checkIndex: number, flags: ɵNodeFlags, matchedQueriesDsl: null | [string | number, ɵQueryValueType][], ngContentIndex: null | number, childCount: number, namespaceAndName: string | null, fixedAttrs?: null | [string, string][], bindings?: null | [ɵBindingFlags, string, string | SecurityContext | null][], outputs?: null | ([string, string])[], handleEvent?: null | ElementHandleEventFn, componentView?: null | ViewDefinitionFactory, componentRendererType?: RendererType2 | null): NodeDef;

export declare const ɵEMPTY_ARRAY: any[];

export declare const ɵEMPTY_MAP: {
    [key: string]: any;
};

/**
 * Index of each type of locale data from the extra locale data array
 */
export declare const enum ɵExtraLocaleDataIndex {
    ExtraDayPeriodFormats = 0,
    ExtraDayPeriodStandalone = 1,
    ExtraDayPeriodsRules = 2
}

/**
 * Finds the locale data for a given locale.
 *
 * @param locale The locale code.
 * @returns The locale data.
 * @see [Internationalization (i18n) Guide](https://angular.io/guide/i18n)
 */
export declare function ɵfindLocaleData(locale: string): any;

/**
 * Loops over queued module definitions, if a given module definition has all of its
 * declarations resolved, it dequeues that module definition and sets the scope on
 * its declarations.
 */
export declare function ɵflushModuleScopingQueueAsMuchAsPossible(): void;

export declare function ɵgetComponentViewDefinitionFactory(componentFactory: ComponentFactory<any>): ViewDefinitionFactory;

export declare function ɵgetDebugNode__POST_R3__(nativeNode: Element): DebugElement__POST_R3__;

export declare function ɵgetDebugNode__POST_R3__(nativeNode: Node): DebugNode__POST_R3__;

export declare function ɵgetDebugNode__POST_R3__(nativeNode: null): null;

export declare const ɵgetDebugNodeR2: (nativeNode: any) => DebugNode | null;

/**
 * Retrieves directive instances associated with a given DOM element. Does not include
 * component instances.
 *
 * @usageNotes
 * Given the following DOM structure:
 * ```
 * <my-app>
 *   <button my-button></button>
 *   <my-comp></my-comp>
 * </my-app>
 * ```
 * Calling `getDirectives` on `<button>` will return an array with an instance of the `MyButton`
 * directive that is associated with the DOM element.
 *
 * Calling `getDirectives` on `<my-comp>` will return an empty array.
 *
 * @param element DOM element for which to get the directives.
 * @returns Array of directives associated with the element.
 *
 * @publicApi
 * @globalApi ng
 */
export declare function ɵgetDirectives(element: Element): {}[];

/**
 * Retrieves the host element of a component or directive instance.
 * The host element is the DOM element that matched the selector of the directive.
 *
 * @param componentOrDirective Component or directive instance for which the host
 *     element should be retrieved.
 * @returns Host element of the target.
 *
 * @publicApi
 * @globalApi ng
 */
export declare function ɵgetHostElement(componentOrDirective: {}): Element;

/**
 * Read the injectable def (`ɵprov`) for `type` in a way which is immune to accidentally reading
 * inherited value.
 *
 * @param type A type which may have its own (non-inherited) `ɵprov`.
 */
export declare function ɵgetInjectableDef<T>(type: any): ɵɵInjectableDef<T> | null;

/** Returns the matching `LContext` data for a given DOM node, directive or component instance.
 *
 * This function will examine the provided DOM element, component, or directive instance\'s
 * monkey-patched property to derive the `LContext` data. Once called then the monkey-patched
 * value will be that of the newly created `LContext`.
 *
 * If the monkey-patched value is the `LView` instance then the context value for that
 * target will be created and the monkey-patch reference will be updated. Therefore when this
 * function is called it may mutate the provided element\'s, component\'s or any of the associated
 * directive\'s monkey-patch values.
 *
 * If the monkey-patch value is not detected then the code will walk up the DOM until an element
 * is found which contains a monkey-patch reference. When that occurs then the provided element
 * will be updated with a new context (which is then returned). If the monkey-patch value is not
 * detected for a component/directive instance then it will throw an error (all components and
 * directives should be automatically monkey-patched by ivy).
 *
 * @param target Component, Directive or DOM Node.
 */
export declare function ɵgetLContext(target: any): ɵLContext | null;

/**
 * Retrieves the default currency code for the given locale.
 *
 * The default is defined as the first currency which is still in use.
 *
 * @param locale The code of the locale whose currency code we want.
 * @returns The code of the default currency for the given locale.
 *
 */
export declare function ɵgetLocaleCurrencyCode(locale: string): string | null;

/**
 * Retrieves the plural function used by ICU expressions to determine the plural case to use
 * for a given locale.
 * @param locale A locale code for the locale format rules to use.
 * @returns The plural function for the locale.
 * @see `NgPlural`
 * @see [Internationalization (i18n) Guide](https://angular.io/guide/i18n)
 */
export declare function ɵgetLocalePluralCase(locale: string): (value: number) => number;

export declare function ɵgetModuleFactory__POST_R3__(id: string): NgModuleFactory<any>;

export declare function ɵgetSanitizationBypassType(value: any): ɵBypassType | null;

export declare type ɵGetterFn = (obj: any) => any;


export declare const ɵglobal: any;

export declare function ɵinitServicesIfNeeded(): void;

export declare function ɵINJECTOR_IMPL__POST_R3__(providers: StaticProvider[], parent: Injector | undefined, name: string): Injector;

/**
 * An internal token whose presence in an injector indicates that the injector should treat itself
 * as a root scoped injector when processing requests for unknown tokens which may indicate
 * they are provided in the root scope.
 */
export declare const ɵINJECTOR_SCOPE: InjectionToken<"root" | "platform" | null>;

export declare function ɵinlineInterpolate(valueCount: number, c0: string, a1: any, c1: string, a2?: any, c2?: string, a3?: any, c3?: string, a4?: any, c4?: string, a5?: any, c5?: string, a6?: any, c6?: string, a7?: any, c7?: string, a8?: any, c8?: string, a9?: any, c9?: string): string;

export declare function ɵinterpolate(valueCount: number, constAndInterp: string[]): string;

export declare function ɵisBoundToModule__POST_R3__<C>(cf: ComponentFactory<C>): boolean;

/**
 * Reports whether a given strategy is currently the default for change detection.
 * @param changeDetectionStrategy The strategy to check.
 * @returns True if the given strategy is the current default, false otherwise.
 * @see `ChangeDetectorStatus`
 * @see `ChangeDetectorRef`
 */
export declare function ɵisDefaultChangeDetectionStrategy(changeDetectionStrategy: ChangeDetectionStrategy): boolean;

export declare function ɵisListLikeIterable(obj: any): boolean;

/**
 * Determine if the argument is an Observable
 */
export declare function ɵisObservable(obj: any | Observable<any>): obj is Observable<any>;

/**
 * Determine if the argument is shaped like a Promise
 */
export declare function ɵisPromise(obj: any): obj is Promise<any>;

export declare const ɵivyEnabled = false;

/**
 * The internal view context which is specific to a given DOM element, directive or
 * component instance. Each value in here (besides the LView and element node details)
 * can be present, null or undefined. If undefined then it implies the value has not been
 * looked up yet, otherwise, if null, then a lookup was executed and nothing was found.
 *
 * Each value will get filled when the respective value is examined within the getContext
 * function. The component, element and each directive instance will share the same instance
 * of the context.
 */
export declare interface ɵLContext {
    /**
     * The component's parent view data.
     */
    lView: ɵangular_packages_core_core_bp;
    /**
     * The index instance of the node.
     */
    nodeIndex: number;
    /**
     * The instance of the DOM node that is attached to the lNode.
     */
    native: RNode;
    /**
     * The instance of the Component node.
     */
    component: {} | null | undefined;
    /**
     * The list of active directives that exist on this element.
     */
    directives: any[] | null | undefined;
    /**
     * The map of local references (local reference name => element or directive instance) that exist
     * on this element.
     */
    localRefs: {
        [key: string]: any;
    } | null | undefined;
}

/**
 * Used to enable lifecycle hooks on the root component.
 *
 * Include this feature when calling `renderComponent` if the root component
 * you are rendering has lifecycle hooks defined. Otherwise, the hooks won't
 * be called properly.
 *
 * Example:
 *
 * ```
 * renderComponent(AppComponent, {hostFeatures: [LifecycleHooksFeature]});
 * ```
 */
export declare function ɵLifecycleHooksFeature(component: any, def: ɵComponentDef<any>): void;

/**
 * Index of each type of locale data from the locale data array
 */
export declare enum ɵLocaleDataIndex {
    LocaleId = 0,
    DayPeriodsFormat = 1,
    DayPeriodsStandalone = 2,
    DaysFormat = 3,
    DaysStandalone = 4,
    MonthsFormat = 5,
    MonthsStandalone = 6,
    Eras = 7,
    FirstDayOfWeek = 8,
    WeekendRange = 9,
    DateFormat = 10,
    TimeFormat = 11,
    DateTimeFormat = 12,
    NumberSymbols = 13,
    NumberFormats = 14,
    CurrencyCode = 15,
    CurrencySymbol = 16,
    CurrencyName = 17,
    Currencies = 18,
    PluralCase = 19,
    ExtraData = 20
}


export declare function ɵlooseIdentical(a: any, b: any): boolean;

/**
 * @suppress {globalThis}
 */
export declare function ɵmakeDecorator<T>(name: string, props?: (...args: any[]) => any, parentClass?: any, additionalProcessing?: (type: Type<T>) => void, typeFn?: (type: Type<T>, ...args: any[]) => void): {
    new (...args: any[]): any;
    (...args: any[]): any;
    (...args: any[]): (cls: any) => any;
};

/**
 * Marks the component as dirty (needing change detection). Marking a component dirty will
 * schedule a change detection on it at some point in the future.
 *
 * Marking an already dirty component as dirty won't do anything. Only one outstanding change
 * detection can be scheduled per component tree.
 *
 * @param component Component to mark as dirty.
 */
export declare function ɵmarkDirty(component: {}): void;

export declare type ɵMethodFn = (obj: any, args: any[]) => any;

export declare function ɵmod(providers: NgModuleProviderDef[]): NgModuleDefinition;

export declare function ɵmpd(flags: ɵNodeFlags, token: any, value: any, deps: ([ɵDepFlags, any] | any)[]): NgModuleProviderDef;

export declare function ɵncd(ngContentIndex: null | number, index: number): NodeDef;


export declare const ɵNG_COMP_DEF: string;

export declare const ɵNG_DIR_DEF: string;

/**
 * If a directive is diPublic, bloomAdd sets a property on the type with this constant as
 * the key and the directive's unique ID as the value. This allows us to map directives to their
 * bloom filter bit for DI.
 */
export declare const ɵNG_ELEMENT_ID: string;

export declare const ɵNG_INJ_DEF: string;

export declare const ɵNG_MOD_DEF: string;

export declare const ɵNG_PIPE_DEF: string;

export declare const ɵNG_PROV_DEF: string;

/**
 * Runtime link information for NgModules.
 *
 * This is the internal data structure used by the runtime to assemble components, directives,
 * pipes, and injectors.
 *
 * NOTE: Always use `ɵɵdefineNgModule` function to create this object,
 * never create the object directly since the shape of this object
 * can change between versions.
 */
export declare interface ɵNgModuleDef<T> {
    /** Token representing the module. Used by DI. */
    type: T;
    /** List of components to bootstrap. */
    bootstrap: Type<any>[] | (() => Type<any>[]);
    /** List of components, directives, and pipes declared by this module. */
    declarations: Type<any>[] | (() => Type<any>[]);
    /** List of modules or `ModuleWithProviders` imported by this module. */
    imports: Type<any>[] | (() => Type<any>[]);
    /**
     * List of modules, `ModuleWithProviders`, components, directives, or pipes exported by this
     * module.
     */
    exports: Type<any>[] | (() => Type<any>[]);
    /**
     * Cached value of computed `transitiveCompileScopes` for this module.
     *
     * This should never be read directly, but accessed via `transitiveScopesFor`.
     */
    transitiveCompileScopes: ɵNgModuleTransitiveScopes | null;
    /** The set of schemas that declare elements to be allowed in the NgModule. */
    schemas: SchemaMetadata[] | null;
    /** Unique ID for the module with which it should be registered.  */
    id: string | null;
}

export declare class ɵNgModuleFactory<T> extends NgModuleFactory<T> {
    moduleType: Type<T>;
    constructor(moduleType: Type<T>);
    create(parentInjector: Injector | null): NgModuleRef<T>;
}

/**
 * Represents the expansion of an `NgModule` into its scopes.
 *
 * A scope is a set of directives and pipes that are visible in a particular context. Each
 * `NgModule` has two scopes. The `compilation` scope is the set of directives and pipes that will
 * be recognized in the templates of components declared by the module. The `exported` scope is the
 * set of directives and pipes exported by a module (that is, module B's exported scope gets added
 * to module A's compilation scope when module A imports B).
 */
export declare interface ɵNgModuleTransitiveScopes {
    compilation: {
        directives: Set<any>;
        pipes: Set<any>;
    };
    exported: {
        directives: Set<any>;
        pipes: Set<any>;
    };
    schemas: SchemaMetadata[] | null;
}

export declare interface ɵNgModuleType<T = any> extends Type<T> {
    ɵmod: ɵNgModuleDef<T>;
}


export declare interface ɵNO_CHANGE {
    __brand__: 'NO_CHANGE';
}

/** A special value which designates that a value has not changed. */
export declare const ɵNO_CHANGE: ɵNO_CHANGE;

/**
 * Bitmask for NodeDef.flags.
 * Naming convention:
 * - `Type...`: flags that are mutually exclusive
 * - `Cat...`: union of multiple `Type...` (short for category).
 */
export declare const enum ɵNodeFlags {
    None = 0,
    TypeElement = 1,
    TypeText = 2,
    ProjectedTemplate = 4,
    CatRenderNode = 3,
    TypeNgContent = 8,
    TypePipe = 16,
    TypePureArray = 32,
    TypePureObject = 64,
    TypePurePipe = 128,
    CatPureExpression = 224,
    TypeValueProvider = 256,
    TypeClassProvider = 512,
    TypeFactoryProvider = 1024,
    TypeUseExistingProvider = 2048,
    LazyProvider = 4096,
    PrivateProvider = 8192,
    TypeDirective = 16384,
    Component = 32768,
    CatProviderNoDirective = 3840,
    CatProvider = 20224,
    OnInit = 65536,
    OnDestroy = 131072,
    DoCheck = 262144,
    OnChanges = 524288,
    AfterContentInit = 1048576,
    AfterContentChecked = 2097152,
    AfterViewInit = 4194304,
    AfterViewChecked = 8388608,
    EmbeddedViews = 16777216,
    ComponentView = 33554432,
    TypeContentQuery = 67108864,
    TypeViewQuery = 134217728,
    StaticQuery = 268435456,
    DynamicQuery = 536870912,
    TypeNgModule = 1073741824,
    CatQuery = 201326592,
    Types = 201347067
}

/**
 * Provides a noop implementation of `NgZone` which does nothing. This zone requires explicit calls
 * to framework to perform rendering.
 */
export declare class ɵNoopNgZone implements NgZone {
    readonly hasPendingMicrotasks: boolean;
    readonly hasPendingMacrotasks: boolean;
    readonly isStable: boolean;
    readonly onUnstable: EventEmitter<any>;
    readonly onMicrotaskEmpty: EventEmitter<any>;
    readonly onStable: EventEmitter<any>;
    readonly onError: EventEmitter<any>;
    run(fn: (...args: any[]) => any, applyThis?: any, applyArgs?: any): any;
    runGuarded(fn: (...args: any[]) => any, applyThis?: any, applyArgs?: any): any;
    runOutsideAngular(fn: (...args: any[]) => any): any;
    runTask(fn: (...args: any[]) => any, applyThis?: any, applyArgs?: any, name?: string): any;
}

export declare const ɵNOT_FOUND_CHECK_ONLY_ELEMENT_INJECTOR: {};

export declare function ɵnov(view: ViewData, index: number): any;

export declare function ɵoverrideComponentView(comp: Type<any>, componentFactory: ComponentFactory<any>): void;

export declare function ɵoverrideProvider(override: ProviderOverride): void;

export declare function ɵpad(checkIndex: number, argCount: number): NodeDef;

/**
 * Patch the definition of a component with directives and pipes from the compilation scope of
 * a given module.
 */
export declare function ɵpatchComponentDefWithScope<C>(componentDef: ɵComponentDef<C>, transitiveScopes: ɵNgModuleTransitiveScopes): void;

export declare function ɵpid(flags: ɵNodeFlags, ctor: any, deps: ([ɵDepFlags, any] | any)[]): NodeDef;

/**
 * Runtime link information for Pipes.
 *
 * This is an internal data structure used by the renderer to link
 * pipes into templates.
 *
 * NOTE: Always use `definePipe` function to create this object,
 * never create the object directly since the shape of this object
 * can change between versions.
 *
 * See: {@link definePipe}
 */
export declare interface ɵPipeDef<T> {
    /** Token representing the pipe. */
    type: Type<T>;
    /**
     * Pipe name.
     *
     * Used to resolve pipe in templates.
     */
    readonly name: string;
    /**
     * Factory function used to create a new pipe instance. Will be null initially.
     * Populated when the factory is first requested by pipe instantiation logic.
     */
    factory: FactoryFn<T> | null;
    /**
     * Whether or not the pipe is pure.
     *
     * Pure pipes result only depends on the pipe input and not on internal
     * state of the pipe.
     */
    readonly pure: boolean;
    onDestroy: (() => void) | null;
}


/**
 * A shared interface which contains an animation player
 */
export declare interface ɵPlayer {
    parent?: ɵPlayer | null;
    state: ɵPlayState;
    play(): void;
    pause(): void;
    finish(): void;
    destroy(): void;
    addEventListener(state: ɵPlayState | string, cb: (data?: any) => any): void;
}

/**
 * Used as a reference to build a player from a styling template binding
 * (`[style]` and `[class]`).
 *
 * The `fn` function will be called once any styling-related changes are
 * evaluated on an element and is expected to return a player that will
 * be then run on the element.
 *
 * `[style]`, `[style.prop]`, `[class]` and `[class.name]` template bindings
 * all accept a `PlayerFactory` as input and this player factories.
 */
export declare interface ɵPlayerFactory {
    '__brand__': 'Brand for PlayerFactory that nothing will match';
}

/**
 * Designed to be used as an injection service to capture all animation players.
 *
 * When present all animation players will be passed into the flush method below.
 * This feature is designed to service application-wide animation testing, live
 * debugging as well as custom animation choreographing tools.
 */
export declare interface ɵPlayerHandler {
    /**
     * Designed to kick off the player at the end of change detection
     */
    flushPlayers(): void;
    /**
     * @param player The player that has been scheduled to run within the application.
     * @param context The context as to where the player was bound to
     */
    queuePlayer(player: ɵPlayer, context: ComponentInstance | DirectiveInstance | HTMLElement): void;
}

/**
 * The state of a given player
 *
 * Do not change the increasing nature of the numbers since the player
 * code may compare state by checking if a number is higher or lower than
 * a certain numeric value.
 */
export declare const enum ɵPlayState {
    Pending = 0,
    Running = 1,
    Paused = 2,
    Finished = 100,
    Destroyed = 200
}

export declare function ɵpod(checkIndex: number, propToIndex: {
    [p: string]: number;
}): NodeDef;

export declare function ɵppd(checkIndex: number, argCount: number): NodeDef;

export declare function ɵprd(flags: ɵNodeFlags, matchedQueries: null | [string | number, ɵQueryValueType][], token: any, value: any, deps: ([ɵDepFlags, any] | any)[]): NodeDef;

/**
 * Publishes a collection of default debug tools onto`window.ng`.
 *
 * These functions are available globally when Angular is in development
 * mode and are automatically stripped away from prod mode is on.
 */
export declare function ɵpublishDefaultGlobalUtils(): void;

/**
 * Publishes the given function to `window.ng` so that it can be
 * used from the browser console when an application is not in production.
 */
export declare function ɵpublishGlobalUtil(name: string, fn: Function): void;

export declare function ɵqud(flags: ɵNodeFlags, id: number, bindings: {
    [propName: string]: ɵQueryBindingType;
}): NodeDef;

export declare const enum ɵQueryBindingType {
    First = 0,
    All = 1
}

export declare const enum ɵQueryValueType {
    ElementRef = 0,
    RenderElement = 1,
    TemplateRef = 2,
    ViewContainerRef = 3,
    Provider = 4
}

export declare class ɵReflectionCapabilities implements PlatformReflectionCapabilities {
    private _reflect;
    constructor(reflect?: any);
    isReflectionEnabled(): boolean;
    factory<T>(t: Type<T>): (args: any[]) => T;
    private _ownParameters;
    parameters(type: Type<any>): any[][];
    private _ownAnnotations;
    annotations(typeOrFunc: Type<any>): any[];
    private _ownPropMetadata;
    propMetadata(typeOrFunc: any): {
        [key: string]: any[];
    };
    ownPropMetadata(typeOrFunc: any): {
        [key: string]: any[];
    };
    hasLifecycleHook(type: any, lcProperty: string): boolean;
    guards(type: any): {
        [key: string]: any;
    };
    getter(name: string): ɵGetterFn;
    setter(name: string): ɵSetterFn;
    method(name: string): ɵMethodFn;
    importUri(type: any): string;
    resourceUri(type: any): string;
    resolveIdentifier(name: string, moduleUrl: string, members: string[], runtime: any): any;
    resolveEnum(enumIdentifier: any, name: string): any;
}

/**
 * Register locale data to be used internally by Angular. See the
 * ["I18n guide"](guide/i18n#i18n-pipes) to know how to import additional locale data.
 *
 * The signature `registerLocaleData(data: any, extraData?: any)` is deprecated since v5.1
 */
export declare function ɵregisterLocaleData(data: any, localeId?: string | any, extraData?: any): void;

/**
 * Registers a loaded module. Should only be called from generated NgModuleFactory code.
 * @publicApi
 */
export declare function ɵregisterModuleFactory(id: string, factory: NgModuleFactory<any>): void;

export declare function ɵregisterNgModuleType(ngModuleType: ɵNgModuleType): void;

/**
 * Render3 implementation of {@link viewEngine_ComponentFactory}.
 */
export declare class ɵRender3ComponentFactory<T> extends ComponentFactory<T> {
    private componentDef;
    private ngModule?;
    selector: string;
    componentType: Type<any>;
    ngContentSelectors: string[];
    isBoundToModule: boolean;
    get inputs(): {
        propName: string;
        templateName: string;
    }[];
    get outputs(): {
        propName: string;
        templateName: string;
    }[];
    /**
     * @param componentDef The component definition.
     * @param ngModule The NgModuleRef to which the factory is bound.
     */
    constructor(componentDef: ɵComponentDef<any>, ngModule?: NgModuleRef<any> | undefined);
    create(injector: Injector, projectableNodes?: any[][] | undefined, rootSelectorOrNode?: any, ngModule?: NgModuleRef<any> | undefined): ComponentRef<T>;
}

/**
 * Represents an instance of a Component created via a {@link ComponentFactory}.
 *
 * `ComponentRef` provides access to the Component Instance as well other objects related to this
 * Component Instance and allows you to destroy the Component Instance via the {@link #destroy}
 * method.
 *
 */
export declare class ɵRender3ComponentRef<T> extends ComponentRef<T> {
    location: ElementRef;
    private _rootLView;
    private _tNode;
    destroyCbs: (() => void)[] | null;
    instance: T;
    hostView: ViewRef_2<T>;
    changeDetectorRef: ChangeDetectorRef;
    componentType: Type<T>;
    constructor(componentType: Type<T>, instance: T, location: ElementRef, _rootLView: ɵangular_packages_core_core_bp, _tNode: ɵangular_packages_core_core_be | TContainerNode | TElementContainerNode);
    get injector(): Injector;
    destroy(): void;
    onDestroy(callback: () => void): void;
}

export declare class ɵRender3NgModuleRef<T> extends NgModuleRef<T> implements InternalNgModuleRef<T> {
    _parent: Injector | null;
    _bootstrapComponents: Type<any>[];
    _r3Injector: R3Injector;
    injector: Injector;
    instance: T;
    destroyCbs: (() => void)[] | null;
    readonly componentFactoryResolver: ComponentFactoryResolver_2;
    constructor(ngModuleType: Type<T>, _parent: Injector | null);
    get(token: any, notFoundValue?: any, injectFlags?: InjectFlags): any;
    destroy(): void;
    onDestroy(callback: () => void): void;
}

/**
 * Bootstraps a Component into an existing host element and returns an instance
 * of the component.
 *
 * Use this function to bootstrap a component into the DOM tree. Each invocation
 * of this function will create a separate tree of components, injectors and
 * change detection cycles and lifetimes. To dynamically insert a new component
 * into an existing tree such that it shares the same injection, change detection
 * and object lifetime, use {@link ViewContainer#createComponent}.
 *
 * @param componentType Component to bootstrap
 * @param options Optional parameters which control bootstrapping
 */
export declare function ɵrenderComponent<T>(componentType: ɵComponentType<T> | Type<T>, opts?: CreateComponentOptions): T;

/**
 * Flags passed into template functions to determine which blocks (i.e. creation, update)
 * should be executed.
 *
 * Typically, a template runs both the creation block and the update block on initialization and
 * subsequent runs only execute the update block. However, dynamically created views require that
 * the creation block be executed separately from the update block (for backwards compat).
 */
export declare const enum ɵRenderFlags {
    Create = 1,
    Update = 2
}

export declare function ɵresetCompiledComponents(): void;

/**
 * Used to resolve resource URLs on `@Component` when used with JIT compilation.
 *
 * Example:
 * ```
 * @Component({
 *   selector: 'my-comp',
 *   templateUrl: 'my-comp.html', // This requires asynchronous resolution
 * })
 * class MyComponent{
 * }
 *
 * // Calling `renderComponent` will fail because `renderComponent` is a synchronous process
 * // and `MyComponent`'s `@Component.templateUrl` needs to be resolved asynchronously.
 *
 * // Calling `resolveComponentResources()` will resolve `@Component.templateUrl` into
 * // `@Component.template`, which allows `renderComponent` to proceed in a synchronous manner.
 *
 * // Use browser's `fetch()` function as the default resource resolution strategy.
 * resolveComponentResources(fetch).then(() => {
 *   // After resolution all URLs have been converted into `template` strings.
 *   renderComponent(MyComponent);
 * });
 *
 * ```
 *
 * NOTE: In AOT the resolution happens during compilation, and so there should be no need
 * to call this method outside JIT mode.
 *
 * @param resourceResolver a function which is responsible for returning a `Promise` to the
 * contents of the resolved URL. Browser's `fetch()` method is a good default implementation.
 */
export declare function ɵresolveComponentResources(resourceResolver: (url: string) => (Promise<string | {
    text(): Promise<string>;
}>)): Promise<void>;

/**
 * Marker interface for a value that's safe to use as HTML.
 *
 * @publicApi
 */
export declare interface ɵSafeHtml extends ɵSafeValue {
}

/**
 * Marker interface for a value that's safe to use as a URL to load executable code from.
 *
 * @publicApi
 */
export declare interface ɵSafeResourceUrl extends ɵSafeValue {
}

/**
 * Marker interface for a value that's safe to use as JavaScript.
 *
 * @publicApi
 */
export declare interface ɵSafeScript extends ɵSafeValue {
}

/**
 * Marker interface for a value that's safe to use as style (CSS).
 *
 * @publicApi
 */
export declare interface ɵSafeStyle extends ɵSafeValue {
}

/**
 * Marker interface for a value that's safe to use as a URL linking to a document.
 *
 * @publicApi
 */
export declare interface ɵSafeUrl extends ɵSafeValue {
}

/**
 * Marker interface for a value that's safe to use in a particular context.
 *
 * @publicApi
 */
export declare interface ɵSafeValue {
}

/**
 * Adds decorator, constructor, and property metadata to a given type via static metadata fields
 * on the type.
 *
 * These metadata fields can later be read with Angular's `ReflectionCapabilities` API.
 *
 * Calls to `setClassMetadata` can be marked as pure, resulting in the metadata assignments being
 * tree-shaken away during production builds.
 */
export declare function ɵsetClassMetadata(type: Type<any>, decorators: any[] | null, ctorParameters: (() => any[]) | null, propDecorators: {
    [field: string]: any;
} | null): void;

export declare function ɵsetCurrentInjector(injector: Injector | null | undefined): Injector | undefined | null;


/**
 * Tell ivy what the `document` is for this platform.
 *
 * It is only necessary to call this if the current platform is not a browser.
 *
 * @param document The object representing the global `document` in this environment.
 */
export declare function ɵsetDocument(document: Document | undefined): void;

/**
 * Sets the locale id that will be used for translations and ICU expressions.
 * This is the ivy version of `LOCALE_ID` that was defined as an injection token for the view engine
 * but is now defined as a global value.
 *
 * @param localeId
 */
export declare function ɵsetLocaleId(localeId: string): void;


export declare type ɵSetterFn = (obj: any, value: any) => void;

/** Store a value in the `data` at a given `index`. */
export declare function ɵstore<T>(tView: TView, lView: ɵangular_packages_core_core_bp, index: number, value: T): void;


export declare function ɵstringify(token: any): string;

export declare const ɵSWITCH_CHANGE_DETECTOR_REF_FACTORY__POST_R3__: typeof injectChangeDetectorRef;

export declare const ɵSWITCH_COMPILE_COMPONENT__POST_R3__: typeof ɵcompileComponent;

export declare const ɵSWITCH_COMPILE_DIRECTIVE__POST_R3__: typeof ɵcompileDirective;

export declare const ɵSWITCH_COMPILE_INJECTABLE__POST_R3__: typeof compileInjectable;

export declare const ɵSWITCH_COMPILE_NGMODULE__POST_R3__: typeof ɵcompileNgModule;

export declare const ɵSWITCH_COMPILE_PIPE__POST_R3__: typeof ɵcompilePipe;

export declare const ɵSWITCH_ELEMENT_REF_FACTORY__POST_R3__: typeof injectElementRef;


export declare const ɵSWITCH_IVY_ENABLED__POST_R3__ = true;

export declare const ɵSWITCH_RENDERER2_FACTORY__POST_R3__: typeof injectRenderer2;

export declare const ɵSWITCH_TEMPLATE_REF_FACTORY__POST_R3__: typeof injectTemplateRef;

export declare const ɵSWITCH_VIEW_CONTAINER_REF_FACTORY__POST_R3__: typeof injectViewContainerRef;

export declare function ɵted(checkIndex: number, ngContentIndex: number | null, staticText: string[]): NodeDef;

/**
 * Compute the pair of transitive scopes (compilation scope and exported scope) for a given module.
 *
 * This operation is memoized and the result is cached on the module's definition. This function can
 * be called on modules with components that have not fully compiled yet, but the result should not
 * be used until they have.
 *
 * @param moduleType module that transitive scope should be calculated for.
 */
export declare function ɵtransitiveScopesFor<T>(moduleType: Type<T>): ɵNgModuleTransitiveScopes;

/**
 * Helper function to remove all the locale data from `LOCALE_DATA`.
 */
export declare function ɵunregisterLocaleData(): void;

export declare function ɵunv(view: ViewData, nodeIdx: number, bindingIdx: number, value: any): any;

export declare function ɵunwrapSafeValue(value: ɵSafeValue): string;

export declare function ɵunwrapSafeValue<T>(value: T): T;

export declare function ɵvid(flags: ɵViewFlags, nodes: NodeDef[], updateDirectives?: null | ViewUpdateFn, updateRenderer?: null | ViewUpdateFn): ɵViewDefinition;

export declare interface ɵViewDefinition extends Definition<ViewDefinitionFactory> {
    flags: ɵViewFlags;
    updateDirectives: ViewUpdateFn;
    updateRenderer: ViewUpdateFn;
    handleEvent: ViewHandleEventFn;
    /**
     * Order: Depth first.
     * Especially providers are before elements / anchors.
     */
    nodes: NodeDef[];
    /** aggregated NodeFlags for all nodes **/
    nodeFlags: ɵNodeFlags;
    rootNodeFlags: ɵNodeFlags;
    lastRenderRootNode: NodeDef | null;
    bindingCount: number;
    outputCount: number;
    /**
     * Binary or of all query ids that are matched by one of the nodes.
     * This includes query ids from templates as well.
     * Used as a bloom filter.
     */
    nodeMatchedQueries: number;
}

/**
 * Bitmask for ViewDefinition.flags.
 */
export declare const enum ɵViewFlags {
    None = 0,
    OnPush = 2
}

/**
 * Wait on component until it is rendered.
 *
 * This function returns a `Promise` which is resolved when the component's
 * change detection is executed. This is determined by finding the scheduler
 * associated with the `component`'s render tree and waiting until the scheduler
 * flushes. If nothing is scheduled, the function returns a resolved promise.
 *
 * Example:
 * ```
 * await whenRendered(myComponent);
 * ```
 *
 * @param component Component to wait upon
 * @returns Promise which resolves when the component is rendered.
 */
export declare function ɵwhenRendered(component: any): Promise<null>;

/**
 * Advances to an element for later binding instructions.
 *
 * Used in conjunction with instructions like {@link property} to act on elements with specified
 * indices, for example those created with {@link element} or {@link elementStart}.
 *
 * ```ts
 * (rf: RenderFlags, ctx: any) => {
  *   if (rf & 1) {
  *     text(0, 'Hello');
  *     text(1, 'Goodbye')
  *     element(2, 'div');
  *   }
  *   if (rf & 2) {
  *     advance(2); // Advance twice to the <div>.
  *     property('title', 'test');
  *   }
  *  }
  * ```
  * @param delta Number of elements to advance forwards by.
  *
  * @codeGenApi
  */
export declare function ɵɵadvance(delta: number): void;

/**
 * Updates the value of or removes a bound attribute on an Element.
 *
 * Used in the case of `[attr.title]="value"`
 *
 * @param name name The name of the attribute.
 * @param value value The attribute is removed when value is `null` or `undefined`.
 *                  Otherwise the attribute value is set to the stringified value.
 * @param sanitizer An optional function used to sanitize the value.
 * @param namespace Optional namespace to use when setting the attribute.
 *
 * @codeGenApi
 */
export declare function ɵɵattribute(name: string, value: any, sanitizer?: SanitizerFn | null, namespace?: string): typeof ɵɵattribute;

/**
 *
 * Update an interpolated attribute on an element with single bound value surrounded by text.
 *
 * Used when the value passed to a property has 1 interpolated value in it:
 *
 * ```html
 * <div attr.title="prefix{{v0}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵattributeInterpolate1('title', 'prefix', v0, 'suffix');
 * ```
 *
 * @param attrName The name of the attribute to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵattributeInterpolate1(attrName: string, prefix: string, v0: any, suffix: string, sanitizer?: SanitizerFn, namespace?: string): typeof ɵɵattributeInterpolate1;

/**
 *
 * Update an interpolated attribute on an element with 2 bound values surrounded by text.
 *
 * Used when the value passed to a property has 2 interpolated values in it:
 *
 * ```html
 * <div attr.title="prefix{{v0}}-{{v1}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵattributeInterpolate2('title', 'prefix', v0, '-', v1, 'suffix');
 * ```
 *
 * @param attrName The name of the attribute to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵattributeInterpolate2(attrName: string, prefix: string, v0: any, i0: string, v1: any, suffix: string, sanitizer?: SanitizerFn, namespace?: string): typeof ɵɵattributeInterpolate2;

/**
 *
 * Update an interpolated attribute on an element with 3 bound values surrounded by text.
 *
 * Used when the value passed to a property has 3 interpolated values in it:
 *
 * ```html
 * <div attr.title="prefix{{v0}}-{{v1}}-{{v2}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵattributeInterpolate3(
 * 'title', 'prefix', v0, '-', v1, '-', v2, 'suffix');
 * ```
 *
 * @param attrName The name of the attribute to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵattributeInterpolate3(attrName: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, suffix: string, sanitizer?: SanitizerFn, namespace?: string): typeof ɵɵattributeInterpolate3;

/**
 *
 * Update an interpolated attribute on an element with 4 bound values surrounded by text.
 *
 * Used when the value passed to a property has 4 interpolated values in it:
 *
 * ```html
 * <div attr.title="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵattributeInterpolate4(
 * 'title', 'prefix', v0, '-', v1, '-', v2, '-', v3, 'suffix');
 * ```
 *
 * @param attrName The name of the attribute to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵattributeInterpolate4(attrName: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, suffix: string, sanitizer?: SanitizerFn, namespace?: string): typeof ɵɵattributeInterpolate4;

/**
 *
 * Update an interpolated attribute on an element with 5 bound values surrounded by text.
 *
 * Used when the value passed to a property has 5 interpolated values in it:
 *
 * ```html
 * <div attr.title="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵattributeInterpolate5(
 * 'title', 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, 'suffix');
 * ```
 *
 * @param attrName The name of the attribute to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵattributeInterpolate5(attrName: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, suffix: string, sanitizer?: SanitizerFn, namespace?: string): typeof ɵɵattributeInterpolate5;

/**
 *
 * Update an interpolated attribute on an element with 6 bound values surrounded by text.
 *
 * Used when the value passed to a property has 6 interpolated values in it:
 *
 * ```html
 * <div attr.title="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵattributeInterpolate6(
 *    'title', 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, 'suffix');
 * ```
 *
 * @param attrName The name of the attribute to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵattributeInterpolate6(attrName: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, suffix: string, sanitizer?: SanitizerFn, namespace?: string): typeof ɵɵattributeInterpolate6;

/**
 *
 * Update an interpolated attribute on an element with 7 bound values surrounded by text.
 *
 * Used when the value passed to a property has 7 interpolated values in it:
 *
 * ```html
 * <div attr.title="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵattributeInterpolate7(
 *    'title', 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, 'suffix');
 * ```
 *
 * @param attrName The name of the attribute to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param i5 Static value used for concatenation only.
 * @param v6 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵattributeInterpolate7(attrName: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, i5: string, v6: any, suffix: string, sanitizer?: SanitizerFn, namespace?: string): typeof ɵɵattributeInterpolate7;

/**
 *
 * Update an interpolated attribute on an element with 8 bound values surrounded by text.
 *
 * Used when the value passed to a property has 8 interpolated values in it:
 *
 * ```html
 * <div attr.title="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}-{{v7}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵattributeInterpolate8(
 *  'title', 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, '-', v7, 'suffix');
 * ```
 *
 * @param attrName The name of the attribute to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param i5 Static value used for concatenation only.
 * @param v6 Value checked for change.
 * @param i6 Static value used for concatenation only.
 * @param v7 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵattributeInterpolate8(attrName: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, i5: string, v6: any, i6: string, v7: any, suffix: string, sanitizer?: SanitizerFn, namespace?: string): typeof ɵɵattributeInterpolate8;

/**
 * Update an interpolated attribute on an element with 9 or more bound values surrounded by text.
 *
 * Used when the number of interpolated values exceeds 8.
 *
 * ```html
 * <div
 *  title="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}-{{v7}}-{{v8}}-{{v9}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵattributeInterpolateV(
 *  'title', ['prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, '-', v7, '-', v9,
 *  'suffix']);
 * ```
 *
 * @param attrName The name of the attribute to update.
 * @param values The collection of values and the strings in-between those values, beginning with
 * a string prefix and ending with a string suffix.
 * (e.g. `['prefix', value0, '-', value1, '-', value2, ..., value99, 'suffix']`)
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵattributeInterpolateV(attrName: string, values: any[], sanitizer?: SanitizerFn, namespace?: string): typeof ɵɵattributeInterpolateV;

/**
 * Update class bindings using an object literal or class-string on an element.
 *
 * This instruction is meant to apply styling via the `[class]="exp"` template bindings.
 * When classes are applied to the element they will then be updated with
 * respect to any styles/classes set via `classProp`. If any
 * classes are set to falsy then they will be removed from the element.
 *
 * Note that the styling instruction will not be applied until `stylingApply` is called.
 * Note that this will the provided classMap value to the host element if this function is called
 * within a host binding.
 *
 * @param classes A key/value map or string of CSS classes that will be added to the
 *        given element. Any missing classes (that have already been applied to the element
 *        beforehand) will be removed (unset) from the element's list of CSS classes.
 *
 * @codeGenApi
 */
export declare function ɵɵclassMap(classes: {
    [className: string]: boolean | undefined | null;
} | string | undefined | null): void;


/**
 *
 * Update an interpolated class on an element with single bound value surrounded by text.
 *
 * Used when the value passed to a property has 1 interpolated value in it:
 *
 * ```html
 * <div class="prefix{{v0}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵclassMapInterpolate1('prefix', v0, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵclassMapInterpolate1(prefix: string, v0: any, suffix: string): void;

/**
 *
 * Update an interpolated class on an element with 2 bound values surrounded by text.
 *
 * Used when the value passed to a property has 2 interpolated values in it:
 *
 * ```html
 * <div class="prefix{{v0}}-{{v1}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵclassMapInterpolate2('prefix', v0, '-', v1, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵclassMapInterpolate2(prefix: string, v0: any, i0: string, v1: any, suffix: string): void;

/**
 *
 * Update an interpolated class on an element with 3 bound values surrounded by text.
 *
 * Used when the value passed to a property has 3 interpolated values in it:
 *
 * ```html
 * <div class="prefix{{v0}}-{{v1}}-{{v2}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵclassMapInterpolate3(
 * 'prefix', v0, '-', v1, '-', v2, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵclassMapInterpolate3(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, suffix: string): void;

/**
 *
 * Update an interpolated class on an element with 4 bound values surrounded by text.
 *
 * Used when the value passed to a property has 4 interpolated values in it:
 *
 * ```html
 * <div class="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵclassMapInterpolate4(
 * 'prefix', v0, '-', v1, '-', v2, '-', v3, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵclassMapInterpolate4(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, suffix: string): void;

/**
 *
 * Update an interpolated class on an element with 5 bound values surrounded by text.
 *
 * Used when the value passed to a property has 5 interpolated values in it:
 *
 * ```html
 * <div class="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵclassMapInterpolate5(
 * 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵclassMapInterpolate5(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, suffix: string): void;

/**
 *
 * Update an interpolated class on an element with 6 bound values surrounded by text.
 *
 * Used when the value passed to a property has 6 interpolated values in it:
 *
 * ```html
 * <div class="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵclassMapInterpolate6(
 *    'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵclassMapInterpolate6(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, suffix: string): void;

/**
 *
 * Update an interpolated class on an element with 7 bound values surrounded by text.
 *
 * Used when the value passed to a property has 7 interpolated values in it:
 *
 * ```html
 * <div class="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵclassMapInterpolate7(
 *    'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param i5 Static value used for concatenation only.
 * @param v6 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵclassMapInterpolate7(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, i5: string, v6: any, suffix: string): void;

/**
 *
 * Update an interpolated class on an element with 8 bound values surrounded by text.
 *
 * Used when the value passed to a property has 8 interpolated values in it:
 *
 * ```html
 * <div class="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}-{{v7}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵclassMapInterpolate8(
 *  'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, '-', v7, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param i5 Static value used for concatenation only.
 * @param v6 Value checked for change.
 * @param i6 Static value used for concatenation only.
 * @param v7 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵclassMapInterpolate8(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, i5: string, v6: any, i6: string, v7: any, suffix: string): void;

/**
 * Update an interpolated class on an element with 9 or more bound values surrounded by text.
 *
 * Used when the number of interpolated values exceeds 8.
 *
 * ```html
 * <div
 *  class="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}-{{v7}}-{{v8}}-{{v9}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵclassMapInterpolateV(
 *  ['prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, '-', v7, '-', v9,
 *  'suffix']);
 * ```
 *.
 * @param values The collection of values and the strings in-between those values, beginning with
 * a string prefix and ending with a string suffix.
 * (e.g. `['prefix', value0, '-', value1, '-', value2, ..., value99, 'suffix']`)
 * @codeGenApi
 */
export declare function ɵɵclassMapInterpolateV(values: any[]): void;

/**
 * Update a class binding on an element with the provided value.
 *
 * This instruction is meant to handle the `[class.foo]="exp"` case and,
 * therefore, the class binding itself must already be allocated using
 * `styling` within the creation block.
 *
 * @param prop A valid CSS class (only one).
 * @param value A true/false value which will turn the class on or off.
 *
 * Note that this will apply the provided class value to the host element if this function
 * is called within a host binding function.
 *
 * @codeGenApi
 */
export declare function ɵɵclassProp(className: string, value: boolean | undefined | null): typeof ɵɵclassProp;

/**
 * @codeGenApi
 */
export declare type ɵɵComponentDefWithMeta<T, Selector extends String, ExportAs extends string[], InputMap extends {
    [key: string]: string;
}, OutputMap extends {
    [key: string]: string;
}, QueryFields extends string[]> = ɵComponentDef<T>;

/**
* Registers a synthetic host listener (e.g. `(@foo.start)`) on a component.
*
* This instruction is for compatibility purposes and is designed to ensure that a
* synthetic host listener (e.g. `@HostListener('@foo.start')`) properly gets rendered
* in the component's renderer. Normally all host listeners are evaluated with the
* parent component's renderer, but, in the case of animation @triggers, they need
* to be evaluated with the sub component's renderer (because that's where the
* animation triggers are defined).
*
* Do not use this instruction as a replacement for `listener`. This instruction
* only exists to ensure compatibility with the ViewEngine's host binding behavior.
*
* @param eventName Name of the event
* @param listenerFn The function to be called when event emits
* @param useCapture Whether or not to use capture in event listener
* @param eventTargetResolver Function that returns global target information in case this listener
* should be attached to a global object like window, document or body
 *
 * @codeGenApi
*/
export declare function ɵɵcomponentHostSyntheticListener(eventName: string, listenerFn: (e?: any) => any, useCapture?: boolean, eventTargetResolver?: GlobalTargetResolver): typeof ɵɵcomponentHostSyntheticListener;

/**
 * Creates an LContainer for inline views, e.g.
 *
 * % if (showing) {
 *   <div></div>
 * % }
 *
 * @param index The index of the container in the data array
 *
 * @codeGenApi
 */
export declare function ɵɵcontainer(index: number): void;

/**
 * Marks the end of the LContainer.
 *
 * Marking the end of LContainer is the time when to child views get inserted or removed.
 *
 * @codeGenApi
 */
export declare function ɵɵcontainerRefreshEnd(): void;

/**
 * Sets a container up to receive views.
 *
 * @param index The index of the container in the data array
 *
 * @codeGenApi
 */
export declare function ɵɵcontainerRefreshStart(index: number): void;

/**
 * Registers a QueryList, associated with a content query, for later refresh (part of a view
 * refresh).
 *
 * @param directiveIndex Current directive index
 * @param predicate The type for which the query will search
 * @param descend Whether or not to descend into children
 * @param read What to save in the query
 * @returns QueryList<T>
 *
 * @codeGenApi
 */
export declare function ɵɵcontentQuery<T>(directiveIndex: number, predicate: Type<any> | string[], descend: boolean, read?: any): void;

/**
 * Copies the fields not handled by the `ɵɵInheritDefinitionFeature` from the supertype of a
 * definition.
 *
 * This exists primarily to support ngcc migration of an existing View Engine pattern, where an
 * entire decorator is inherited from a parent to a child class. When ngcc detects this case, it
 * generates a skeleton definition on the child class, and applies this feature.
 *
 * The `ɵɵCopyDefinitionFeature` then copies any needed fields from the parent class' definition,
 * including things like the component template function.
 *
 * @param definition The definition of a child class which inherits from a parent class with its
 * own definition.
 *
 * @codeGenApi
 */
export declare function ɵɵCopyDefinitionFeature(definition: ɵDirectiveDef<any> | ɵComponentDef<any>): void;

/**
 * The default style sanitizer will handle sanitization for style properties by
 * sanitizing any CSS property that can include a `url` value (usually image-based properties)
 *
 * @publicApi
 */
export declare const ɵɵdefaultStyleSanitizer: StyleSanitizeFn;

/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵcmp = defineComponent({
 *     ...
 *   });
 * }
 * ```
 * @codeGenApi
 */
export declare function ɵɵdefineComponent<T>(componentDefinition: {
    /**
     * Directive type, needed to configure the injector.
     */
    type: Type<T>;
    /** The selectors that will be used to match nodes to this component. */
    selectors?: ɵCssSelectorList;
    /**
     * The number of nodes, local refs, and pipes in this component template.
     *
     * Used to calculate the length of this component's LView array, so we
     * can pre-fill the array and set the binding start index.
     */
    decls: number;
    /**
     * The number of bindings in this component template (including pure fn bindings).
     *
     * Used to calculate the length of this component's LView array, so we
     * can pre-fill the array and set the host binding start index.
     */
    vars: number;
    /**
     * A map of input names.
     *
     * The format is in: `{[actualPropertyName: string]:(string|[string, string])}`.
     *
     * Given:
     * ```
     * class MyComponent {
     *   @Input()
     *   publicInput1: string;
     *
     *   @Input('publicInput2')
     *   declaredInput2: string;
     * }
     * ```
     *
     * is described as:
     * ```
     * {
     *   publicInput1: 'publicInput1',
     *   declaredInput2: ['publicInput2', 'declaredInput2'],
     * }
     * ```
     *
     * Which the minifier may translate to:
     * ```
     * {
     *   minifiedPublicInput1: 'publicInput1',
     *   minifiedDeclaredInput2: ['publicInput2', 'declaredInput2'],
     * }
     * ```
     *
     * This allows the render to re-construct the minified, public, and declared names
     * of properties.
     *
     * NOTE:
     *  - Because declared and public name are usually same we only generate the array
     *    `['public', 'declared']` format when they differ.
     *  - The reason why this API and `outputs` API is not the same is that `NgOnChanges` has
     *    inconsistent behavior in that it uses declared names rather than minified or public. For
     *    this reason `NgOnChanges` will be deprecated and removed in future version and this
     *    API will be simplified to be consistent with `output`.
     */
    inputs?: {
        [P in keyof T]?: string | [string, string];
    };
    /**
     * A map of output names.
     *
     * The format is in: `{[actualPropertyName: string]:string}`.
     *
     * Which the minifier may translate to: `{[minifiedPropertyName: string]:string}`.
     *
     * This allows the render to re-construct the minified and non-minified names
     * of properties.
     */
    outputs?: {
        [P in keyof T]?: string;
    };
    /**
     * Function executed by the parent template to allow child directive to apply host bindings.
     */
    hostBindings?: HostBindingsFunction<T>;
    /**
     * The number of bindings in this directive `hostBindings` (including pure fn bindings).
     *
     * Used to calculate the length of the component's LView array, so we
     * can pre-fill the array and set the host binding start index.
     */
    hostVars?: number;
    /**
     * Assign static attribute values to a host element.
     *
     * This property will assign static attribute values as well as class and style
     * values to a host element. Since attribute values can consist of different types of values, the
     * `hostAttrs` array must include the values in the following format:
     *
     * attrs = [
     *   // static attributes (like `title`, `name`, `id`...)
     *   attr1, value1, attr2, value,
     *
     *   // a single namespace value (like `x:id`)
     *   NAMESPACE_MARKER, namespaceUri1, name1, value1,
     *
     *   // another single namespace value (like `x:name`)
     *   NAMESPACE_MARKER, namespaceUri2, name2, value2,
     *
     *   // a series of CSS classes that will be applied to the element (no spaces)
     *   CLASSES_MARKER, class1, class2, class3,
     *
     *   // a series of CSS styles (property + value) that will be applied to the element
     *   STYLES_MARKER, prop1, value1, prop2, value2
     * ]
     *
     * All non-class and non-style attributes must be defined at the start of the list
     * first before all class and style values are set. When there is a change in value
     * type (like when classes and styles are introduced) a marker must be used to separate
     * the entries. The marker values themselves are set via entries found in the
     * [AttributeMarker] enum.
     */
    hostAttrs?: TAttributes;
    /**
     * Function to create instances of content queries associated with a given directive.
     */
    contentQueries?: ContentQueriesFunction<T>;
    /**
     * Defines the name that can be used in the template to assign this directive to a variable.
     *
     * See: {@link Directive.exportAs}
     */
    exportAs?: string[];
    /**
     * Template function use for rendering DOM.
     *
     * This function has following structure.
     *
     * ```
     * function Template<T>(ctx:T, creationMode: boolean) {
     *   if (creationMode) {
     *     // Contains creation mode instructions.
     *   }
     *   // Contains binding update instructions
     * }
     * ```
     *
     * Common instructions are:
     * Creation mode instructions:
     *  - `elementStart`, `elementEnd`
     *  - `text`
     *  - `container`
     *  - `listener`
     *
     * Binding update instructions:
     * - `bind`
     * - `elementAttribute`
     * - `elementProperty`
     * - `elementClass`
     * - `elementStyle`
     *
     */
    template: ComponentTemplate<T>;
    /**
     * Constants for the nodes in the component's view.
     * Includes attribute arrays, local definition arrays etc.
     */
    consts?: TConstants;
    /**
     * An array of `ngContent[selector]` values that were found in the template.
     */
    ngContentSelectors?: string[];
    /**
     * Additional set of instructions specific to view query processing. This could be seen as a
     * set of instruction to be inserted into the template function.
     *
     * Query-related instructions need to be pulled out to a specific function as a timing of
     * execution is different as compared to all other instructions (after change detection hooks but
     * before view hooks).
     */
    viewQuery?: ViewQueriesFunction<T> | null;
    /**
     * A list of optional features to apply.
     *
     * See: {@link NgOnChangesFeature}, {@link ProvidersFeature}
     */
    features?: ComponentDefFeature[];
    /**
     * Defines template and style encapsulation options available for Component's {@link Component}.
     */
    encapsulation?: ViewEncapsulation;
    /**
     * Defines arbitrary developer-defined data to be stored on a renderer instance.
     * This is useful for renderers that delegate to other renderers.
     *
     * see: animation
     */
    data?: {
        [kind: string]: any;
    };
    /**
     * A set of styles that the component needs to be present for component to render correctly.
     */
    styles?: string[];
    /**
     * The strategy that the default change detector uses to detect changes.
     * When set, takes effect the next time change detection is triggered.
     */
    changeDetection?: ChangeDetectionStrategy;
    /**
     * Registry of directives and components that may be found in this component's view.
     *
     * The property is either an array of `DirectiveDef`s or a function which returns the array of
     * `DirectiveDef`s. The function is necessary to be able to support forward declarations.
     */
    directives?: DirectiveTypesOrFactory | null;
    /**
     * Registry of pipes that may be found in this component's view.
     *
     * The property is either an array of `PipeDefs`s or a function which returns the array of
     * `PipeDefs`s. The function is necessary to be able to support forward declarations.
     */
    pipes?: PipeTypesOrFactory | null;
    /**
     * The set of schemas that declare elements to be allowed in the component's template.
     */
    schemas?: SchemaMetadata[] | null;
}): never;

/**
 * Create a directive definition object.
 *
 * # Example
 * ```ts
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵdir = ɵɵdefineDirective({
 *     ...
 *   });
 * }
 * ```
 *
 * @codeGenApi
 */
export declare const ɵɵdefineDirective: <T>(directiveDefinition: {
    /**
     * Directive type, needed to configure the injector.
     */
    type: Type<T>;
    /** The selectors that will be used to match nodes to this directive. */
    selectors?: ɵCssSelectorList | undefined;
    /**
     * A map of input names.
     *
     * The format is in: `{[actualPropertyName: string]:(string|[string, string])}`.
     *
     * Given:
     * ```
     * class MyComponent {
     *   @Input()
     *   publicInput1: string;
     *
     *   @Input('publicInput2')
     *   declaredInput2: string;
     * }
     * ```
     *
     * is described as:
     * ```
     * {
     *   publicInput1: 'publicInput1',
     *   declaredInput2: ['declaredInput2', 'publicInput2'],
     * }
     * ```
     *
     * Which the minifier may translate to:
     * ```
     * {
     *   minifiedPublicInput1: 'publicInput1',
     *   minifiedDeclaredInput2: [ 'publicInput2', 'declaredInput2'],
     * }
     * ```
     *
     * This allows the render to re-construct the minified, public, and declared names
     * of properties.
     *
     * NOTE:
     *  - Because declared and public name are usually same we only generate the array
     *    `['declared', 'public']` format when they differ.
     *  - The reason why this API and `outputs` API is not the same is that `NgOnChanges` has
     *    inconsistent behavior in that it uses declared names rather than minified or public. For
     *    this reason `NgOnChanges` will be deprecated and removed in future version and this
     *    API will be simplified to be consistent with `output`.
     */
    inputs?: { [P in keyof T]?: string | [string, string] | undefined; } | undefined;
    /**
     * A map of output names.
     *
     * The format is in: `{[actualPropertyName: string]:string}`.
     *
     * Which the minifier may translate to: `{[minifiedPropertyName: string]:string}`.
     *
     * This allows the render to re-construct the minified and non-minified names
     * of properties.
     */
    outputs?: { [P_1 in keyof T]?: string | undefined; } | undefined;
    /**
     * A list of optional features to apply.
     *
     * See: {@link NgOnChangesFeature}, {@link ProvidersFeature}, {@link InheritDefinitionFeature}
     */
    features?: DirectiveDefFeature[] | undefined;
    /**
     * Function executed by the parent template to allow child directive to apply host bindings.
     */
    hostBindings?: HostBindingsFunction<T> | undefined;
    /**
     * The number of bindings in this directive `hostBindings` (including pure fn bindings).
     *
     * Used to calculate the length of the component's LView array, so we
     * can pre-fill the array and set the host binding start index.
     */
    hostVars?: number | undefined;
    /**
     * Assign static attribute values to a host element.
     *
     * This property will assign static attribute values as well as class and style
     * values to a host element. Since attribute values can consist of different types of values, the
     * `hostAttrs` array must include the values in the following format:
     *
     * attrs = [
     *   // static attributes (like `title`, `name`, `id`...)
     *   attr1, value1, attr2, value,
     *
     *   // a single namespace value (like `x:id`)
     *   NAMESPACE_MARKER, namespaceUri1, name1, value1,
     *
     *   // another single namespace value (like `x:name`)
     *   NAMESPACE_MARKER, namespaceUri2, name2, value2,
     *
     *   // a series of CSS classes that will be applied to the element (no spaces)
     *   CLASSES_MARKER, class1, class2, class3,
     *
     *   // a series of CSS styles (property + value) that will be applied to the element
     *   STYLES_MARKER, prop1, value1, prop2, value2
     * ]
     *
     * All non-class and non-style attributes must be defined at the start of the list
     * first before all class and style values are set. When there is a change in value
     * type (like when classes and styles are introduced) a marker must be used to separate
     * the entries. The marker values themselves are set via entries found in the
     * [AttributeMarker] enum.
     */
    hostAttrs?: TAttributes | undefined;
    /**
     * Function to create instances of content queries associated with a given directive.
     */
    contentQueries?: ContentQueriesFunction<T> | undefined;
    /**
     * Additional set of instructions specific to view query processing. This could be seen as a
     * set of instructions to be inserted into the template function.
     */
    viewQuery?: ViewQueriesFunction<T> | null | undefined;
    /**
     * Defines the name that can be used in the template to assign this directive to a variable.
     *
     * See: {@link Directive.exportAs}
     */
    exportAs?: string[] | undefined;
}) => never;

/**
 * Construct an `InjectableDef` which defines how a token will be constructed by the DI system, and
 * in which injectors (if any) it will be available.
 *
 * This should be assigned to a static `ɵprov` field on a type, which will then be an
 * `InjectableType`.
 *
 * Options:
 * * `providedIn` determines which injectors will include the injectable, by either associating it
 *   with an `@NgModule` or other `InjectorType`, or by specifying that this injectable should be
 *   provided in the `'root'` injector, which will be the application-level injector in most apps.
 * * `factory` gives the zero argument function which will create an instance of the injectable.
 *   The factory can call `inject` to access the `Injector` and request injection of dependencies.
 *
 * @codeGenApi
 */
export declare function ɵɵdefineInjectable<T>(opts: {
    token: unknown;
    providedIn?: Type<any> | 'root' | 'platform' | 'any' | null;
    factory: () => T;
}): never;

/**
 * Construct an `InjectorDef` which configures an injector.
 *
 * This should be assigned to a static injector def (`ɵinj`) field on a type, which will then be an
 * `InjectorType`.
 *
 * Options:
 *
 * * `factory`: an `InjectorType` is an instantiable type, so a zero argument `factory` function to
 *   create the type must be provided. If that factory function needs to inject arguments, it can
 *   use the `inject` function.
 * * `providers`: an optional array of providers to add to the injector. Each provider must
 *   either have a factory or point to a type which has a `ɵprov` static property (the
 *   type must be an `InjectableType`).
 * * `imports`: an optional array of imports of other `InjectorType`s or `InjectorTypeWithModule`s
 *   whose providers will also be added to the injector. Locally provided types will override
 *   providers from imports.
 *
 * @publicApi
 */
export declare function ɵɵdefineInjector(options: {
    factory: () => any;
    providers?: any[];
    imports?: any[];
}): never;

/**
 * @codeGenApi
 */
export declare function ɵɵdefineNgModule<T>(def: {
    /** Token representing the module. Used by DI. */
    type: T;
    /** List of components to bootstrap. */
    bootstrap?: Type<any>[] | (() => Type<any>[]);
    /** List of components, directives, and pipes declared by this module. */
    declarations?: Type<any>[] | (() => Type<any>[]);
    /** List of modules or `ModuleWithProviders` imported by this module. */
    imports?: Type<any>[] | (() => Type<any>[]);
    /**
     * List of modules, `ModuleWithProviders`, components, directives, or pipes exported by this
     * module.
     */
    exports?: Type<any>[] | (() => Type<any>[]);
    /** The set of schemas that declare elements to be allowed in the NgModule. */
    schemas?: SchemaMetadata[] | null;
    /** Unique ID for the module that is used with `getModuleFactory`. */
    id?: string | null;
}): never;

/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ɵpipe = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @param pipeDef Pipe definition generated by the compiler
 *
 * @codeGenApi
 */
export declare function ɵɵdefinePipe<T>(pipeDef: {
    /** Name of the pipe. Used for matching pipes in template to pipe defs. */
    name: string;
    /** Pipe class reference. Needed to extract pipe lifecycle hooks. */
    type: Type<T>;
    /** Whether the pipe is pure. */
    pure?: boolean;
}): never;

/**
 * @codeGenApi
 */
export declare type ɵɵDirectiveDefWithMeta<T, Selector extends string, ExportAs extends string[], InputMap extends {
    [key: string]: string;
}, OutputMap extends {
    [key: string]: string;
}, QueryFields extends string[]> = ɵDirectiveDef<T>;

/**
 * Returns the value associated to the given token from the injectors.
 *
 * `directiveInject` is intended to be used for directive, component and pipe factories.
 *  All other injection use `inject` which does not walk the node injector tree.
 *
 * Usage example (in factory function):
 *
 * ```ts
 * class SomeDirective {
 *   constructor(directive: DirectiveA) {}
 *
 *   static ɵdir = ɵɵdefineDirective({
 *     type: SomeDirective,
 *     factory: () => new SomeDirective(ɵɵdirectiveInject(DirectiveA))
 *   });
 * }
 * ```
 * @param token the type or token to inject
 * @param flags Injection flags
 * @returns the value from the injector or `null` when not found
 *
 * @codeGenApi
 */
export declare function ɵɵdirectiveInject<T>(token: Type<T> | InjectionToken<T>): T;

export declare function ɵɵdirectiveInject<T>(token: Type<T> | InjectionToken<T>, flags: InjectFlags): T;

/**
 * Disables directive matching on element.
 *
 *  * Example:
 * ```
 * <my-comp my-directive>
 *   Should match component / directive.
 * </my-comp>
 * <div ngNonBindable>
 *   <!-- ɵɵdisableBindings() -->
 *   <my-comp my-directive>
 *     Should not match component / directive because we are in ngNonBindable.
 *   </my-comp>
 *   <!-- ɵɵenableBindings() -->
 * </div>
 * ```
 *
 * @codeGenApi
 */
export declare function ɵɵdisableBindings(): void;

/**
 * Creates an empty element using {@link elementStart} and {@link elementEnd}
 *
 * @param index Index of the element in the data array
 * @param name Name of the DOM Node
 * @param attrsIndex Index of the element's attributes in the `consts` array.
 * @param localRefsIndex Index of the element's local references in the `consts` array.
 *
 * @codeGenApi
 */
export declare function ɵɵelement(index: number, name: string, attrsIndex?: number | null, localRefsIndex?: number): void;

/**
 * Creates an empty logical container using {@link elementContainerStart}
 * and {@link elementContainerEnd}
 *
 * @param index Index of the element in the LView array
 * @param attrsIndex Index of the container attributes in the `consts` array.
 * @param localRefsIndex Index of the container's local references in the `consts` array.
 *
 * @codeGenApi
 */
export declare function ɵɵelementContainer(index: number, attrsIndex?: number | null, localRefsIndex?: number): void;

/**
 * Mark the end of the <ng-container>.
 *
 * @codeGenApi
 */
export declare function ɵɵelementContainerEnd(): void;

/**
 * Creates a logical container for other nodes (<ng-container>) backed by a comment node in the DOM.
 * The instruction must later be followed by `elementContainerEnd()` call.
 *
 * @param index Index of the element in the LView array
 * @param attrsIndex Index of the container attributes in the `consts` array.
 * @param localRefsIndex Index of the container's local references in the `consts` array.
 *
 * Even if this instruction accepts a set of attributes no actual attribute values are propagated to
 * the DOM (as a comment node can't have attributes). Attributes are here only for directive
 * matching purposes and setting initial inputs of directives.
 *
 * @codeGenApi
 */
export declare function ɵɵelementContainerStart(index: number, attrsIndex?: number | null, localRefsIndex?: number): void;

/**
 * Mark the end of the element.
 *
 * @codeGenApi
 */
export declare function ɵɵelementEnd(): void;


/**
 * Create DOM element. The instruction must later be followed by `elementEnd()` call.
 *
 * @param index Index of the element in the LView array
 * @param name Name of the DOM Node
 * @param attrsIndex Index of the element's attributes in the `consts` array.
 * @param localRefsIndex Index of the element's local references in the `consts` array.
 *
 * Attributes and localRefs are passed as an array of strings where elements with an even index
 * hold an attribute name and elements with an odd index hold an attribute value, ex.:
 * ['id', 'warning5', 'class', 'alert']
 *
 * @codeGenApi
 */
export declare function ɵɵelementStart(index: number, name: string, attrsIndex?: number | null, localRefsIndex?: number): void;

/**
 * Marks the end of an embedded view.
 *
 * @codeGenApi
 */
export declare function ɵɵembeddedViewEnd(): void;

/**
 * Marks the start of an embedded view.
 *
 * @param viewBlockId The ID of this view
 * @return boolean Whether or not this view is in creation mode
 *
 * @codeGenApi
 */
export declare function ɵɵembeddedViewStart(viewBlockId: number, decls: number, vars: number): ɵRenderFlags;

/**
 * Enables directive matching on elements.
 *
 *  * Example:
 * ```
 * <my-comp my-directive>
 *   Should match component / directive.
 * </my-comp>
 * <div ngNonBindable>
 *   <!-- ɵɵdisableBindings() -->
 *   <my-comp my-directive>
 *     Should not match component / directive because we are in ngNonBindable.
 *   </my-comp>
 *   <!-- ɵɵenableBindings() -->
 * </div>
 * ```
 *
 * @codeGenApi
 */
export declare function ɵɵenableBindings(): void;

/**
 * @codeGenApi
 */
export declare type ɵɵFactoryDef<T> = () => T;

/**
 * Returns the current OpaqueViewState instance.
 *
 * Used in conjunction with the restoreView() instruction to save a snapshot
 * of the current view and restore it when listeners are invoked. This allows
 * walking the declaration view tree in listeners to get vars from parent views.
 *
 * @codeGenApi
 */
export declare function ɵɵgetCurrentView(): OpaqueViewState;

/**
 * @codeGenApi
 */
export declare function ɵɵgetFactoryOf<T>(type: Type<any>): FactoryFn<T> | null;

/**
 * @codeGenApi
 */
export declare function ɵɵgetInheritedFactory<T>(type: Type<any>): (type: Type<T>) => T;

/**
 * Update a property on a host element. Only applies to native node properties, not inputs.
 *
 * Operates on the element selected by index via the {@link select} instruction.
 *
 * @param propName Name of property. Because it is going to DOM, this is not subject to
 *        renaming as part of minification.
 * @param value New value to write.
 * @param sanitizer An optional function used to sanitize the value.
 * @returns This function returns itself so that it may be chained
 * (e.g. `property('name', ctx.name)('title', ctx.title)`)
 *
 * @codeGenApi
 */
export declare function ɵɵhostProperty<T>(propName: string, value: T, sanitizer?: SanitizerFn | null): typeof ɵɵhostProperty;

/**
 *
 * Use this instruction to create a translation block that doesn't contain any placeholder.
 * It calls both {@link i18nStart} and {@link i18nEnd} in one instruction.
 *
 * The translation `message` is the value which is locale specific. The translation string may
 * contain placeholders which associate inner elements and sub-templates within the translation.
 *
 * The translation `message` placeholders are:
 * - `�{index}(:{block})�`: *Binding Placeholder*: Marks a location where an expression will be
 *   interpolated into. The placeholder `index` points to the expression binding index. An optional
 *   `block` that matches the sub-template in which it was declared.
 * - `�#{index}(:{block})�`/`�/#{index}(:{block})�`: *Element Placeholder*:  Marks the beginning
 *   and end of DOM element that were embedded in the original translation block. The placeholder
 *   `index` points to the element index in the template instructions set. An optional `block` that
 *   matches the sub-template in which it was declared.
 * - `�*{index}:{block}�`/`�/*{index}:{block}�`: *Sub-template Placeholder*: Sub-templates must be
 *   split up and translated separately in each angular template function. The `index` points to the
 *   `template` instruction index. A `block` that matches the sub-template in which it was declared.
 *
 * @param index A unique index of the translation in the static block.
 * @param message The translation message.
 * @param subTemplateIndex Optional sub-template index in the `message`.
 *
 * @codeGenApi
 */
export declare function ɵɵi18n(index: number, message: string, subTemplateIndex?: number): void;

/**
 * Updates a translation block or an i18n attribute when the bindings have changed.
 *
 * @param index Index of either {@link i18nStart} (translation block) or {@link i18nAttributes}
 * (i18n attribute) on which it should update the content.
 *
 * @codeGenApi
 */
export declare function ɵɵi18nApply(index: number): void;

/**
 * Marks a list of attributes as translatable.
 *
 * @param index A unique index in the static block
 * @param values
 *
 * @codeGenApi
 */
export declare function ɵɵi18nAttributes(index: number, values: string[]): void;

/**
 * Translates a translation block marked by `i18nStart` and `i18nEnd`. It inserts the text/ICU nodes
 * into the render tree, moves the placeholder nodes and removes the deleted nodes.
 *
 * @codeGenApi
 */
export declare function ɵɵi18nEnd(): void;

/**
 * Stores the values of the bindings during each update cycle in order to determine if we need to
 * update the translated nodes.
 *
 * @param value The binding's value
 * @returns This function returns itself so that it may be chained
 * (e.g. `i18nExp(ctx.name)(ctx.title)`)
 *
 * @codeGenApi
 */
export declare function ɵɵi18nExp<T>(value: T): typeof ɵɵi18nExp;

/**
 * Handles message string post-processing for internationalization.
 *
 * Handles message string post-processing by transforming it from intermediate
 * format (that might contain some markers that we need to replace) to the final
 * form, consumable by i18nStart instruction. Post processing steps include:
 *
 * 1. Resolve all multi-value cases (like [�*1:1��#2:1�|�#4:1�|�5�])
 * 2. Replace all ICU vars (like "VAR_PLURAL")
 * 3. Replace all placeholders used inside ICUs in a form of {PLACEHOLDER}
 * 4. Replace all ICU references with corresponding values (like �ICU_EXP_ICU_1�)
 *    in case multiple ICUs have the same placeholder name
 *
 * @param message Raw translation string for post processing
 * @param replacements Set of replacements that should be applied
 *
 * @returns Transformed string that can be consumed by i18nStart instruction
 *
 * @codeGenApi
 */
export declare function ɵɵi18nPostprocess(message: string, replacements?: {
    [key: string]: (string | string[]);
}): string;

/**
 * Marks a block of text as translatable.
 *
 * The instructions `i18nStart` and `i18nEnd` mark the translation block in the template.
 * The translation `message` is the value which is locale specific. The translation string may
 * contain placeholders which associate inner elements and sub-templates within the translation.
 *
 * The translation `message` placeholders are:
 * - `�{index}(:{block})�`: *Binding Placeholder*: Marks a location where an expression will be
 *   interpolated into. The placeholder `index` points to the expression binding index. An optional
 *   `block` that matches the sub-template in which it was declared.
 * - `�#{index}(:{block})�`/`�/#{index}(:{block})�`: *Element Placeholder*:  Marks the beginning
 *   and end of DOM element that were embedded in the original translation block. The placeholder
 *   `index` points to the element index in the template instructions set. An optional `block` that
 *   matches the sub-template in which it was declared.
 * - `�!{index}(:{block})�`/`�/!{index}(:{block})�`: *Projection Placeholder*:  Marks the
 *   beginning and end of <ng-content> that was embedded in the original translation block.
 *   The placeholder `index` points to the element index in the template instructions set.
 *   An optional `block` that matches the sub-template in which it was declared.
 * - `�*{index}:{block}�`/`�/*{index}:{block}�`: *Sub-template Placeholder*: Sub-templates must be
 *   split up and translated separately in each angular template function. The `index` points to the
 *   `template` instruction index. A `block` that matches the sub-template in which it was declared.
 *
 * @param index A unique index of the translation in the static block.
 * @param message The translation message.
 * @param subTemplateIndex Optional sub-template index in the `message`.
 *
 * @codeGenApi
 */
export declare function ɵɵi18nStart(index: number, message: string, subTemplateIndex?: number): void;

/**
 * Merges the definition from a super class to a sub class.
 * @param definition The definition that is a SubClass of another directive of component
 *
 * @codeGenApi
 */
export declare function ɵɵInheritDefinitionFeature(definition: ɵDirectiveDef<any> | ɵComponentDef<any>): void;

/**
 * Generated instruction: Injects a token from the currently active injector.
 *
 * Must be used in the context of a factory function such as one defined for an
 * `InjectionToken`. Throws an error if not called from such a context.
 *
 * (Additional documentation moved to `inject`, as it is the public API, and an alias for this instruction)
 *
 * @see inject
 * @codeGenApi
 */
export declare function ɵɵinject<T>(token: Type<T> | InjectionToken<T>): T;

export declare function ɵɵinject<T>(token: Type<T> | InjectionToken<T>, flags?: InjectFlags): T | null;

/**
 * Information about how a type or `InjectionToken` interfaces with the DI system.
 *
 * At a minimum, this includes a `factory` which defines how to create the given type `T`, possibly
 * requesting injection of other types if necessary.
 *
 * Optionally, a `providedIn` parameter specifies that the given type belongs to a particular
 * `InjectorDef`, `NgModule`, or a special scope (e.g. `'root'`). A value of `null` indicates
 * that the injectable does not belong to any scope.
 *
 * NOTE: This is a private type and should not be exported
 *
 * @publicApi
 */
export declare interface ɵɵInjectableDef<T> {
    /**
     * Specifies that the given type belongs to a particular injector:
     * - `InjectorType` such as `NgModule`,
     * - `'root'` the root injector
     * - `'any'` all injectors.
     * - `null`, does not belong to any injector. Must be explicitly listed in the injector
     *   `providers`.
     */
    providedIn: InjectorType<any> | 'root' | 'platform' | 'any' | null;
    /**
     * The token to which this definition belongs.
     *
     * Note that this may not be the same as the type that the `factory` will create.
     */
    token: unknown;
    /**
     * Factory method to execute to create an instance of the injectable.
     */
    factory: (t?: Type<any>) => T;
    /**
     * In a case of no explicit injector, a location where the instance of the injectable is stored.
     */
    value: T | undefined;
}

/**
 * Facade for the attribute injection from DI.
 *
 * @codeGenApi
 */
export declare function ɵɵinjectAttribute(attrNameToInject: string): string | null;

/**
 * Information about the providers to be included in an `Injector` as well as how the given type
 * which carries the information should be created by the DI system.
 *
 * An `InjectorDef` can import other types which have `InjectorDefs`, forming a deep nested
 * structure of providers with a defined priority (identically to how `NgModule`s also have
 * an import/dependency structure).
 *
 * NOTE: This is a private type and should not be exported
 *
 * @publicApi
 */
export declare interface ɵɵInjectorDef<T> {
    factory: () => T;
    providers: (Type<any> | ValueProvider | ExistingProvider | FactoryProvider | ConstructorProvider | StaticClassProvider | ClassProvider | any[])[];
    imports: (InjectorType<any> | InjectorTypeWithProviders<any>)[];
}

/**
 * Returns the appropriate `ChangeDetectorRef` for a pipe.
 *
 * @codeGenApi
 */
export declare function ɵɵinjectPipeChangeDetectorRef(flags?: InjectFlags): ChangeDetectorRef | null;

/**
 * Throws an error indicating that a factory function could not be generated by the compiler for a
 * particular class.
 *
 * This instruction allows the actual error message to be optimized away when ngDevMode is turned
 * off, saving bytes of generated code while still providing a good experience in dev mode.
 *
 * The name of the class is not mentioned here, but will be in the generated factory function name
 * and thus in the stack trace.
 *
 * @codeGenApi
 */
export declare function ɵɵinvalidFactory(): never;

/**
 * Throws an error indicating that a factory function could not be generated by the compiler for a
 * particular class.
 *
 * This instruction allows the actual error message to be optimized away when ngDevMode is turned
 * off, saving bytes of generated code while still providing a good experience in dev mode.
 *
 * The name of the class is not mentioned here, but will be in the generated factory function name
 * and thus in the stack trace.
 *
 * @codeGenApi
 */
export declare function ɵɵinvalidFactoryDep(index: number): never;

/**
 * Adds an event listener to the current node.
 *
 * If an output exists on one of the node's directives, it also subscribes to the output
 * and saves the subscription for later cleanup.
 *
 * @param eventName Name of the event
 * @param listenerFn The function to be called when event emits
 * @param useCapture Whether or not to use capture in event listener
 * @param eventTargetResolver Function that returns global target information in case this listener
 * should be attached to a global object like window, document or body
 *
 * @codeGenApi
 */
export declare function ɵɵlistener(eventName: string, listenerFn: (e?: any) => any, useCapture?: boolean, eventTargetResolver?: GlobalTargetResolver): typeof ɵɵlistener;

/**
 * Loads a QueryList corresponding to the current view or content query.
 *
 * @codeGenApi
 */
export declare function ɵɵloadQuery<T>(): QueryList<T>;

/**
 * Sets the namespace used to create elements to `null`, which forces element creation to use
 * `createElement` rather than `createElementNS`.
 *
 * @codeGenApi
 */
export declare function ɵɵnamespaceHTML(): void;

/**
 * Sets the namespace used to create elements to `'http://www.w3.org/1998/MathML/'` in global state.
 *
 * @codeGenApi
 */
export declare function ɵɵnamespaceMathML(): void;

/**
 * Sets the namespace used to create elements to `'http://www.w3.org/2000/svg'` in global state.
 *
 * @codeGenApi
 */
export declare function ɵɵnamespaceSVG(): void;

/**
 * Retrieves a context at the level specified and saves it as the global, contextViewData.
 * Will get the next level up if level is not specified.
 *
 * This is used to save contexts of parent views so they can be bound in embedded views, or
 * in conjunction with reference() to bind a ref from a parent view.
 *
 * @param level The relative level of the view from which to grab context compared to contextVewData
 * @returns context
 *
 * @codeGenApi
 */
export declare function ɵɵnextContext<T = any>(level?: number): T;

/**
 * @publicApi
 */
export declare type ɵɵNgModuleDefWithMeta<T, Declarations, Imports, Exports> = ɵNgModuleDef<T>;

/**
 * The NgOnChangesFeature decorates a component with support for the ngOnChanges
 * lifecycle hook, so it should be included in any component that implements
 * that hook.
 *
 * If the component or directive uses inheritance, the NgOnChangesFeature MUST
 * be included as a feature AFTER {@link InheritDefinitionFeature}, otherwise
 * inherited properties will not be propagated to the ngOnChanges lifecycle
 * hook.
 *
 * Example usage:
 *
 * ```
 * static ɵcmp = defineComponent({
 *   ...
 *   inputs: {name: 'publicName'},
 *   features: [NgOnChangesFeature]
 * });
 * ```
 *
 * @codeGenApi
 */
export declare function ɵɵNgOnChangesFeature<T>(definition: ɵDirectiveDef<T>): void;


/**
 * Create a pipe.
 *
 * @param index Pipe index where the pipe will be stored.
 * @param pipeName The name of the pipe
 * @returns T the instance of the pipe.
 *
 * @codeGenApi
 */
export declare function ɵɵpipe(index: number, pipeName: string): any;

/**
 * Invokes a pipe with 1 arguments.
 *
 * This instruction acts as a guard to {@link PipeTransform#transform} invoking
 * the pipe only when an input to the pipe changes.
 *
 * @param index Pipe index where the pipe was stored on creation.
 * @param slotOffset the offset in the reserved slot space
 * @param v1 1st argument to {@link PipeTransform#transform}.
 *
 * @codeGenApi
 */
export declare function ɵɵpipeBind1(index: number, slotOffset: number, v1: any): any;

/**
 * Invokes a pipe with 2 arguments.
 *
 * This instruction acts as a guard to {@link PipeTransform#transform} invoking
 * the pipe only when an input to the pipe changes.
 *
 * @param index Pipe index where the pipe was stored on creation.
 * @param slotOffset the offset in the reserved slot space
 * @param v1 1st argument to {@link PipeTransform#transform}.
 * @param v2 2nd argument to {@link PipeTransform#transform}.
 *
 * @codeGenApi
 */
export declare function ɵɵpipeBind2(index: number, slotOffset: number, v1: any, v2: any): any;

/**
 * Invokes a pipe with 3 arguments.
 *
 * This instruction acts as a guard to {@link PipeTransform#transform} invoking
 * the pipe only when an input to the pipe changes.
 *
 * @param index Pipe index where the pipe was stored on creation.
 * @param slotOffset the offset in the reserved slot space
 * @param v1 1st argument to {@link PipeTransform#transform}.
 * @param v2 2nd argument to {@link PipeTransform#transform}.
 * @param v3 4rd argument to {@link PipeTransform#transform}.
 *
 * @codeGenApi
 */
export declare function ɵɵpipeBind3(index: number, slotOffset: number, v1: any, v2: any, v3: any): any;

/**
 * Invokes a pipe with 4 arguments.
 *
 * This instruction acts as a guard to {@link PipeTransform#transform} invoking
 * the pipe only when an input to the pipe changes.
 *
 * @param index Pipe index where the pipe was stored on creation.
 * @param slotOffset the offset in the reserved slot space
 * @param v1 1st argument to {@link PipeTransform#transform}.
 * @param v2 2nd argument to {@link PipeTransform#transform}.
 * @param v3 3rd argument to {@link PipeTransform#transform}.
 * @param v4 4th argument to {@link PipeTransform#transform}.
 *
 * @codeGenApi
 */
export declare function ɵɵpipeBind4(index: number, slotOffset: number, v1: any, v2: any, v3: any, v4: any): any;

/**
 * Invokes a pipe with variable number of arguments.
 *
 * This instruction acts as a guard to {@link PipeTransform#transform} invoking
 * the pipe only when an input to the pipe changes.
 *
 * @param index Pipe index where the pipe was stored on creation.
 * @param slotOffset the offset in the reserved slot space
 * @param values Array of arguments to pass to {@link PipeTransform#transform} method.
 *
 * @codeGenApi
 */
export declare function ɵɵpipeBindV(index: number, slotOffset: number, values: [any, ...any[]]): any;

/**
 * @codeGenApi
 */
export declare type ɵɵPipeDefWithMeta<T, Name extends string> = ɵPipeDef<T>;

/**
 * Inserts previously re-distributed projected nodes. This instruction must be preceded by a call
 * to the projectionDef instruction.
 *
 * @param nodeIndex
 * @param selectorIndex:
 *        - 0 when the selector is `*` (or unspecified as this is the default value),
 *        - 1 based index of the selector from the {@link projectionDef}
 *
 * @codeGenApi
*/
export declare function ɵɵprojection(nodeIndex: number, selectorIndex?: number, attrs?: TAttributes): void;

/**
 * Instruction to distribute projectable nodes among <ng-content> occurrences in a given template.
 * It takes all the selectors from the entire component's template and decides where
 * each projected node belongs (it re-distributes nodes among "buckets" where each "bucket" is
 * backed by a selector).
 *
 * This function requires CSS selectors to be provided in 2 forms: parsed (by a compiler) and text,
 * un-parsed form.
 *
 * The parsed form is needed for efficient matching of a node against a given CSS selector.
 * The un-parsed, textual form is needed for support of the ngProjectAs attribute.
 *
 * Having a CSS selector in 2 different formats is not ideal, but alternatives have even more
 * drawbacks:
 * - having only a textual form would require runtime parsing of CSS selectors;
 * - we can't have only a parsed as we can't re-construct textual form from it (as entered by a
 * template author).
 *
 * @param projectionSlots? A collection of projection slots. A projection slot can be based
 *        on a parsed CSS selectors or set to the wildcard selector ("*") in order to match
 *        all nodes which do not match any selector. If not specified, a single wildcard
 *        selector projection slot will be defined.
 *
 * @codeGenApi
 */
export declare function ɵɵprojectionDef(projectionSlots?: ProjectionSlots): void;

/**
 * Update a property on a selected element.
 *
 * Operates on the element selected by index via the {@link select} instruction.
 *
 * If the property name also exists as an input property on one of the element's directives,
 * the component property will be set instead of the element property. This check must
 * be conducted at runtime so child components that add new `@Inputs` don't have to be re-compiled
 *
 * @param propName Name of property. Because it is going to DOM, this is not subject to
 *        renaming as part of minification.
 * @param value New value to write.
 * @param sanitizer An optional function used to sanitize the value.
 * @returns This function returns itself so that it may be chained
 * (e.g. `property('name', ctx.name)('title', ctx.title)`)
 *
 * @codeGenApi
 */
export declare function ɵɵproperty<T>(propName: string, value: T, sanitizer?: SanitizerFn | null): typeof ɵɵproperty;

/**
 *
 * Update an interpolated property on an element with a lone bound value
 *
 * Used when the value passed to a property has 1 interpolated value in it, an no additional text
 * surrounds that interpolated value:
 *
 * ```html
 * <div title="{{v0}}"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵpropertyInterpolate('title', v0);
 * ```
 *
 * If the property name also exists as an input property on one of the element's directives,
 * the component property will be set instead of the element property. This check must
 * be conducted at runtime so child components that add new `@Inputs` don't have to be re-compiled.
 *
 * @param propName The name of the property to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵpropertyInterpolate(propName: string, v0: any, sanitizer?: SanitizerFn): typeof ɵɵpropertyInterpolate;

/**
 *
 * Update an interpolated property on an element with single bound value surrounded by text.
 *
 * Used when the value passed to a property has 1 interpolated value in it:
 *
 * ```html
 * <div title="prefix{{v0}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵpropertyInterpolate1('title', 'prefix', v0, 'suffix');
 * ```
 *
 * If the property name also exists as an input property on one of the element's directives,
 * the component property will be set instead of the element property. This check must
 * be conducted at runtime so child components that add new `@Inputs` don't have to be re-compiled.
 *
 * @param propName The name of the property to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵpropertyInterpolate1(propName: string, prefix: string, v0: any, suffix: string, sanitizer?: SanitizerFn): typeof ɵɵpropertyInterpolate1;

/**
 *
 * Update an interpolated property on an element with 2 bound values surrounded by text.
 *
 * Used when the value passed to a property has 2 interpolated values in it:
 *
 * ```html
 * <div title="prefix{{v0}}-{{v1}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵpropertyInterpolate2('title', 'prefix', v0, '-', v1, 'suffix');
 * ```
 *
 * If the property name also exists as an input property on one of the element's directives,
 * the component property will be set instead of the element property. This check must
 * be conducted at runtime so child components that add new `@Inputs` don't have to be re-compiled.
 *
 * @param propName The name of the property to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵpropertyInterpolate2(propName: string, prefix: string, v0: any, i0: string, v1: any, suffix: string, sanitizer?: SanitizerFn): typeof ɵɵpropertyInterpolate2;

/**
 *
 * Update an interpolated property on an element with 3 bound values surrounded by text.
 *
 * Used when the value passed to a property has 3 interpolated values in it:
 *
 * ```html
 * <div title="prefix{{v0}}-{{v1}}-{{v2}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵpropertyInterpolate3(
 * 'title', 'prefix', v0, '-', v1, '-', v2, 'suffix');
 * ```
 *
 * If the property name also exists as an input property on one of the element's directives,
 * the component property will be set instead of the element property. This check must
 * be conducted at runtime so child components that add new `@Inputs` don't have to be re-compiled.
 *
 * @param propName The name of the property to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵpropertyInterpolate3(propName: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, suffix: string, sanitizer?: SanitizerFn): typeof ɵɵpropertyInterpolate3;

/**
 *
 * Update an interpolated property on an element with 4 bound values surrounded by text.
 *
 * Used when the value passed to a property has 4 interpolated values in it:
 *
 * ```html
 * <div title="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵpropertyInterpolate4(
 * 'title', 'prefix', v0, '-', v1, '-', v2, '-', v3, 'suffix');
 * ```
 *
 * If the property name also exists as an input property on one of the element's directives,
 * the component property will be set instead of the element property. This check must
 * be conducted at runtime so child components that add new `@Inputs` don't have to be re-compiled.
 *
 * @param propName The name of the property to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵpropertyInterpolate4(propName: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, suffix: string, sanitizer?: SanitizerFn): typeof ɵɵpropertyInterpolate4;

/**
 *
 * Update an interpolated property on an element with 5 bound values surrounded by text.
 *
 * Used when the value passed to a property has 5 interpolated values in it:
 *
 * ```html
 * <div title="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵpropertyInterpolate5(
 * 'title', 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, 'suffix');
 * ```
 *
 * If the property name also exists as an input property on one of the element's directives,
 * the component property will be set instead of the element property. This check must
 * be conducted at runtime so child components that add new `@Inputs` don't have to be re-compiled.
 *
 * @param propName The name of the property to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵpropertyInterpolate5(propName: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, suffix: string, sanitizer?: SanitizerFn): typeof ɵɵpropertyInterpolate5;

/**
 *
 * Update an interpolated property on an element with 6 bound values surrounded by text.
 *
 * Used when the value passed to a property has 6 interpolated values in it:
 *
 * ```html
 * <div title="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵpropertyInterpolate6(
 *    'title', 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, 'suffix');
 * ```
 *
 * If the property name also exists as an input property on one of the element's directives,
 * the component property will be set instead of the element property. This check must
 * be conducted at runtime so child components that add new `@Inputs` don't have to be re-compiled.
 *
 * @param propName The name of the property to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵpropertyInterpolate6(propName: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, suffix: string, sanitizer?: SanitizerFn): typeof ɵɵpropertyInterpolate6;

/**
 *
 * Update an interpolated property on an element with 7 bound values surrounded by text.
 *
 * Used when the value passed to a property has 7 interpolated values in it:
 *
 * ```html
 * <div title="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵpropertyInterpolate7(
 *    'title', 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, 'suffix');
 * ```
 *
 * If the property name also exists as an input property on one of the element's directives,
 * the component property will be set instead of the element property. This check must
 * be conducted at runtime so child components that add new `@Inputs` don't have to be re-compiled.
 *
 * @param propName The name of the property to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param i5 Static value used for concatenation only.
 * @param v6 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵpropertyInterpolate7(propName: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, i5: string, v6: any, suffix: string, sanitizer?: SanitizerFn): typeof ɵɵpropertyInterpolate7;

/**
 *
 * Update an interpolated property on an element with 8 bound values surrounded by text.
 *
 * Used when the value passed to a property has 8 interpolated values in it:
 *
 * ```html
 * <div title="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}-{{v7}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵpropertyInterpolate8(
 *  'title', 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, '-', v7, 'suffix');
 * ```
 *
 * If the property name also exists as an input property on one of the element's directives,
 * the component property will be set instead of the element property. This check must
 * be conducted at runtime so child components that add new `@Inputs` don't have to be re-compiled.
 *
 * @param propName The name of the property to update
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param i5 Static value used for concatenation only.
 * @param v6 Value checked for change.
 * @param i6 Static value used for concatenation only.
 * @param v7 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵpropertyInterpolate8(propName: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, i5: string, v6: any, i6: string, v7: any, suffix: string, sanitizer?: SanitizerFn): typeof ɵɵpropertyInterpolate8;

/**
 * Update an interpolated property on an element with 9 or more bound values surrounded by text.
 *
 * Used when the number of interpolated values exceeds 8.
 *
 * ```html
 * <div
 *  title="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}-{{v7}}-{{v8}}-{{v9}}suffix"></div>
 * ```
 *
 * Its compiled representation is::
 *
 * ```ts
 * ɵɵpropertyInterpolateV(
 *  'title', ['prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, '-', v7, '-', v9,
 *  'suffix']);
 * ```
 *
 * If the property name also exists as an input property on one of the element's directives,
 * the component property will be set instead of the element property. This check must
 * be conducted at runtime so child components that add new `@Inputs` don't have to be re-compiled.
 *
 * @param propName The name of the property to update.
 * @param values The collection of values and the strings inbetween those values, beginning with a
 * string prefix and ending with a string suffix.
 * (e.g. `['prefix', value0, '-', value1, '-', value2, ..., value99, 'suffix']`)
 * @param sanitizer An optional sanitizer function
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵpropertyInterpolateV(propName: string, values: any[], sanitizer?: SanitizerFn): typeof ɵɵpropertyInterpolateV;

/**
 * This feature resolves the providers of a directive (or component),
 * and publish them into the DI system, making it visible to others for injection.
 *
 * For example:
 * ```ts
 * class ComponentWithProviders {
 *   constructor(private greeter: GreeterDE) {}
 *
 *   static ɵcmp = defineComponent({
 *     type: ComponentWithProviders,
 *     selectors: [['component-with-providers']],
 *    factory: () => new ComponentWithProviders(directiveInject(GreeterDE as any)),
 *    decls: 1,
 *    vars: 1,
 *    template: function(fs: RenderFlags, ctx: ComponentWithProviders) {
 *      if (fs & RenderFlags.Create) {
 *        ɵɵtext(0);
 *      }
 *      if (fs & RenderFlags.Update) {
 *        ɵɵtextInterpolate(ctx.greeter.greet());
 *      }
 *    },
 *    features: [ProvidersFeature([GreeterDE])]
 *  });
 * }
 * ```
 *
 * @param definition
 *
 * @codeGenApi
 */
export declare function ɵɵProvidersFeature<T>(providers: Provider[], viewProviders?: Provider[]): (definition: ɵDirectiveDef<T>) => void;

/**
 * Bindings for pure functions are stored after regular bindings.
 *
 * |-------decls------|---------vars---------|                 |----- hostVars (dir1) ------|
 * ------------------------------------------------------------------------------------------
 * | nodes/refs/pipes | bindings | fn slots  | injector | dir1 | host bindings | host slots |
 * ------------------------------------------------------------------------------------------
 *                    ^                      ^
 *      TView.bindingStartIndex      TView.expandoStartIndex
 *
 * Pure function instructions are given an offset from the binding root. Adding the offset to the
 * binding root gives the first index where the bindings are stored. In component views, the binding
 * root is the bindingStartIndex. In host bindings, the binding root is the expandoStartIndex +
 * any directive instances + any hostVars in directives evaluated before it.
 *
 * See VIEW_DATA.md for more information about host binding resolution.
 */
/**
 * If the value hasn't been saved, calls the pure function to store and return the
 * value. If it has been saved, returns the saved value.
 *
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn Function that returns a value
 * @param thisArg Optional calling context of pureFn
 * @returns value
 *
 * @codeGenApi
 */
export declare function ɵɵpureFunction0<T>(slotOffset: number, pureFn: () => T, thisArg?: any): T;

/**
 * If the value of the provided exp has changed, calls the pure function to return
 * an updated value. Or if the value has not changed, returns cached value.
 *
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn Function that returns an updated value
 * @param exp Updated expression value
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 *
 * @codeGenApi
 */
export declare function ɵɵpureFunction1(slotOffset: number, pureFn: (v: any) => any, exp: any, thisArg?: any): any;

/**
 * If the value of any provided exp has changed, calls the pure function to return
 * an updated value. Or if no values have changed, returns cached value.
 *
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn
 * @param exp1
 * @param exp2
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 *
 * @codeGenApi
 */
export declare function ɵɵpureFunction2(slotOffset: number, pureFn: (v1: any, v2: any) => any, exp1: any, exp2: any, thisArg?: any): any;

/**
 * If the value of any provided exp has changed, calls the pure function to return
 * an updated value. Or if no values have changed, returns cached value.
 *
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn
 * @param exp1
 * @param exp2
 * @param exp3
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 *
 * @codeGenApi
 */
export declare function ɵɵpureFunction3(slotOffset: number, pureFn: (v1: any, v2: any, v3: any) => any, exp1: any, exp2: any, exp3: any, thisArg?: any): any;

/**
 * If the value of any provided exp has changed, calls the pure function to return
 * an updated value. Or if no values have changed, returns cached value.
 *
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn
 * @param exp1
 * @param exp2
 * @param exp3
 * @param exp4
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 *
 * @codeGenApi
 */
export declare function ɵɵpureFunction4(slotOffset: number, pureFn: (v1: any, v2: any, v3: any, v4: any) => any, exp1: any, exp2: any, exp3: any, exp4: any, thisArg?: any): any;

/**
 * If the value of any provided exp has changed, calls the pure function to return
 * an updated value. Or if no values have changed, returns cached value.
 *
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn
 * @param exp1
 * @param exp2
 * @param exp3
 * @param exp4
 * @param exp5
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 *
 * @codeGenApi
 */
export declare function ɵɵpureFunction5(slotOffset: number, pureFn: (v1: any, v2: any, v3: any, v4: any, v5: any) => any, exp1: any, exp2: any, exp3: any, exp4: any, exp5: any, thisArg?: any): any;

/**
 * If the value of any provided exp has changed, calls the pure function to return
 * an updated value. Or if no values have changed, returns cached value.
 *
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn
 * @param exp1
 * @param exp2
 * @param exp3
 * @param exp4
 * @param exp5
 * @param exp6
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 *
 * @codeGenApi
 */
export declare function ɵɵpureFunction6(slotOffset: number, pureFn: (v1: any, v2: any, v3: any, v4: any, v5: any, v6: any) => any, exp1: any, exp2: any, exp3: any, exp4: any, exp5: any, exp6: any, thisArg?: any): any;

/**
 * If the value of any provided exp has changed, calls the pure function to return
 * an updated value. Or if no values have changed, returns cached value.
 *
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn
 * @param exp1
 * @param exp2
 * @param exp3
 * @param exp4
 * @param exp5
 * @param exp6
 * @param exp7
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 *
 * @codeGenApi
 */
export declare function ɵɵpureFunction7(slotOffset: number, pureFn: (v1: any, v2: any, v3: any, v4: any, v5: any, v6: any, v7: any) => any, exp1: any, exp2: any, exp3: any, exp4: any, exp5: any, exp6: any, exp7: any, thisArg?: any): any;

/**
 * If the value of any provided exp has changed, calls the pure function to return
 * an updated value. Or if no values have changed, returns cached value.
 *
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn
 * @param exp1
 * @param exp2
 * @param exp3
 * @param exp4
 * @param exp5
 * @param exp6
 * @param exp7
 * @param exp8
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 *
 * @codeGenApi
 */
export declare function ɵɵpureFunction8(slotOffset: number, pureFn: (v1: any, v2: any, v3: any, v4: any, v5: any, v6: any, v7: any, v8: any) => any, exp1: any, exp2: any, exp3: any, exp4: any, exp5: any, exp6: any, exp7: any, exp8: any, thisArg?: any): any;

/**
 * pureFunction instruction that can support any number of bindings.
 *
 * If the value of any provided exp has changed, calls the pure function to return
 * an updated value. Or if no values have changed, returns cached value.
 *
 * @param slotOffset the offset from binding root to the reserved slot
 * @param pureFn A pure function that takes binding values and builds an object or array
 * containing those values.
 * @param exps An array of binding values
 * @param thisArg Optional calling context of pureFn
 * @returns Updated or cached value
 *
 * @codeGenApi
 */
export declare function ɵɵpureFunctionV(slotOffset: number, pureFn: (...v: any[]) => any, exps: any[], thisArg?: any): any;

/**
 * Refreshes a query by combining matches from all active views and removing matches from deleted
 * views.
 *
 * @returns `true` if a query got dirty during change detection or if this is a static query
 * resolving in creation mode, `false` otherwise.
 *
 * @codeGenApi
 */
export declare function ɵɵqueryRefresh(queryList: QueryList<any>): boolean;

/**
 * Retrieves a local reference from the current contextViewData.
 *
 * If the reference to retrieve is in a parent view, this instruction is used in conjunction
 * with a nextContext() call, which walks up the tree and updates the contextViewData instance.
 *
 * @param index The index of the local ref in contextViewData.
 *
 * @codeGenApi
 */
export declare function ɵɵreference<T>(index: number): T;

/**
 *
 * @codeGenApi
 */
export declare function ɵɵresolveBody(element: RElement & {
    ownerDocument: Document;
}): {
    name: string;
    target: HTMLElement;
};

/**
 *
 * @codeGenApi
 */
export declare function ɵɵresolveDocument(element: RElement & {
    ownerDocument: Document;
}): {
    name: string;
    target: Document;
};

/**
 *
 * @codeGenApi
 */
export declare function ɵɵresolveWindow(element: RElement & {
    ownerDocument: Document;
}): {
    name: string;
    target: Window | null;
};

/**
 * Restores `contextViewData` to the given OpaqueViewState instance.
 *
 * Used in conjunction with the getCurrentView() instruction to save a snapshot
 * of the current view and restore it when listeners are invoked. This allows
 * walking the declaration view tree in listeners to get vars from parent views.
 *
 * @param viewToRestore The OpaqueViewState instance to restore.
 *
 * @codeGenApi
 */
export declare function ɵɵrestoreView(viewToRestore: OpaqueViewState): void;

/**
 * An `html` sanitizer which converts untrusted `html` **string** into trusted string by removing
 * dangerous content.
 *
 * This method parses the `html` and locates potentially dangerous content (such as urls and
 * javascript) and removes it.
 *
 * It is possible to mark a string as trusted by calling {@link bypassSanitizationTrustHtml}.
 *
 * @param unsafeHtml untrusted `html`, typically from the user.
 * @returns `html` string which is safe to display to user, because all of the dangerous javascript
 * and urls have been removed.
 *
 * @publicApi
 */
export declare function ɵɵsanitizeHtml(unsafeHtml: any): string;

/**
 * A `url` sanitizer which only lets trusted `url`s through.
 *
 * This passes only `url`s marked trusted by calling {@link bypassSanitizationTrustResourceUrl}.
 *
 * @param unsafeResourceUrl untrusted `url`, typically from the user.
 * @returns `url` string which is safe to bind to the `src` properties such as `<img src>`, because
 * only trusted `url`s have been allowed to pass.
 *
 * @publicApi
 */
export declare function ɵɵsanitizeResourceUrl(unsafeResourceUrl: any): string;

/**
 * A `script` sanitizer which only lets trusted javascript through.
 *
 * This passes only `script`s marked trusted by calling {@link
 * bypassSanitizationTrustScript}.
 *
 * @param unsafeScript untrusted `script`, typically from the user.
 * @returns `url` string which is safe to bind to the `<script>` element such as `<img src>`,
 * because only trusted `scripts` have been allowed to pass.
 *
 * @publicApi
 */
export declare function ɵɵsanitizeScript(unsafeScript: any): string;

/**
 * A `style` sanitizer which converts untrusted `style` **string** into trusted string by removing
 * dangerous content.
 *
 * This method parses the `style` and locates potentially dangerous content (such as urls and
 * javascript) and removes it.
 *
 * It is possible to mark a string as trusted by calling {@link bypassSanitizationTrustStyle}.
 *
 * @param unsafeStyle untrusted `style`, typically from the user.
 * @returns `style` string which is safe to bind to the `style` properties, because all of the
 * dangerous javascript and urls have been removed.
 *
 * @publicApi
 */
export declare function ɵɵsanitizeStyle(unsafeStyle: any): string;

/**
 * A `url` sanitizer which converts untrusted `url` **string** into trusted string by removing
 * dangerous
 * content.
 *
 * This method parses the `url` and locates potentially dangerous content (such as javascript) and
 * removes it.
 *
 * It is possible to mark a string as trusted by calling {@link bypassSanitizationTrustUrl}.
 *
 * @param unsafeUrl untrusted `url`, typically from the user.
 * @returns `url` string which is safe to bind to the `src` properties such as `<img src>`, because
 * all of the dangerous javascript has been removed.
 *
 * @publicApi
 */
export declare function ɵɵsanitizeUrl(unsafeUrl: any): string;

/**
 * Sanitizes URL, selecting sanitizer function based on tag and property names.
 *
 * This function is used in case we can't define security context at compile time, when only prop
 * name is available. This happens when we generate host bindings for Directives/Components. The
 * host element is unknown at compile time, so we defer calculation of specific sanitizer to
 * runtime.
 *
 * @param unsafeUrl untrusted `url`, typically from the user.
 * @param tag target element tag name.
 * @param prop name of the property that contains the value.
 * @returns `url` string which is safe to bind.
 *
 * @publicApi
 */
export declare function ɵɵsanitizeUrlOrResourceUrl(unsafeUrl: any, tag: string, prop: string): any;

/**
 * Selects an element for later binding instructions.
 * @deprecated No longer being generated, but still used in unit tests.
 * @codeGenApi
 */
export declare function ɵɵselect(index: number): void;

/**
 * @codeGenApi
 */
export declare function ɵɵsetComponentScope(type: ɵComponentType<any>, directives: Type<any>[], pipes: Type<any>[]): void;

/**
 * Adds the module metadata that is necessary to compute the module's transitive scope to an
 * existing module definition.
 *
 * Scope metadata of modules is not used in production builds, so calls to this function can be
 * marked pure to tree-shake it from the bundle, allowing for all referenced declarations
 * to become eligible for tree-shaking as well.
 *
 * @codeGenApi
 */
export declare function ɵɵsetNgModuleScope(type: any, scope: {
    /** List of components, directives, and pipes declared by this module. */
    declarations?: Type<any>[] | (() => Type<any>[]);
    /** List of modules or `ModuleWithProviders` imported by this module. */
    imports?: Type<any>[] | (() => Type<any>[]);
    /**
     * List of modules, `ModuleWithProviders`, components, directives, or pipes exported by this
     * module.
     */
    exports?: Type<any>[] | (() => Type<any>[]);
}): void;

/**
 * Registers a QueryList, associated with a static content query, for later refresh
 * (part of a view refresh).
 *
 * @param directiveIndex Current directive index
 * @param predicate The type for which the query will search
 * @param descend Whether or not to descend into children
 * @param read What to save in the query
 * @returns QueryList<T>
 *
 * @codeGenApi
 */
export declare function ɵɵstaticContentQuery<T>(directiveIndex: number, predicate: Type<any> | string[], descend: boolean, read?: any): void;

/**
 * Creates new QueryList for a static view query.
 *
 * @param predicate The type for which the query will search
 * @param descend Whether or not to descend into children
 * @param read What to save in the query
 *
 * @codeGenApi
 */
export declare function ɵɵstaticViewQuery<T>(predicate: Type<any> | string[], descend: boolean, read?: any): void;

/**
 * Update style bindings using an object literal on an element.
 *
 * This instruction is meant to apply styling via the `[style]="exp"` template bindings.
 * When styles are applied to the element they will then be updated with respect to
 * any styles/classes set via `styleProp`. If any styles are set to falsy
 * then they will be removed from the element.
 *
 * Note that the styling instruction will not be applied until `stylingApply` is called.
 *
 * @param styles A key/value style map of the styles that will be applied to the given element.
 *        Any missing styles (that have already been applied to the element beforehand) will be
 *        removed (unset) from the element's styling.
 *
 * Note that this will apply the provided styleMap value to the host element if this function
 * is called within a host binding.
 *
 * @codeGenApi
 */
export declare function ɵɵstyleMap(styles: {
    [styleName: string]: any;
} | string | undefined | null): void;


/**
 *
 * Update an interpolated style on an element with single bound value surrounded by text.
 *
 * Used when the value passed to a property has 1 interpolated value in it:
 *
 * ```html
 * <div style="key: {{v0}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstyleMapInterpolate1('key: ', v0, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵstyleMapInterpolate1(prefix: string, v0: any, suffix: string): void;

/**
 *
 * Update an interpolated style on an element with 2 bound values surrounded by text.
 *
 * Used when the value passed to a property has 2 interpolated values in it:
 *
 * ```html
 * <div style="key: {{v0}}; key1: {{v1}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstyleMapInterpolate2('key: ', v0, '; key1: ', v1, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵstyleMapInterpolate2(prefix: string, v0: any, i0: string, v1: any, suffix: string): void;

/**
 *
 * Update an interpolated style on an element with 3 bound values surrounded by text.
 *
 * Used when the value passed to a property has 3 interpolated values in it:
 *
 * ```html
 * <div style="key: {{v0}}; key2: {{v1}}; key2: {{v2}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstyleMapInterpolate3(
 *     'key: ', v0, '; key1: ', v1, '; key2: ', v2, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵstyleMapInterpolate3(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, suffix: string): void;

/**
 *
 * Update an interpolated style on an element with 4 bound values surrounded by text.
 *
 * Used when the value passed to a property has 4 interpolated values in it:
 *
 * ```html
 * <div style="key: {{v0}}; key1: {{v1}}; key2: {{v2}}; key3: {{v3}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstyleMapInterpolate4(
 *     'key: ', v0, '; key1: ', v1, '; key2: ', v2, '; key3: ', v3, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵstyleMapInterpolate4(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, suffix: string): void;

/**
 *
 * Update an interpolated style on an element with 5 bound values surrounded by text.
 *
 * Used when the value passed to a property has 5 interpolated values in it:
 *
 * ```html
 * <div style="key: {{v0}}; key1: {{v1}}; key2: {{v2}}; key3: {{v3}}; key4: {{v4}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstyleMapInterpolate5(
 *     'key: ', v0, '; key1: ', v1, '; key2: ', v2, '; key3: ', v3, '; key4: ', v4, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵstyleMapInterpolate5(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, suffix: string): void;

/**
 *
 * Update an interpolated style on an element with 6 bound values surrounded by text.
 *
 * Used when the value passed to a property has 6 interpolated values in it:
 *
 * ```html
 * <div style="key: {{v0}}; key1: {{v1}}; key2: {{v2}}; key3: {{v3}}; key4: {{v4}};
 *             key5: {{v5}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstyleMapInterpolate6(
 *    'key: ', v0, '; key1: ', v1, '; key2: ', v2, '; key3: ', v3, '; key4: ', v4, '; key5: ', v5,
 *    'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵstyleMapInterpolate6(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, suffix: string): void;

/**
 *
 * Update an interpolated style on an element with 7 bound values surrounded by text.
 *
 * Used when the value passed to a property has 7 interpolated values in it:
 *
 * ```html
 * <div style="key: {{v0}}; key1: {{v1}}; key2: {{v2}}; key3: {{v3}}; key4: {{v4}}; key5: {{v5}};
 *             key6: {{v6}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstyleMapInterpolate7(
 *    'key: ', v0, '; key1: ', v1, '; key2: ', v2, '; key3: ', v3, '; key4: ', v4, '; key5: ', v5,
 *    '; key6: ', v6, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param i5 Static value used for concatenation only.
 * @param v6 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵstyleMapInterpolate7(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, i5: string, v6: any, suffix: string): void;

/**
 *
 * Update an interpolated style on an element with 8 bound values surrounded by text.
 *
 * Used when the value passed to a property has 8 interpolated values in it:
 *
 * ```html
 * <div style="key: {{v0}}; key1: {{v1}}; key2: {{v2}}; key3: {{v3}}; key4: {{v4}}; key5: {{v5}};
 *             key6: {{v6}}; key7: {{v7}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstyleMapInterpolate8(
 *    'key: ', v0, '; key1: ', v1, '; key2: ', v2, '; key3: ', v3, '; key4: ', v4, '; key5: ', v5,
 *    '; key6: ', v6, '; key7: ', v7, 'suffix');
 * ```
 *
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param i5 Static value used for concatenation only.
 * @param v6 Value checked for change.
 * @param i6 Static value used for concatenation only.
 * @param v7 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @codeGenApi
 */
export declare function ɵɵstyleMapInterpolate8(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, i5: string, v6: any, i6: string, v7: any, suffix: string): void;

/**
 * Update an interpolated style on an element with 9 or more bound values surrounded by text.
 *
 * Used when the number of interpolated values exceeds 8.
 *
 * ```html
 * <div
 *  class="key: {{v0}}; key1: {{v1}}; key2: {{v2}}; key3: {{v3}}; key4: {{v4}}; key5: {{v5}};
 *         key6: {{v6}}; key7: {{v7}}; key8: {{v8}}; key9: {{v9}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstyleMapInterpolateV(
 *    ['key: ', v0, '; key1: ', v1, '; key2: ', v2, '; key3: ', v3, '; key4: ', v4, '; key5: ', v5,
 *     '; key6: ', v6, '; key7: ', v7, '; key8: ', v8, '; key9: ', v9, 'suffix']);
 * ```
 *.
 * @param values The collection of values and the strings in-between those values, beginning with
 * a string prefix and ending with a string suffix.
 * (e.g. `['prefix', value0, '; key2: ', value1, '; key2: ', value2, ..., value99, 'suffix']`)
 * @codeGenApi
 */
export declare function ɵɵstyleMapInterpolateV(values: any[]): void;

/**
 * Update a style binding on an element with the provided value.
 *
 * If the style value is falsy then it will be removed from the element
 * (or assigned a different value depending if there are any styles placed
 * on the element with `styleMap` or any static styles that are
 * present from when the element was created with `styling`).
 *
 * Note that the styling element is updated as part of `stylingApply`.
 *
 * @param prop A valid CSS property.
 * @param value New value to write (`null` or an empty string to remove).
 * @param suffix Optional suffix. Used with scalar values to add unit such as `px`.
 *        Note that when a suffix is provided then the underlying sanitizer will
 *        be ignored.
 *
 * Note that this will apply the provided style value to the host element if this function is called
 * within a host binding function.
 *
 * @codeGenApi
 */
export declare function ɵɵstyleProp(prop: string, value: string | number | ɵSafeValue | undefined | null, suffix?: string | null): typeof ɵɵstyleProp;


/**
 *
 * Update an interpolated style property on an element with single bound value surrounded by text.
 *
 * Used when the value passed to a property has 1 interpolated value in it:
 *
 * ```html
 * <div style.color="prefix{{v0}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstylePropInterpolate1(0, 'prefix', v0, 'suffix');
 * ```
 *
 * @param styleIndex Index of style to update. This index value refers to the
 *        index of the style in the style bindings array that was passed into
 *        `styling`.
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param valueSuffix Optional suffix. Used with scalar values to add unit such as `px`.
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵstylePropInterpolate1(prop: string, prefix: string, v0: any, suffix: string, valueSuffix?: string | null): typeof ɵɵstylePropInterpolate1;

/**
 *
 * Update an interpolated style property on an element with 2 bound values surrounded by text.
 *
 * Used when the value passed to a property has 2 interpolated values in it:
 *
 * ```html
 * <div style.color="prefix{{v0}}-{{v1}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstylePropInterpolate2(0, 'prefix', v0, '-', v1, 'suffix');
 * ```
 *
 * @param styleIndex Index of style to update. This index value refers to the
 *        index of the style in the style bindings array that was passed into
 *        `styling`.
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param valueSuffix Optional suffix. Used with scalar values to add unit such as `px`.
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵstylePropInterpolate2(prop: string, prefix: string, v0: any, i0: string, v1: any, suffix: string, valueSuffix?: string | null): typeof ɵɵstylePropInterpolate2;

/**
 *
 * Update an interpolated style property on an element with 3 bound values surrounded by text.
 *
 * Used when the value passed to a property has 3 interpolated values in it:
 *
 * ```html
 * <div style.color="prefix{{v0}}-{{v1}}-{{v2}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstylePropInterpolate3(0, 'prefix', v0, '-', v1, '-', v2, 'suffix');
 * ```
 *
 * @param styleIndex Index of style to update. This index value refers to the
 *        index of the style in the style bindings array that was passed into
 *        `styling`.
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param valueSuffix Optional suffix. Used with scalar values to add unit such as `px`.
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵstylePropInterpolate3(prop: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, suffix: string, valueSuffix?: string | null): typeof ɵɵstylePropInterpolate3;

/**
 *
 * Update an interpolated style property on an element with 4 bound values surrounded by text.
 *
 * Used when the value passed to a property has 4 interpolated values in it:
 *
 * ```html
 * <div style.color="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstylePropInterpolate4(0, 'prefix', v0, '-', v1, '-', v2, '-', v3, 'suffix');
 * ```
 *
 * @param styleIndex Index of style to update. This index value refers to the
 *        index of the style in the style bindings array that was passed into
 *        `styling`.
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param valueSuffix Optional suffix. Used with scalar values to add unit such as `px`.
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵstylePropInterpolate4(prop: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, suffix: string, valueSuffix?: string | null): typeof ɵɵstylePropInterpolate4;

/**
 *
 * Update an interpolated style property on an element with 5 bound values surrounded by text.
 *
 * Used when the value passed to a property has 5 interpolated values in it:
 *
 * ```html
 * <div style.color="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstylePropInterpolate5(0, 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, 'suffix');
 * ```
 *
 * @param styleIndex Index of style to update. This index value refers to the
 *        index of the style in the style bindings array that was passed into
 *        `styling`.
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param valueSuffix Optional suffix. Used with scalar values to add unit such as `px`.
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵstylePropInterpolate5(prop: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, suffix: string, valueSuffix?: string | null): typeof ɵɵstylePropInterpolate5;

/**
 *
 * Update an interpolated style property on an element with 6 bound values surrounded by text.
 *
 * Used when the value passed to a property has 6 interpolated values in it:
 *
 * ```html
 * <div style.color="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstylePropInterpolate6(0, 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, 'suffix');
 * ```
 *
 * @param styleIndex Index of style to update. This index value refers to the
 *        index of the style in the style bindings array that was passed into
 *        `styling`.
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param valueSuffix Optional suffix. Used with scalar values to add unit such as `px`.
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵstylePropInterpolate6(prop: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, suffix: string, valueSuffix?: string | null): typeof ɵɵstylePropInterpolate6;

/**
 *
 * Update an interpolated style property on an element with 7 bound values surrounded by text.
 *
 * Used when the value passed to a property has 7 interpolated values in it:
 *
 * ```html
 * <div style.color="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstylePropInterpolate7(
 *    0, 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, 'suffix');
 * ```
 *
 * @param styleIndex Index of style to update. This index value refers to the
 *        index of the style in the style bindings array that was passed into
 *        `styling`.
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param i5 Static value used for concatenation only.
 * @param v6 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param valueSuffix Optional suffix. Used with scalar values to add unit such as `px`.
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵstylePropInterpolate7(prop: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, i5: string, v6: any, suffix: string, valueSuffix?: string | null): typeof ɵɵstylePropInterpolate7;

/**
 *
 * Update an interpolated style property on an element with 8 bound values surrounded by text.
 *
 * Used when the value passed to a property has 8 interpolated values in it:
 *
 * ```html
 * <div style.color="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}-{{v7}}suffix"></div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstylePropInterpolate8(0, 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6,
 * '-', v7, 'suffix');
 * ```
 *
 * @param styleIndex Index of style to update. This index value refers to the
 *        index of the style in the style bindings array that was passed into
 *        `styling`.
 * @param prefix Static value used for concatenation only.
 * @param v0 Value checked for change.
 * @param i0 Static value used for concatenation only.
 * @param v1 Value checked for change.
 * @param i1 Static value used for concatenation only.
 * @param v2 Value checked for change.
 * @param i2 Static value used for concatenation only.
 * @param v3 Value checked for change.
 * @param i3 Static value used for concatenation only.
 * @param v4 Value checked for change.
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change.
 * @param i5 Static value used for concatenation only.
 * @param v6 Value checked for change.
 * @param i6 Static value used for concatenation only.
 * @param v7 Value checked for change.
 * @param suffix Static value used for concatenation only.
 * @param valueSuffix Optional suffix. Used with scalar values to add unit such as `px`.
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵstylePropInterpolate8(prop: string, prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, i5: string, v6: any, i6: string, v7: any, suffix: string, valueSuffix?: string | null): typeof ɵɵstylePropInterpolate8;

/**
 * Update an interpolated style property on an element with 9 or more bound values surrounded by
 * text.
 *
 * Used when the number of interpolated values exceeds 8.
 *
 * ```html
 * <div
 *  style.color="prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}-{{v7}}-{{v8}}-{{v9}}suffix">
 * </div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵstylePropInterpolateV(
 *  0, ['prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, '-', v7, '-', v9,
 *  'suffix']);
 * ```
 *
 * @param styleIndex Index of style to update. This index value refers to the
 *        index of the style in the style bindings array that was passed into
 *        `styling`..
 * @param values The collection of values and the strings in-between those values, beginning with
 * a string prefix and ending with a string suffix.
 * (e.g. `['prefix', value0, '-', value1, '-', value2, ..., value99, 'suffix']`)
 * @param valueSuffix Optional suffix. Used with scalar values to add unit such as `px`.
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵstylePropInterpolateV(prop: string, values: any[], valueSuffix?: string | null): typeof ɵɵstylePropInterpolateV;

/**
 * Sets the current style sanitizer function which will then be used
 * within all follow-up prop and map-based style binding instructions
 * for the given element.
 *
 * Note that once styling has been applied to the element (i.e. once
 * `advance(n)` is executed or the hostBindings/template function exits)
 * then the active `sanitizerFn` will be set to `null`. This means that
 * once styling is applied to another element then a another call to
 * `styleSanitizer` will need to be made.
 *
 * @param sanitizerFn The sanitization function that will be used to
 *       process style prop/value entries.
 *
 * @codeGenApi
 */
export declare function ɵɵstyleSanitizer(sanitizer: StyleSanitizeFn | null): void;

/**
 * Creates an LContainer for an ng-template (dynamically-inserted view), e.g.
 *
 * <ng-template #foo>
 *    <div></div>
 * </ng-template>
 *
 * @param index The index of the container in the data array
 * @param templateFn Inline template
 * @param decls The number of nodes, local refs, and pipes for this template
 * @param vars The number of bindings for this template
 * @param tagName The name of the container element, if applicable
 * @param attrsIndex Index of template attributes in the `consts` array.
 * @param localRefs Index of the local references in the `consts` array.
 * @param localRefExtractor A function which extracts local-refs values from the template.
 *        Defaults to the current element associated with the local-ref.
 *
 * @codeGenApi
 */
export declare function ɵɵtemplate(index: number, templateFn: ComponentTemplate<any> | null, decls: number, vars: number, tagName?: string | null, attrsIndex?: number | null, localRefsIndex?: number | null, localRefExtractor?: LocalRefExtractor): void;

/**
 * Retrieves `TemplateRef` instance from `Injector` when a local reference is placed on the
 * `<ng-template>` element.
 *
 * @codeGenApi
 */
export declare function ɵɵtemplateRefExtractor(tNode: TNode, currentView: ɵangular_packages_core_core_bp): TemplateRef<unknown> | null;

/**
 * Create static text node
 *
 * @param index Index of the node in the data array
 * @param value Static string value to write.
 *
 * @codeGenApi
 */
export declare function ɵɵtext(index: number, value?: string): void;

/**
 *
 * Update text content with a lone bound value
 *
 * Used when a text node has 1 interpolated value in it, an no additional text
 * surrounds that interpolated value:
 *
 * ```html
 * <div>{{v0}}</div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵtextInterpolate(v0);
 * ```
 * @returns itself, so that it may be chained.
 * @see textInterpolateV
 * @codeGenApi
 */
export declare function ɵɵtextInterpolate(v0: any): typeof ɵɵtextInterpolate;

/**
 *
 * Update text content with single bound value surrounded by other text.
 *
 * Used when a text node has 1 interpolated value in it:
 *
 * ```html
 * <div>prefix{{v0}}suffix</div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵtextInterpolate1('prefix', v0, 'suffix');
 * ```
 * @returns itself, so that it may be chained.
 * @see textInterpolateV
 * @codeGenApi
 */
export declare function ɵɵtextInterpolate1(prefix: string, v0: any, suffix: string): typeof ɵɵtextInterpolate1;

/**
 *
 * Update text content with 2 bound values surrounded by other text.
 *
 * Used when a text node has 2 interpolated values in it:
 *
 * ```html
 * <div>prefix{{v0}}-{{v1}}suffix</div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵtextInterpolate2('prefix', v0, '-', v1, 'suffix');
 * ```
 * @returns itself, so that it may be chained.
 * @see textInterpolateV
 * @codeGenApi
 */
export declare function ɵɵtextInterpolate2(prefix: string, v0: any, i0: string, v1: any, suffix: string): typeof ɵɵtextInterpolate2;

/**
 *
 * Update text content with 3 bound values surrounded by other text.
 *
 * Used when a text node has 3 interpolated values in it:
 *
 * ```html
 * <div>prefix{{v0}}-{{v1}}-{{v2}}suffix</div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵtextInterpolate3(
 * 'prefix', v0, '-', v1, '-', v2, 'suffix');
 * ```
 * @returns itself, so that it may be chained.
 * @see textInterpolateV
 * @codeGenApi
 */
export declare function ɵɵtextInterpolate3(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, suffix: string): typeof ɵɵtextInterpolate3;

/**
 *
 * Update text content with 4 bound values surrounded by other text.
 *
 * Used when a text node has 4 interpolated values in it:
 *
 * ```html
 * <div>prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}suffix</div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵtextInterpolate4(
 * 'prefix', v0, '-', v1, '-', v2, '-', v3, 'suffix');
 * ```
 * @returns itself, so that it may be chained.
 * @see ɵɵtextInterpolateV
 * @codeGenApi
 */
export declare function ɵɵtextInterpolate4(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, suffix: string): typeof ɵɵtextInterpolate4;

/**
 *
 * Update text content with 5 bound values surrounded by other text.
 *
 * Used when a text node has 5 interpolated values in it:
 *
 * ```html
 * <div>prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}suffix</div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵtextInterpolate5(
 * 'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, 'suffix');
 * ```
 * @returns itself, so that it may be chained.
 * @see textInterpolateV
 * @codeGenApi
 */
export declare function ɵɵtextInterpolate5(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, suffix: string): typeof ɵɵtextInterpolate5;

/**
 *
 * Update text content with 6 bound values surrounded by other text.
 *
 * Used when a text node has 6 interpolated values in it:
 *
 * ```html
 * <div>prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}suffix</div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵtextInterpolate6(
 *    'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, 'suffix');
 * ```
 *
 * @param i4 Static value used for concatenation only.
 * @param v5 Value checked for change. @returns itself, so that it may be chained.
 * @see textInterpolateV
 * @codeGenApi
 */
export declare function ɵɵtextInterpolate6(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, suffix: string): typeof ɵɵtextInterpolate6;

/**
 *
 * Update text content with 7 bound values surrounded by other text.
 *
 * Used when a text node has 7 interpolated values in it:
 *
 * ```html
 * <div>prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}suffix</div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵtextInterpolate7(
 *    'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, 'suffix');
 * ```
 * @returns itself, so that it may be chained.
 * @see textInterpolateV
 * @codeGenApi
 */
export declare function ɵɵtextInterpolate7(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, i5: string, v6: any, suffix: string): typeof ɵɵtextInterpolate7;

/**
 *
 * Update text content with 8 bound values surrounded by other text.
 *
 * Used when a text node has 8 interpolated values in it:
 *
 * ```html
 * <div>prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}-{{v7}}suffix</div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵtextInterpolate8(
 *  'prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, '-', v7, 'suffix');
 * ```
 * @returns itself, so that it may be chained.
 * @see textInterpolateV
 * @codeGenApi
 */
export declare function ɵɵtextInterpolate8(prefix: string, v0: any, i0: string, v1: any, i1: string, v2: any, i2: string, v3: any, i3: string, v4: any, i4: string, v5: any, i5: string, v6: any, i6: string, v7: any, suffix: string): typeof ɵɵtextInterpolate8;

/**
 * Update text content with 9 or more bound values other surrounded by text.
 *
 * Used when the number of interpolated values exceeds 8.
 *
 * ```html
 * <div>prefix{{v0}}-{{v1}}-{{v2}}-{{v3}}-{{v4}}-{{v5}}-{{v6}}-{{v7}}-{{v8}}-{{v9}}suffix</div>
 * ```
 *
 * Its compiled representation is:
 *
 * ```ts
 * ɵɵtextInterpolateV(
 *  ['prefix', v0, '-', v1, '-', v2, '-', v3, '-', v4, '-', v5, '-', v6, '-', v7, '-', v9,
 *  'suffix']);
 * ```
 *.
 * @param values The collection of values and the strings in between those values, beginning with
 * a string prefix and ending with a string suffix.
 * (e.g. `['prefix', value0, '-', value1, '-', value2, ..., value99, 'suffix']`)
 *
 * @returns itself, so that it may be chained.
 * @codeGenApi
 */
export declare function ɵɵtextInterpolateV(values: any[]): typeof ɵɵtextInterpolateV;

/**
 * Updates a synthetic host binding (e.g. `[@foo]`) on a component.
 *
 * This instruction is for compatibility purposes and is designed to ensure that a
 * synthetic host binding (e.g. `@HostBinding('@foo')`) properly gets rendered in
 * the component's renderer. Normally all host bindings are evaluated with the parent
 * component's renderer, but, in the case of animation @triggers, they need to be
 * evaluated with the sub component's renderer (because that's where the animation
 * triggers are defined).
 *
 * Do not use this instruction as a replacement for `elementProperty`. This instruction
 * only exists to ensure compatibility with the ViewEngine's host binding behavior.
 *
 * @param index The index of the element to update in the data array
 * @param propName Name of property. Because it is going to DOM, this is not subject to
 *        renaming as part of minification.
 * @param value New value to write.
 * @param sanitizer An optional function used to sanitize the value.
 *
 * @codeGenApi
 */
export declare function ɵɵupdateSyntheticHostBinding<T>(propName: string, value: T | ɵNO_CHANGE, sanitizer?: SanitizerFn | null): typeof ɵɵupdateSyntheticHostBinding;

/**
 * Creates new QueryList, stores the reference in LView and returns QueryList.
 *
 * @param predicate The type for which the query will search
 * @param descend Whether or not to descend into children
 * @param read What to save in the query
 *
 * @codeGenApi
 */
export declare function ɵɵviewQuery<T>(predicate: Type<any> | string[], descend: boolean, read?: any): void;

export { }

//# sourceMappingURL=data:application/json;charset=utf-8;base64,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