/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
'use strict';
Zone.__load_patch('mocha', function (global, Zone) {
    var Mocha = global.Mocha;
    if (typeof Mocha === 'undefined') {
        // return if Mocha is not available, because now zone-testing
        // will load mocha patch with jasmine/jest patch
        return;
    }
    if (typeof Zone === 'undefined') {
        throw new Error('Missing Zone.js');
    }
    var ProxyZoneSpec = Zone['ProxyZoneSpec'];
    var SyncTestZoneSpec = Zone['SyncTestZoneSpec'];
    if (!ProxyZoneSpec) {
        throw new Error('Missing ProxyZoneSpec');
    }
    if (Mocha['__zone_patch__']) {
        throw new Error('"Mocha" has already been patched with "Zone".');
    }
    Mocha['__zone_patch__'] = true;
    var rootZone = Zone.current;
    var syncZone = rootZone.fork(new SyncTestZoneSpec('Mocha.describe'));
    var testZone = null;
    var suiteZone = rootZone.fork(new ProxyZoneSpec());
    var mochaOriginal = {
        after: Mocha.after,
        afterEach: Mocha.afterEach,
        before: Mocha.before,
        beforeEach: Mocha.beforeEach,
        describe: Mocha.describe,
        it: Mocha.it
    };
    function modifyArguments(args, syncTest, asyncTest) {
        var _loop_1 = function (i) {
            var arg = args[i];
            if (typeof arg === 'function') {
                // The `done` callback is only passed through if the function expects at
                // least one argument.
                // Note we have to make a function with correct number of arguments,
                // otherwise mocha will
                // think that all functions are sync or async.
                args[i] = (arg.length === 0) ? syncTest(arg) : asyncTest(arg);
                // Mocha uses toString to view the test body in the result list, make sure we return the
                // correct function body
                args[i].toString = function () { return arg.toString(); };
            }
        };
        for (var i = 0; i < args.length; i++) {
            _loop_1(i);
        }
        return args;
    }
    function wrapDescribeInZone(args) {
        var syncTest = function (fn) {
            return function () { return syncZone.run(fn, this, arguments); };
        };
        return modifyArguments(args, syncTest);
    }
    function wrapTestInZone(args) {
        var asyncTest = function (fn) {
            return function (done) { return testZone.run(fn, this, [done]); };
        };
        var syncTest = function (fn) {
            return function () { return testZone.run(fn, this); };
        };
        return modifyArguments(args, syncTest, asyncTest);
    }
    function wrapSuiteInZone(args) {
        var asyncTest = function (fn) {
            return function (done) { return suiteZone.run(fn, this, [done]); };
        };
        var syncTest = function (fn) {
            return function () { return suiteZone.run(fn, this); };
        };
        return modifyArguments(args, syncTest, asyncTest);
    }
    global.describe = global.suite = Mocha.describe = function () {
        return mochaOriginal.describe.apply(this, wrapDescribeInZone(arguments));
    };
    global.xdescribe = global.suite.skip = Mocha.describe.skip = function () {
        return mochaOriginal.describe.skip.apply(this, wrapDescribeInZone(arguments));
    };
    global.describe.only = global.suite.only = Mocha.describe.only = function () {
        return mochaOriginal.describe.only.apply(this, wrapDescribeInZone(arguments));
    };
    global.it = global.specify = global.test =
        Mocha.it = function () { return mochaOriginal.it.apply(this, wrapTestInZone(arguments)); };
    global.xit = global.xspecify = Mocha.it.skip = function () {
        return mochaOriginal.it.skip.apply(this, wrapTestInZone(arguments));
    };
    global.it.only = global.test.only = Mocha.it.only = function () {
        return mochaOriginal.it.only.apply(this, wrapTestInZone(arguments));
    };
    global.after = global.suiteTeardown = Mocha.after = function () {
        return mochaOriginal.after.apply(this, wrapSuiteInZone(arguments));
    };
    global.afterEach = global.teardown = Mocha.afterEach = function () {
        return mochaOriginal.afterEach.apply(this, wrapTestInZone(arguments));
    };
    global.before = global.suiteSetup = Mocha.before = function () {
        return mochaOriginal.before.apply(this, wrapSuiteInZone(arguments));
    };
    global.beforeEach = global.setup = Mocha.beforeEach = function () {
        return mochaOriginal.beforeEach.apply(this, wrapTestInZone(arguments));
    };
    (function (originalRunTest, originalRun) {
        Mocha.Runner.prototype.runTest = function (fn) {
            var _this = this;
            Zone.current.scheduleMicroTask('mocha.forceTask', function () { originalRunTest.call(_this, fn); });
        };
        Mocha.Runner.prototype.run = function (fn) {
            this.on('test', function (e) { testZone = rootZone.fork(new ProxyZoneSpec()); });
            this.on('fail', function (test, err) {
                var proxyZoneSpec = testZone && testZone.get('ProxyZoneSpec');
                if (proxyZoneSpec && err) {
                    try {
                        // try catch here in case err.message is not writable
                        err.message += proxyZoneSpec.getAndClearPendingTasksInfo();
                    }
                    catch (error) {
                    }
                }
            });
            return originalRun.call(this, fn);
        };
    })(Mocha.Runner.prototype.runTest, Mocha.Runner.prototype.run);
});
//# sourceMappingURL=data:application/json;base64,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